package volumeviz.sample.multiChannel_ex1;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.nio.FloatBuffer;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoMultiDataSeparator;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeShader;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  // //////////////////////////////////////////////////////////////////////
  //
  // Make the volume shader node
  //
  // Builds the FragmentShader and VolumeShader objects and sets up
  // the basic parameters.
  //
  // Note: This version assumes there are 3 channels of data and the
  // data will be rendered using SoVolumeRender.

  private IViewerExaminer myViewer;

  SoVolumeShader makeVolumeShader()
  {
    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    String prefix =
        SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator;

    // Load fragment shader program
    SoFragmentShader fragmentShader = new SoFragmentShader();
    fragmentShader.sourceProgram.setValue(  prefix+"MultiChannel_ex1_color_frag.glsl" );

    // Create uniform parameters allowing shader to access textures
    //
    // Transfer function is in texture unit 0 (default)
    SoShaderParameter1i paramTex0 = new SoShaderParameter1i();
    paramTex0.name.setValue("transfer");
    paramTex0.value.setValue(0);

    // Volume data is in texture units 1, 2 and 3
    SoShaderParameter1i paramTex1 = new SoShaderParameter1i();
    paramTex1.name.setValue("voldata1");
    paramTex1.value.setValue(1);
    SoShaderParameter1i paramTex2 = new SoShaderParameter1i();
    paramTex2.name.setValue("voldata2");
    paramTex2.value.setValue(2);
    SoShaderParameter1i paramTex3 = new SoShaderParameter1i();
    paramTex3.name.setValue("voldata3");
    paramTex3.value.setValue(3);

    fragmentShader.parameter.set1Value(0, paramTex0);
    fragmentShader.parameter.set1Value(1, paramTex1);
    fragmentShader.parameter.set1Value(2, paramTex2);
    fragmentShader.parameter.set1Value(3, paramTex3);

    // Initialize the volume shader
    // Note in this version we completely replace the default VolumeViz
    // fragment shader main program (to keep the example simple).
    SoVolumeShader volShader = new SoVolumeShader();
    volShader.shaderObject.set1Value(SoVolumeShader.ShaderPositions.FRAGMENT_COMPUTE_COLOR.getValue(), fragmentShader);

    // Specify shader should be applied to SoVolumeRender nodes.
    // (Name is a little deceptive because you might think that false means
    // for BOTH volumes and slices, but it's really one or the other.)
    volShader.forVolumeOnly.setValue(true);

    return volShader;
  }

  // //////////////////////////////////////////////////////////////////////
  //
  // Make a multi-channel colormap (transfer function)
  //
  // This function creates a colormap appropriate for (some) multi-channel
  // volume data. It contains a linear color ramp, i.e. the minimum data
  // value is mapped to black and the maximum data value is mapped to the
  // full color. The alpha ramp is a gamma curve, so the higher the gamma
  // value the more strongly low valued voxels are suppressed.
  //
  // The colormaps must have unique id numbers, starting at zero.

  SoTransferFunction makeColorMap(SbColor color, float gamma, int id)
  {
    SoTransferFunction tf = new SoTransferFunction();
    tf.transferFunctionId.setValue((short) id);

    // Color map will contain 256 RGBA values -> 1024 float values.
    // setNum pre-allocates memory so we can edit the field directly.
    tf.colorMap.setNum(256 * 4);

    float R = color.getValueAt(0);
    float G = color.getValueAt(1);
    float B = color.getValueAt(2);

    // Get an edit pointer, assign values, then finish editing.
    FloatBuffer fBfr = tf.colorMap.startEditing();
    for ( int i = 0; i < 256; ++i )
    {
      float factor = i / 255f;
      fBfr.put(R * factor);
      fBfr.put(G * factor);
      fBfr.put(B * factor);
      fBfr.put((float) Math.pow(factor, gamma));
    }
    tf.colorMap.finishEditing();
    return tf;
  }

  SoSeparator makeBBox(SbBox3f box, SbColor color, float width)
  {
    // The box will be easier to see without lighting
    SoLightModel lModel = new SoLightModel();
    lModel.model.setValue(SoLightModel.Models.BASE_COLOR);

    // And with wide lines
    SoDrawStyle style = new SoDrawStyle();
    style.lineWidth.setValue(width);

    // The box should be unpickable
    // (so geometry inside can be picked, draggers work inside, etc)
    SoPickStyle pickable = new SoPickStyle();
    pickable.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    // Create a cube outlining the specified box
    float[] bounds = box.getBounds();
    float xmin = bounds[0];
    float ymin = bounds[1];
    float zmin = bounds[2];
    float xmax = bounds[3];
    float ymax = bounds[4];
    float zmax = bounds[5];

    SoVertexProperty prop = new SoVertexProperty();
    prop.vertex.set1Value(0, new SbVec3f(xmin, ymin, zmin));
    prop.vertex.set1Value(1, new SbVec3f(xmax, ymin, zmin));
    prop.vertex.set1Value(2, new SbVec3f(xmax, ymax, zmin));
    prop.vertex.set1Value(3, new SbVec3f(xmin, ymax, zmin));
    prop.vertex.set1Value(4, new SbVec3f(xmin, ymin, zmax));
    prop.vertex.set1Value(5, new SbVec3f(xmax, ymin, zmax));
    prop.vertex.set1Value(6, new SbVec3f(xmax, ymax, zmax));
    prop.vertex.set1Value(7, new SbVec3f(xmin, ymax, zmax));
    prop.orderedRGBA.set1Value(0, color.getPackedValue());

    // Draw it with a line set
    int coordIndices[] = {
        0, 1, 2, 3, 0, -1, 4, 5, 6, 7, 4, -1,
        0, 4, -1, 1, 5, -1, 2, 6, -1, 3, 7
        };
    SoIndexedLineSet lines = new SoIndexedLineSet();
    lines.vertexProperty.setValue(prop);
    lines.coordIndex.setValues(0, coordIndices);

    // Assemble scene graph
    SoSeparator boxSep = new SoSeparator();
    boxSep.addChild(lModel);
    boxSep.addChild(pickable);
    boxSep.addChild(style);
    boxSep.addChild(lines);
    return boxSep;
  }

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // ---------- Load volume data ----------
    // Each channel is in a separate file.
    // Note that each volume must have a unique dataSetId.
    SoVolumeData volData0 = new SoVolumeData();
    volData0.fileName.setValue("$OIVJHOME/data/volumeviz/syn_r_chan.vol");
    volData0.dataSetId.setValue((short) 1);

    SoVolumeData volData1 = new SoVolumeData();
    volData1.fileName.setValue("$OIVJHOME/data/volumeviz/syn_g_chan.vol");
    volData1.dataSetId.setValue((short) 2);

    SoVolumeData volData2 = new SoVolumeData();
    volData2.fileName.setValue("$OIVJHOME/data/volumeviz/syn_b_chan.vol");
    volData2.dataSetId.setValue((short) 3);

    // ---------- Map actual range of data values ----------
    // This test data only uses values 0..192, so we will not use all the
    // values in the colormap unless we explicitly set the data range.
    // Note:
    // 1f Each volume data node must have its own data range node.
    // 2. The dataRangeId must be the same as the dataSetId of the
    // corresponding volume data node.
    //
    // In many cases we don't need data range nodes for 8-bit (byte) data
    // because all values 0..255 are used, but we do need them in most
    // cases for larger data types, e.g. 16 bit data sets often only
    // contain 12 bits of actual data.
    //
    // Note that the getMinMax query is "free" for LDM format data,
    // because the min and max values are stored in the file header.
    // Custom readers may also support this query. But many of the
    // standard volume readers have to load the entire data set from
    // disk to answer this query and this can be very time consuming.
    // If using such a reader, when possible, preprocess the data and
    // store the min/max values in some sort of "info" file that the
    // application can load quickly and avoid calling getMinMax().
    long[] dataMinMax = volData0.getMinMax();
    SoDataRange dataRange0 = new SoDataRange();
    dataRange0.min.setValue(dataMinMax[0]);
    dataRange0.max.setValue(dataMinMax[1]);
    dataRange0.dataRangeId.setValue((short) 1);

    dataMinMax = volData1.getMinMax();
    SoDataRange dataRange1 = new SoDataRange();
    dataRange1.min.setValue(dataMinMax[0]);
    dataRange1.max.setValue(dataMinMax[1]);
    dataRange1.dataRangeId.setValue((short) 2);

    dataMinMax = volData2.getMinMax();
    SoDataRange dataRange2 = new SoDataRange();
    dataRange2.min.setValue(dataMinMax[0]);
    dataRange2.max.setValue(dataMinMax[1]);
    dataRange2.dataRangeId.setValue((short) 3);

    // ---------- Make color map for each channel ----------
    // Typically for multichannel rendering we will create a colormap
    // with a linear color ramp, i.e. black . full color, and a gamma
    // curve alpha ramp.
    // Note that each volume must have a unique transferFunctionId,
    // which generally should be the same as the corresponding volume
    // data node's dataSetId.
    SoTransferFunction tf0 = makeColorMap(new SbColor(1, 0, 0), 2.39f, 1);
    SoTransferFunction tf1 = makeColorMap(new SbColor(0, 1, 0), 2.39f, 2);
    SoTransferFunction tf2 = makeColorMap(new SbColor(0, 0, 1), 2.39f, 3);

    // Render node
    SoVolumeRender volRender = new SoVolumeRender();
    volRender.numSlices.setValue(128); // To improve image quality

    // Volume shader node
    SoVolumeShader volShader = makeVolumeShader();

    // Assemble the volume rendering scene graph.
    // Note that we use a MultiDataSeparator node to hold multiple data
    // sets that will be combined in a single rendering node.
    SoMultiDataSeparator volSep = new SoMultiDataSeparator();
    volSep.addChild(volShader);
    volSep.addChild(volData0);
    volSep.addChild(volData1);
    volSep.addChild(volData2);
    volSep.addChild(dataRange0);
    volSep.addChild(dataRange1);
    volSep.addChild(dataRange2);
    volSep.addChild(tf0);
    volSep.addChild(tf1);
    volSep.addChild(tf2);
    volSep.addChild(volRender);

    // Assemble the main scene graph.
    // Includes a representation of the volume's bounding box.
    SoSeparator root = new SoSeparator();
    root.addChild(volSep);
    root.addChild(makeBBox(volData1.extent.getValue(), new SbColor(1, 0, 0), 2f));

    // Set up viewer:
    myViewer.setSceneGraph(root);
    myViewer.getRenderArea().setTransparencyType(SoGLRenderAction.TransparencyTypes.DELAYED_BLEND);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  public static void main(String argv[])
  {
    Main example = new Main();
    example.demoMain("Multichannel rendering");
  }
}
