package volumeviz.sample.physicalRendering;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.io.IOException;

import javax.swing.*;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.SbColorRGBA;
import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoOffscreenRenderer;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.fields.SoSFFloat;
import com.openinventor.inventor.helpers.SbFileHelper;
import com.openinventor.inventor.nodes.SoInteractiveComplexity;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShadowGroup;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.sensors.SoFieldSensor;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingPhysicalQuality;

import util.Example;
import util.ViewerComponentsFactory;
import util.editors.MaterialEditor;

/**
 * This demo shows off the physical rendering effects introduced in OIV 9.8.
 *
 * This demo allows to switch between two datasets, medicalfoot and 3dhead, and
 * adjust most of the parameters of the physical rendring, including shadow
 * type, opacity and quality; lighting material (down to custom material,
 * environment map and tone mapping); depth of field (blur factor and focal
 * distance) and also allows to take snapshots of up to 10 times the original
 * resolution.
 *
 */
public class Main extends Example
{
  SoVolumeRenderingPhysicalQuality m_physicalvrq;
  SoMaterial m_lightingMaterial;
  SoShadowGroup m_shadowGroup;
  SoInteractiveComplexity m_interactiveComplexity;
  SoSwitch m_dataSetSwitch;
  MaterialEditor m_materialEditor;
  private IViewerExaminer m_viewer;
  private SoFieldSensor m_cameraFocalDistanceSensor;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("Physical Rendering");
  }

  @Override
  public void start()
  {
    m_viewer = ViewerComponentsFactory.createViewerExaminer();

    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    String prefix =
        SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator;
    SoSeparator root = readSceneGraph(prefix + "scene_physicalRendering.iv");

    m_physicalvrq = (SoVolumeRenderingPhysicalQuality) SoNode.getByName("PHYSICAL_RENDERING_QUALITY");
    m_lightingMaterial = (SoMaterial) SoNode.getByName("MATERIAL");
    m_shadowGroup = (SoShadowGroup) SoNode.getByName("SHADOW_GROUP");
    m_dataSetSwitch = (SoSwitch) SoNode.getByName("DATA_SWITCH");
    m_interactiveComplexity = (SoInteractiveComplexity) SoNode.getByName("INTERACTIVE_COMPLEXITY");

    m_materialEditor = new MaterialEditor();
    m_materialEditor.attach(m_lightingMaterial);

    m_viewer.getRenderArea().setTransparencyType(SoGLRenderAction.TransparencyTypes.BLEND);
    m_viewer.setSceneGraph(root);
    m_viewer.viewAll();

    initGUI();
  }

  @Override
  public void stop()
  {
    m_materialEditor.dispose();
    m_viewer.dispose();
  }

  private static SoSeparator readSceneGraph(String fileName)
  {
    SoInput input = new SoInput();

    if ( !input.openFile(fileName) )
    {
      System.err.println("Cannot open file " + fileName);
      return null;
    }

    SoSeparator node = SoDB.readAll(input);
    if ( node == null )
    {
      System.err.println("Problem reading file");
      input.closeFile();
      return null;
    }

    input.closeFile();
    return node;
  }

  private void initGUI()
  {
    // ====================================
    // dataset panel
    JPanel dataset_panel = new JPanel();
    String[] datasets = { "FOOT", "HEAD" };
    final JComboBox<String> dataset_box = new JComboBox<>(datasets);
    dataset_box.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        m_dataSetSwitch.whichChild.setValue(dataset_box.getSelectedIndex());
        m_viewer.viewAll();
      }
    });
    dataset_panel.add(dataset_box);

    // ====================================
    // shadow panel
    JPanel shadow_panel = new JPanel();
    shadow_panel.setLayout(new BoxLayout(shadow_panel, BoxLayout.PAGE_AXIS));
    shadow_panel.setBorder(new TitledBorder(new EtchedBorder(), "Shadows"));

    final JCheckBox enable_shadows_box = new JCheckBox("Enable");
    enable_shadows_box.setSelected(true);

    final JComboBox<SoVolumeRenderingPhysicalQuality.ShadowsStyles> shadow_style_box =
        new JComboBox<>(SoVolumeRenderingPhysicalQuality.ShadowsStyles.values());
    shadow_style_box.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        m_physicalvrq.shadowsStyle
            .setValue((SoVolumeRenderingPhysicalQuality.ShadowsStyles) shadow_style_box.getSelectedItem());
      }
    });
    shadow_style_box.setPreferredSize(new Dimension(145, 20));
    shadow_style_box.setSelectedItem(SoVolumeRenderingPhysicalQuality.ShadowsStyles.RAYTRACED);

    final SliderPanel shadow_opacity_slider = new SliderPanel(0.f, 1.f, 0.8f, 3);
    shadow_opacity_slider.setSliderSize(new Dimension(300, 20));
    shadow_opacity_slider.addInfoText("opacity");
    shadow_opacity_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_shadowGroup.intensity.setValue(value);
      }
    });

    final SliderPanel shadow_quality_slider = new SliderPanel(0.f, 1.f, 0.5f, 3);
    shadow_quality_slider.setSliderSize(new Dimension(300, 20));
    shadow_quality_slider.addInfoText("quality");
    shadow_quality_slider.setBoxSelected(false);
    shadow_quality_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_shadowGroup.quality.setValue(value);
      }
    });

    enable_shadows_box.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        boolean selected = (e.getStateChange() == ItemEvent.SELECTED);
        m_shadowGroup.isActive.setValue(selected);
        shadow_style_box.setEnabled(selected);
        shadow_opacity_slider.setEnabled(selected);
        shadow_quality_slider.setEnabled(selected);
      }
    });
    shadow_panel.add(enable_shadows_box);
    shadow_panel.add(shadow_style_box);
    shadow_panel.add(shadow_opacity_slider);
    shadow_panel.add(shadow_quality_slider);

    // ====================================
    // lighting panel
    JPanel lighting_panel = new JPanel();
    lighting_panel.setLayout(new BoxLayout(lighting_panel, BoxLayout.PAGE_AXIS));
    lighting_panel.setBorder(new TitledBorder(new EtchedBorder(), "Physically-based Lighting"));

    // environment map combobox
    JPanel envmap_panel = new JPanel();
    JLabel envmap_label = new JLabel("Environment map");
    final JComboBox<SoVolumeRenderingPhysicalQuality.EnvironmentMaps> envmap_box =
        new JComboBox<>(SoVolumeRenderingPhysicalQuality.EnvironmentMaps.values());
    envmap_box.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        m_physicalvrq.environmentMap
            .setValue((SoVolumeRenderingPhysicalQuality.EnvironmentMaps) envmap_box.getSelectedItem());
      }
    });
    envmap_box.setPreferredSize(new Dimension(145, 20));
    envmap_box.setSelectedItem(SoVolumeRenderingPhysicalQuality.EnvironmentMaps.UFFIZI);
    envmap_box.setEnabled(false);
    envmap_panel.add(envmap_label);
    envmap_panel.add(envmap_box);

    // shading style combobox
    JPanel shadingStyle_panel = new JPanel();
    JLabel shadingStyle_label = new JLabel("Shading Style");
    final JComboBox<SoVolumeRenderingPhysicalQuality.ShadingStyles> shadingStyle_box =
        new JComboBox<>(SoVolumeRenderingPhysicalQuality.ShadingStyles.values());
    shadingStyle_box.setSelectedItem(SoVolumeRenderingPhysicalQuality.ShadingStyles.PHYSICALLY_BASED);

    shadingStyle_box.setPreferredSize(new Dimension(145, 20));
    shadingStyle_panel.add(shadingStyle_label);
    shadingStyle_panel.add(shadingStyle_box);

    // material combobox
    JPanel material_panel = new JPanel();
    JLabel material_label = new JLabel("predefined material");
    final JComboBox<SoVolumeRenderingPhysicalQuality.PredefinedMaterials> material_box =
        new JComboBox<>(SoVolumeRenderingPhysicalQuality.PredefinedMaterials.values());
    material_box.setPreferredSize(new Dimension(145, 20));
    material_box.setSelectedItem(SoVolumeRenderingPhysicalQuality.PredefinedMaterials.GLOSSY);
    material_panel.add(material_label);
    material_panel.add(material_box);

    // material editor button
    final JButton materialEditor_button = new JButton("Show Material Editor");
    materialEditor_button.setEnabled(false);
    materialEditor_button.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        m_materialEditor.setVisible(true);
      }
    });

    // environment map combobox
    JPanel toneMapping_panel = new JPanel();
    JLabel toneMapping_label = new JLabel("Tone Mapping");
    final JComboBox<SoVolumeRenderingPhysicalQuality.ToneMappings> toneMapping_box =
        new JComboBox<>(SoVolumeRenderingPhysicalQuality.ToneMappings.values());
    toneMapping_box.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        m_physicalvrq.toneMapping
            .setValue((SoVolumeRenderingPhysicalQuality.ToneMappings) toneMapping_box.getSelectedItem());
      }
    });
    toneMapping_box.setPreferredSize(new Dimension(145, 20));
    toneMapping_box.setSelectedItem(SoVolumeRenderingPhysicalQuality.ToneMappings.MEDIUM);
    toneMapping_box.setEnabled(false);
    toneMapping_panel.add(toneMapping_label);
    toneMapping_panel.add(toneMapping_box);

    material_box.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        SoVolumeRenderingPhysicalQuality.PredefinedMaterials newMaterial =
            (SoVolumeRenderingPhysicalQuality.PredefinedMaterials) material_box.getSelectedItem();
        m_physicalvrq.predefinedMaterial.setValue(newMaterial);
        boolean customMaterial = newMaterial == SoVolumeRenderingPhysicalQuality.PredefinedMaterials.CUSTOM_MATERIAL;
        materialEditor_button.setEnabled(customMaterial);
        envmap_box.setEnabled(customMaterial);
        toneMapping_box.setEnabled(customMaterial);
      }
    });

    shadingStyle_box.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        SoVolumeRenderingPhysicalQuality.ShadingStyles newStyle =
            (SoVolumeRenderingPhysicalQuality.ShadingStyles) shadingStyle_box.getSelectedItem();
        boolean phongShading = newStyle == SoVolumeRenderingPhysicalQuality.ShadingStyles.PHONG;
        boolean customMaterial = (SoVolumeRenderingPhysicalQuality.PredefinedMaterials) material_box
            .getSelectedItem() == SoVolumeRenderingPhysicalQuality.PredefinedMaterials.CUSTOM_MATERIAL;
        boolean enableButtons = !phongShading && customMaterial;
        material_box.setEnabled(!phongShading);
        envmap_box.setEnabled(enableButtons);
        toneMapping_box.setEnabled(enableButtons);
        materialEditor_button.setEnabled(phongShading || customMaterial);
        m_physicalvrq.shadingStyle.setValue(newStyle);

      }
    });

    lighting_panel.add(shadingStyle_panel);
    lighting_panel.add(material_panel);
    lighting_panel.add(materialEditor_button);
    lighting_panel.add(envmap_panel);
    lighting_panel.add(toneMapping_panel);

    // ====================================
    // depth of field panel
    JPanel dof_panel = new JPanel();
    dof_panel.setLayout(new BoxLayout(dof_panel, BoxLayout.PAGE_AXIS));
    dof_panel.setBorder(new TitledBorder(new EtchedBorder(), "Depth Of Field"));

    final JCheckBox enable_dof_box = new JCheckBox("Enable");
    enable_dof_box.setSelected(true);

    // blur factor slider
    final SliderPanel blurFactor_slider = new SliderPanel(0.f, 0.2f, 0.02f, 3);
    blurFactor_slider.setSliderSize(new Dimension(300, 20));
    blurFactor_slider.addInfoText("Blur Factor");
    blurFactor_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_physicalvrq.blurFactor.setValue(value);
      }
    });

    // focal distance slider
    SoSFFloat cameraFocalDistance = m_viewer.getRenderArea().getRootSceneGraph().getCamera().focalDistance;
    float focalDistance = cameraFocalDistance.getValue();
    m_cameraFocalDistanceSensor = new SoFieldSensor();
    final SliderPanel focalDistance_slider = new SliderPanel(0.f, 2.0f * focalDistance, focalDistance, 3);
    focalDistance_slider.setSliderSize(new Dimension(300, 20));
    focalDistance_slider.addInfoText("Focal Distance");
    focalDistance_slider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        // Don't trigger sensor unless necessary
        m_cameraFocalDistanceSensor.detach();
        if ( cameraFocalDistance.getValue() != value )
          cameraFocalDistance.setValue(value);
        m_cameraFocalDistanceSensor.attach(cameraFocalDistance);
      }
    });

    // Schedule a sensor to update the slider when camera focal distance
    // changes.
    m_cameraFocalDistanceSensor.setTask(new Runnable()
    {
      @Override
      public void run()
      {
        // Update UI with actual focal distance
        float focalDistance = cameraFocalDistance.getValue();

        // Don't trigger an event unless necessary
        if ( focalDistance_slider.getSliderValue() != focalDistance )
        {
          focalDistance_slider.setSliderMax(2.0f * focalDistance);
          focalDistance_slider.setSliderValue(focalDistance);
        }
      }
    });
    m_cameraFocalDistanceSensor.attach(cameraFocalDistance);

    enable_dof_box.addItemListener(new ItemListener()
    {

      @Override
      public void itemStateChanged(ItemEvent e)
      {
        boolean selected = (e.getStateChange() == ItemEvent.SELECTED);
        m_physicalvrq.enableDepthOfField.setValue(selected);
        blurFactor_slider.setEnabled(selected);
        focalDistance_slider.setEnabled(selected);
      }
    });

    dof_panel.add(enable_dof_box);
    dof_panel.add(blurFactor_slider);
    dof_panel.add(focalDistance_slider);

    // ====================================
    // snapshot panel
    JPanel snapshot_panel = new JPanel();
    snapshot_panel.setLayout(new BoxLayout(snapshot_panel, BoxLayout.PAGE_AXIS));
    snapshot_panel.setBorder(new TitledBorder(new EtchedBorder(), "Snapshot"));

    final SliderPanel scale_slider = new SliderPanel(1.f, 10.f, 2.f, 3);
    scale_slider.setSliderSize(new Dimension(300, 20));
    scale_slider.addInfoText("Scale");

    JPanel snapshot_filename_panel = new JPanel();
    snapshot_filename_panel.setLayout(new BoxLayout(snapshot_filename_panel, BoxLayout.LINE_AXIS));
    final JTextField snapshot_filename =
        new JTextField("$OIVJHOME/examples/volumeviz/sample/physicalRendering/snapshot.png");
    final JButton snapshot_filechooser = new JButton("...");
    snapshot_filechooser.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        JFileChooser fileChooser = new JFileChooser();
        int option = fileChooser.showSaveDialog(snapshot_filechooser);

        if ( option == JFileChooser.APPROVE_OPTION )
        {
          try
          {
            snapshot_filename.setText(fileChooser.getSelectedFile().getCanonicalPath());
          }
          catch (IOException e)
          {
            e.printStackTrace();
          }
        }

      }
    });

    snapshot_filename_panel.add(snapshot_filename);
    snapshot_filename_panel.add(snapshot_filechooser);

    JButton snaphot_button = new JButton("Snapshot");
    snaphot_button.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent arg0)
      {
        float scale = scale_slider.getSliderValue();
        // Make image same size and shape as viewer image
        SbViewportRegion vpRegion = new SbViewportRegion(m_viewer.getRenderArea().getComponent().getSize());
        vpRegion.setWindowSize((short) (scale * vpRegion.getWindowSizei32().getX()),
            (short) (scale * vpRegion.getWindowSizei32().getY()));
        SoOffscreenRenderer renderer = new SoOffscreenRenderer(vpRegion);

        // Transfer viewer settings to renderer
        renderer.getGLRenderAction().setTransparencyType(m_viewer.getRenderArea().getTransparencyType());
        renderer.setFullSceneAntialiasing(true);
        renderer.setBackgroundColorRGBA(new SbColorRGBA(0, 0, 0, 0));
        renderer.setComponents(SoOffscreenRenderer.Components.RGB_TRANSPARENCY);
        renderer.setMaxTileSize(new SbVec2s((short) 1024, (short) 1024));
        renderer.setNumEdgePixels(5);

        // Render scene (including camera and headlight)
        renderer.render(m_viewer.getRenderArea().getRootSceneGraph());

        renderer.writeToPNG(SbFileHelper.expandString(snapshot_filename.getText()));
      }
    });

    snapshot_panel.add(scale_slider);
    snapshot_panel.add(snapshot_filename_panel);
    snapshot_panel.add(snaphot_button);

    // ====================================
    // main panel
    JPanel main_panel = new JPanel();
    main_panel.setLayout(new BoxLayout(main_panel, BoxLayout.PAGE_AXIS));
    main_panel.setBorder(BorderFactory.createRaisedBevelBorder());
    main_panel.add(dataset_panel);
    main_panel.add(shadow_panel);
    main_panel.add(lighting_panel);
    main_panel.add(dof_panel);
    main_panel.add(snapshot_panel);

    // ScrollPane
    JScrollPane scroll_pane = new JScrollPane(main_panel);

    // SplitPane
    JPanel iPanel = new JPanel(new BorderLayout());
    final Component component = m_viewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    iPanel.add(component, BorderLayout.CENTER);
    iPanel.setMinimumSize(new Dimension(1, 1));

    JSplitPane split_pane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, true, iPanel, scroll_pane);
    split_pane.setOneTouchExpandable(true);
    split_pane.setResizeWeight(0.6);

    setLayout(new BorderLayout());
    add(split_pane);
  }

}
