package volumeviz.sample.simpleLightedVolume;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.BufferedReader;
import java.io.FileReader;

import com.openinventor.inventor.SbEventListener;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.SoProgressIndicator;
import com.openinventor.inventor.SoProgressIndicator.StepEventArg;
import com.openinventor.inventor.SoProgressIndicator.SubTaskEventArg;
import com.openinventor.inventor.SoProgressIndicator.TaskEventArg;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Demonstrates how to use a SoVolumeRenderingQuality property node for lighting
 * a volume. It is also showing how to track progress of a fixedResolution
 * rendering.
 */
public class Main extends Example
{

  private IViewerExaminer myViewer;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("SimpleLightedVolume");
  }

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    String fileName = SoPreferences.getValue("OIVJHOME") + "/data/volumeviz/bonsai.am";

    // Node to hold the volume data
    SoVolumeData pVolData = new SoVolumeData();

    // Load the model in the SoVolumeData
    pVolData.fileName.setValue(fileName);
    pVolData.ldmResourceParameters.getValue().resolution.setValue(0);
    pVolData.ldmResourceParameters.getValue().fixedResolution.setValue(true);

    // If necessary, specify the actual range of the data values.
    // By default VolumeViz maps the entire range of the voxel data type
    // (e.g. 0..65535 for unsigned short) into the colormap. This works
    // great for byte (8 bit) voxels, but not so well for 16 bit voxels
    // and not at all for floating point voxels. So it's not actually
    // necessary for this data set, but shown here for completeness.
    // NOTE: Min/max values are stored in the header for LDM format
    // files, but for other formats the getMinMax query can take a
    // long time because VolumeViz has to examine every voxel.
    SoDataRange pRange = new SoDataRange();
    int voxelSize = pVolData.getDataSize();
    if ( voxelSize > 1 )
    {
      double[] minmax = pVolData.getDoubleMinMax();
      pRange.min.setValue(minmax[0]);
      pRange.max.setValue(minmax[1]);
    }

    // Load the colorMap from a file
    String colormapFileName = SoPreferences.getValue("OIVJHOME") + "/data/volumeviz/bonsai.txt";
    float[] colormap = loadColormap(colormapFileName);
    SoTransferFunction pTransFunc = new SoTransferFunction();
    pTransFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.NONE);
    pTransFunc.colorMapType.setValue(SoTransferFunction.ColorMapTypes.RGBA);
    pTransFunc.colorMap.setValues(0, colormap);

    // Property node which allows SoVolumeRender to draw lighted volumes
    SoVolumeRenderingQuality pVRVolQuality = new SoVolumeRenderingQuality();
    pVRVolQuality.lighting.setValue(true);
    pVRVolQuality.preIntegrated.setValue(true);
    pVRVolQuality.lightingModel.setValue(SoVolumeRenderingQuality.LightingModels.OPENGL);

    // Node in charge of drawing the volume
    SoVolumeRender pVolRender = new SoVolumeRender();
    pVolRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.ALL);
    pVolRender.subdivideTile.setValue(true);

    // To track volumeRender progression we need to attach an
    // SoProgressIndicator
    SoProgressIndicator ps = new SoProgressIndicator();
    ps.onBeginTask.addEventListener(new BeginTaskListener());
    ps.onEndTask.addEventListener(new EndTaskListener());
    ps.onBeginSubTask.addEventListener(new BeginSubTaskListener());
    ps.onEndSubTask.addEventListener(new EndSubTaskListener());
    ps.onEndStep.addEventListener(new EndStepListener());
    pVolRender.setRenderProgress(ps);

    // Set the ambient, diffuse, transparency, specular and shininess of the
    // material
    SoMaterial material = new SoMaterial();
    material.ambientColor.setValue(0.0f, 0.0f, 0.0f);
    material.diffuseColor.setValue(1.f, 1.f, 1.f);
    material.transparency.setValue(0.0f);
    material.specularColor.setValue(1.f, 1.f, 1.f);
    material.shininess.setValue(0.5f);

    SoSeparator root = new SoSeparator();
    { // Assemble the scene graph
      // Note: SoVolumeRender must appear after the SoVolumeData node.
      root.addChild(material);
      root.addChild(pVolData);
      root.addChild(pRange);
      root.addChild(pTransFunc);
      root.addChild(pVRVolQuality);
      root.addChild(pVolRender);
    }

    // Set up viewer:
    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  private float[] loadColormap(String filename)
  {
    FileReader fileReader = null;
    try
    {
      fileReader = new FileReader(filename);
    }
    catch (Exception e)
    {
      System.err.println("Can't open the color map file " + filename);
      e.printStackTrace();
      return null;
    }

    BufferedReader b = new BufferedReader(fileReader);

    int numColors = 0;
    String s;

    try
    {
      // Get first line of file and try to identify it.
      s = b.readLine();
      numColors = Integer.valueOf(s).intValue();
      if ( numColors <= 0 )
      {
        b.close();
        return null;
      }

      float[][] rgba = new float[256][4];
      int numComponents = 0;

      s = b.readLine();
      for ( int i = 0; s != null && i < numColors; i++, s = b.readLine() )
      {
        float[] colors = new float[4];
        String[] colorsString = s.split(" ");
        numComponents = colorsString.length;

        for ( int k = 0; k < numComponents; k++ )
          colors[k] = Float.valueOf(colorsString[k]).floatValue() / 65535.0f;

        switch ( numComponents )
        {
        case 1 : // A (RGB=1)
          colors[3] = colors[0];
          colors[0] = colors[1] = colors[2] = 1;
          break;
        case 3 : // RGB (A=1)
          colors[3] = 1;
          break;
        case 4 : // RGBA
          break;
        default: // empty line
          continue;
        }
        rgba[i][0] = colors[0];
        rgba[i][1] = colors[1];
        rgba[i][2] = colors[2];
        rgba[i][3] = colors[3];
      }
      b.close();

      int mode = 256 / numColors;

      float[] colormap = new float[256 * 4];
      int index = 0;
      for ( int k = 0; k < numColors; k++ )
      {
        int k1 = k + 1;
        if ( k1 >= numColors )
          k1 = numColors - 1;
        for ( int i = 0; i < mode; i++ )
        {
          colormap[index++] = rgba[k][0] + (float) i / mode * (rgba[k1][0] - rgba[k][0]);
          colormap[index++] = rgba[k][1] + (float) i / mode * (rgba[k1][1] - rgba[k][1]);
          colormap[index++] = rgba[k][2] + (float) i / mode * (rgba[k1][2] - rgba[k][2]);
          colormap[index++] = rgba[k][3] + (float) i / mode * (rgba[k1][3] - rgba[k][3]);
        }
      }

      return colormap;
    }
    catch (Exception ex)
    {
      ex.printStackTrace();
      return null;
    }
  }

  class BeginTaskListener implements SbEventListener<TaskEventArg>
  {
    @Override
    public void onEvent(TaskEventArg eventArg)
    {
      System.out
          .println("BEGIN_TASK, name: " + eventArg.getEventName() + ", with " + eventArg.getNumSubTasks() + " phases.");
    }
  }

  class EndTaskListener implements SbEventListener<TaskEventArg>
  {
    @Override
    public void onEvent(TaskEventArg eventArg)
    {
      System.out
          .println("END_TASK, name: " + eventArg.getEventName() + ", with " + eventArg.getNumSubTasks() + " phases.");

      SoVolumeRender vr = (SoVolumeRender) eventArg.getSource();
      if ( vr != null )
        vr.setRenderProgress(null);
    }
  }

  class BeginSubTaskListener implements SbEventListener<SubTaskEventArg>
  {
    @Override
    public void onEvent(SubTaskEventArg eventArg)
    {
      System.out.println("\tBEGIN_SUBTASK " + eventArg.getNumSubTasksDone() + "/" + eventArg.getNumSubTasks()
          + ", name: " + eventArg.getEventName() + ", with " + eventArg.getNumSteps() + " steps.");
    }
  }

  class EndSubTaskListener implements SbEventListener<SubTaskEventArg>
  {
    @Override
    public void onEvent(SubTaskEventArg eventArg)
    {
      System.out.println("\tEND_SUBTASK " + eventArg.getNumSubTasksDone() + "/" + eventArg.getNumSubTasks() + ", name: "
          + eventArg.getEventName() + ", with " + eventArg.getNumSteps() + " steps.");
    }
  }

  class EndStepListener implements SbEventListener<StepEventArg>
  {
    @Override
    public void onEvent(StepEventArg eventArg)
    {
      System.out.print("\t\t"
          + String.format("%1$,.2f", eventArg.getNumStepsDone() * 100 / (double) eventArg.getNumSteps()) + "%\r");
    }
  }
}
