#ifndef ALGEBRAIC_SHAPE_H
#define ALGEBRAIC_SHAPE_H

#ifndef OIV_HAS_ASRAY
#define OIV_HAS_ASRAY


/**
 * @ingroup AlgebraicShapeShaders
 * @file
 * This file contains utilities fonction when defining custom SoAlgebraicShape.
 */

/**
 * Structure for parameter of the ray.
 */
struct OivASRay {
    vec3 rs; // ray start
    vec3 re; // ray end
    vec3 rd; // ray direction
};
#endif

/**
 * Structure containing information about the the raycast
 * intersection point.
 */
struct OivASPoint {
    vec3 position;
    vec3 normal;
    vec4 color;
};

/**
 * Helper function to solve quadric of type -> a*x*x + b*x + c = 0.
 * Return true if there are roots, stored in 'roots' vector.
 * Note that it can be used in fragment shader only.
 */
bool OivASSolveQuadric ( in vec3 abc, inout vec2 roots );

/**
 * Ray intersects plane, returns true is there is an intersection at 't'
 * Plane 'p' is define such as 'xyz' represents the orientation and 'w', the
 * distance from origin (i.e. the point p.w*p.xyz lies on the plane).
 */
bool OivASRayPlaneIntersection ( in OivASRay ray, in vec4 p, inout float t );

/**
 * Get the color (i.e. from material or vertex color) of the shape.
 */
vec4 OivASGetColor ();

/**
 * Get the opacity (i.e. alpha component) of the shape.
 */
float OivASGetOpacity ();

/**
 * Ask if the shape is opaque (i.e. OivASGetOpacity() == 1.0).
 */
bool OivASIsOpaque ();

/******************************** SLOT ***************************************/

/**
 * [Slot] VERTEX_SHADER_ENTRY
 * Defines actions from the vertex shader such as initialize varying parameters
 * from vertex attribute and so on.
 */
void OivASVertexShaderEntry ();

/**
 * [Slot] COMPUTE_COLOR
 * Returns the color of the fragment. Note that for shading facilities, the position,
 * normal and color of the fragment are filled in the input parameter 'p'.
 */
vec4 OivASComputeColor ( in OivASPoint p );

/**
 * [Slot] rayIntersection
 * Computes the intersection between ray and shape.
 *   * ray: ray parameters
 *   * p: structure containing the intersection point
 *        position (if any), normal (if any) and color (if any)
 *   * return true if there is intersection, false otherwise
 *
 * Note that all positions and directions are defined in the space specified
 * in the constructor of the SoAlgebraicShape node. The default one is the
 * local frame coordinate centered at (0,0,0) with bounding box coordinates
 * between -1.0 and 1.0.
 *
 * For example, for a sphere, we consider in this function that the sphere has
 * a radius of 1.0, centered at the origin.
 */
bool OivASRayIntersection ( in OivASRay ray, inout OivASPoint p );

#endif // ALGEBRAIC_SHAPE_H
