#ifndef OIV_SHADER_STATE_H
#define OIV_SHADER_STATE_H

//!oiv_include <Inventor/oivShaderStages.h>

/**
 * @ingroup InventorShaders
 * @file
 * This file contains fonction to acces built-in GLSL uniforms
 */


/*****************************************************************************/
// Constants

/**
 * Get the maximum number of lights sets by Open Inventor.
 */
int OivMaxLights()
{
  return gl_MaxLights;
}

/**
 * Get the maximum number of clipping planes sets by Open Inventor.
 */
int OivMaxClipPlanes()
{
  return gl_MaxClipPlanes;
}

/**
 * Get the maximum number of texture coordinates sets by Open Inventor.
 */
int OivMaxTextureCoordinates()
{
  return gl_MaxTextureCoords;
}

/**
 * Get the maximum number of texture units sets by Open Inventor.
 */
int OivMaxTextureUnits()
{
  return gl_MaxTextureUnits;
}

/*****************************************************************************/
// Transforms

/**
 * Get the combination of model and view transformations (MV),
 * i.e., MV = V * M.
 * @return the model-view matrix.
 */
mat4 OivModelViewMatrix()
{
  return gl_ModelViewMatrix;
}

/**
 * Get the inverse of the model-view matrix,
 * @return the inverse of the model-view matrix.
 */
mat4 OivModelViewMatrixInverse()
{
  return gl_ModelViewMatrixInverse;
}

/**
 * Get the normal matrix (N),
 * i.e., N = tr(MV^{-1}).
 * @return the normal matrix.
 */
mat3 OivNormalMatrix()
{
  return gl_NormalMatrix;
}

/**
 * Get the projection matrix (P).
 * @return the projection matrix.
 */
mat4 OivProjectionMatrix()
{
  return gl_ProjectionMatrix;
}

/**
 * Get the inverse of the projection matrix (P^{-1}).
 * @return the inverse projection matrix.
 */
mat4 OivProjectionMatrixInverse()
{
  return gl_ProjectionMatrixInverse;
}

/**
 * Get the combination of projection and model-view matrices (MVP),
 * i.e., MVP = P * V * M.
 * @return the model-view-projection matrix
 */
mat4 OivModelViewProjectionMatrix()
{
  return gl_ModelViewProjectionMatrix;
}

/**
 * Get texture matrices.
 * @param unit specifies which texture unit between 0 and OivMaxTextureUnit()-1.
 * @return the texture matrix for a given unit.
 */
mat4 OivTextureMatrix(int unit)
{
  return gl_TextureMatrix[unit];
}

/*****************************************************************************/
// Lighting

/**
 * Gets the global ambient color value
 * @return The global ambient color value
 */
vec4 OivGlobalAmbient()
{
  return gl_LightModel.ambient;
}

/**
 * Ask if the light is enabled.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return true if light with a given id is enabled, false otherwise.
 */
bool OivLightSourceEnabled(int id)
{
  return bool(gl_LightSource[id].ambient != vec4(0.0)) || bool(gl_LightSource[id].diffuse != vec4(0.0)) || bool(gl_LightSource[id].specular != vec4(0.0));
}

/**
 * Get the light position.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light position for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec4 OivLightSourcePosition(int id)
{
  return gl_LightSource[id].position;
}

/**
 * Get the light spot direction.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light spot direction for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec3 OivLightSourceSpotDirection(int id)
{
  return gl_LightSource[id].spotDirection;
}

/**
 * Get the light spot exponent.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light spot exponent for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceSpotExponent(int id)
{
  return gl_LightSource[id].spotExponent;
}

/**
 * Get the light spot cutoff.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light spot cutoff for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceSpotCutoff(int id)
{
  return gl_LightSource[id].spotCutoff;
}

/**
 * Get the light spot cutoff cosine.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light spot cutoff cosine for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceSpotCosCutoff(int id)
{
  return gl_LightSource[id].spotCosCutoff;
}

/**
 * Get the light ambient color.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light ambient color for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec4 OivLightSourceAmbient(int id)
{
  return gl_LightSource[id].ambient;
}

/**
 * Get the light diffuse color.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light diffuse color for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec4 OivLightSourceDiffuse(int id)
{
  return gl_LightSource[id].diffuse;
}

/**
 * Get the light specular color.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light specular color for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec4 OivLightSourceSpecular(int id)
{
  return gl_LightSource[id].specular;
}

/**
 * Get the light constant attenuation value.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light constant attenuation value for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceConstantAttenuation(int id)
{
  return gl_LightSource[id].constantAttenuation;
}

/**
 * Get the light linear attenuation value.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light linear attenuation value for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceLinearAttenuation(int id)
{
  return gl_LightSource[id].linearAttenuation;
}

/**
 * Get the light quadratic attenuation value.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light quadratic attenuation value for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceQuadraticAttenuation(int id)
{
  return gl_LightSource[id].quadraticAttenuation;
}

/*****************************************************************************/
// Material

/**
 * Get front material ambient color.
 * @return ambient color for front material.
 */
vec4 OivFrontMaterialAmbient()
{
  return gl_FrontMaterial.ambient;
}

/**
 * Get front material diffuse color.
 * @return diffuse color for front material.
 */
vec4 OivFrontMaterialDiffuse()
{
  return gl_FrontMaterial.diffuse;
}

/**
 * Get front material specular color.
 * @return specular color for front material.
 */
vec4 OivFrontMaterialSpecular()
{
  return gl_FrontMaterial.specular;
}

/**
 * Get front material emissive color.
 * @return emissive color for front material.
 */
vec4 OivFrontMaterialEmissive()
{
  return gl_FrontMaterial.emission;
}

/**
 * Get front material shininess value.
 * @return shininess value for front material.
 */
float OivFrontMaterialShininess()
{
  return gl_FrontMaterial.shininess;
}

/**
 * Get back material ambient color.
 * @return ambient color for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackMaterialAmbient()
{
  return gl_BackMaterial.ambient;
}

/**
 * Get back material diffuse color.
 * @return diffuse color for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackMaterialDiffuse()
{
  return gl_BackMaterial.diffuse;
}

/**
 * Get back material specular color.
 * @return specular color for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackMaterialSpecular()
{
  return gl_BackMaterial.specular;
}

/**
 * Get back material emissive color.
 * @return emissive color for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackMaterialEmissive()
{
  return gl_BackMaterial.emission;
}

/**
 * Get back material shininess value.
 * @return shininess value for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
float OivBackMaterialShininess()
{
  return gl_BackMaterial.shininess;
}

/**
 * Get sum of emissive component of front material and product
 * between front material ambient and global ambient value.
 * @return sum of emissive component of front material and product
 * between front material ambient and global ambient value.
 */
vec4 OivFrontSceneColor()
{
  return gl_FrontLightModelProduct.sceneColor;
}

/**
 * Get product of ambient component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of ambient component between back material and id'th light.
 */
vec4 OivFrontLightProductAmbient(int id)
{
  return gl_FrontLightProduct[id].ambient;
}

/**
 * Get product of diffuse component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of diffuse component between back material and id'th light.
 */
vec4 OivFrontLightProductDiffuse(int id)
{
  return gl_FrontLightProduct[id].diffuse;
}

/**
 * Get product of specular component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of specular component between back material and id'th light.
 */
vec4 OivFrontLightProductSpecular(int id)
{
  return gl_FrontLightProduct[id].specular;
}

/**
 * Get product of ambient component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of ambient component between back material and id'th light.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackLightProductAmbient(int id)
{
  return gl_BackLightProduct[id].ambient;
}

/**
 * Get product of diffuse component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of diffuse component between back material and id'th light.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackLightProductDiffuse(int id)
{
  return gl_BackLightProduct[id].diffuse;
}

/**
 * Get product of specular component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of specular component between back material and id'th light.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackLightProductSpecular(int id)
{
  return gl_BackLightProduct[id].specular;
}

/*****************************************************************************/
// Clip planes

/**
 * Get the clip plane coordinates.
 * @param id specifies which clip plane id to ask, must be between 0 and OivMaxClipPlanes()-1.
 * @return the clip plane .
 */
vec4 OivClipPlane(int id)
{
  return gl_ClipPlane[id];
}

/**
 * Ask if clip plane is enabled.
 * @param id specifies which clip plane id to ask, must be between 0 and OivMaxClipPlanes()-1.
 * @return true if clip plane with a given id is enabled, false otherwise.
 */
bool OivClipPlaneEnabled(int id)
{
  return (OivClipPlane(id) != vec4(0.0));
}

/*****************************************************************************/
// Fog parameters

/**
 * Get the fog color.
 * @return fog color.
 */
vec4 OivFogColor()
{
  return gl_Fog.color;
}


/**
 * Get fog density used in exponential fog equations.
 * @return fog density.
 */
float OivFogDensity()
{
  return gl_Fog.density;
}


/**
 * Get the near distance used in the linear fog equation.
 * @return fog start.
 */
float OivFogStart()
{
  return gl_Fog.start;
}


/**
 * Get the far distance used in the linear fog equation.
 * @return fog end.
 */
float OivFogEnd()
{
  return gl_Fog.end;
}


/**
 * Get the fog scale derived from start and end parameters : 1.0 / (end - start)
 * @return fog scale.
 */
float OivFogScale()
{
  return gl_Fog.scale;
}

/**
* Get the user-defined planes for texture coordinate generation
* @return user-defined plane
*/
vec4 OivObjectPlaneS(int i)
{
  return gl_ObjectPlaneS[i];
}

/**
* Get the user-defined planes for texture coordinate generation
* @return user-defined plane
*/
vec4 OivObjectPlaneT(int i)
{
  return gl_ObjectPlaneT[i];
}

/**
* Get the user-defined planes for texture coordinate generation
* @return user-defined plane
*/
vec4 OivObjectPlaneR(int i)
{
  return gl_ObjectPlaneR[i];
}

/**
* Get the user-defined planes for texture coordinate generation
* @return user-defined plane
*/
vec4 OivObjectPlaneQ(int i)
{
  return gl_ObjectPlaneQ[i];
}

#endif /* OIV_SHADER_STATE_H */
