#ifndef OIV_SHADER_VARIABLES_H
#define OIV_SHADER_VARIABLES_H

//!oiv_include <Inventor/oivShaderStages.h>

/**
 * @ingroup InventorShaders
 * @file
 * This file contains fonction to acces built-in GLSL varyings
 */


/*****************************************************************************/
/** IN **/

#if (OIV_SHADER_STAGE == OIV_FRAGMENT_SHADER)
/**
 * Get the interpolated color for this fragment
 * (as defined by OivSetFrontColor and OivSetBackColor in previous stages).
 * Available in:
 *   - Fragment Shader
 */
vec4 OivFragmentColor()
{
  return gl_Color;
}

/**
 * Get the interpolated texture coordinates for this fragment.
 * Available in:
 *   - Fragment Shader
 */
vec4 OivFragmentTexCoord( in int unit )
{
#if !defined(OIV_USE_NVIDIA)
  switch (unit)
  {
    default:
    case 0:
      return gl_TexCoord[0];
    case 1:
      return gl_TexCoord[1];
    case 2:
      return gl_TexCoord[2];
    case 3:
      return gl_TexCoord[3];
    case 4:
      return gl_TexCoord[4];
    case 5:
      return gl_TexCoord[5];
    case 6:
      return gl_TexCoord[6];
    case 7:
      return gl_TexCoord[7];
  }
#else
  return gl_TexCoord[unit];
#endif
}

#endif // OIV_SHADER_STAGE

#if (OIV_SHADER_STAGE == OIV_VERTEX_SHADER) || (OIV_SHADER_STAGE == OIV_FRAGMENT_SHADER)
/**
 * Get the fog frag coord of the input vertex or fragment.
 * Available in:
 *   - Vertex Shader
 *   - Fragment Shader
 */
float OivFogFragCoord()
{
#if (OIV_SHADER_STAGE == OIV_VERTEX_SHADER)
  return gl_FogCoord;
#else
  return gl_FogFragCoord;
#endif
}
#endif // OIV_SHADER_STAGE

#if (OIV_SHADER_STAGE == OIV_GEOMETRY_SHADER) || (OIV_SHADER_STAGE == OIV_TESSELLATION_CONTROL_SHADER) || (OIV_SHADER_STAGE == OIV_TESSELLATION_EVALUATION_SHADER)
/**
 * Get the front color of the input vertex or fragment at index.
 * Available in:
 *   - Tessellation Control Shader
 *   - Tessellation Evaluation Shader
 *   - Geometry Shader
 */
vec4 OivFrontColor( in int index )
{
  return gl_in[index].gl_FrontColor;
}

/**
 * Get the back color of the input vertex or fragment at index.
 * Available in:
 *   - Tessellation Control Shader
 *   - Tessellation Evaluation Shader
 *   - Geometry Shader
 */
vec4 OivBackColor( in int index )
{
  return gl_in[index].gl_BackColor;
}

/**
 * Get the texture coordinates of the input vertex at index.
 * Available in:
 *   - Tessellation Control Shader
 *   - Tessellation Evaluation Shader
 *   - Geometry Shader
 */
vec4 OivTexCoord( in int index, in int unit )
{
#if !defined(OIV_USE_NVIDIA)
  switch (unit)
  {
    default:
    case 0:
      return gl_in[index].gl_TexCoord[0];
    case 1:
      return gl_in[index].gl_TexCoord[1];
    case 2:
      return gl_in[index].gl_TexCoord[2];
    case 3:
      return gl_in[index].gl_TexCoord[3];
    case 4:
      return gl_in[index].gl_TexCoord[4];
    case 5:
      return gl_in[index].gl_TexCoord[5];
    case 6:
      return gl_in[index].gl_TexCoord[6];
    case 7:
      return gl_in[index].gl_TexCoord[7];
  }
#else
  return gl_in[index].gl_TexCoord[unit];
#endif
}

/**
 * Get the fog frag coord of the input vertex at index.
 * Available in:
 *   - Tessellation Control Shader
 *   - Tessellation Evaluation Shader
 *   - Geometry Shader
 */
float OivFogFragCoord( in int index )
{
  return gl_in[index].gl_FogFragCoord;
}

#endif // OIV_SHADER_STAGE

/*****************************************************************************/
/** OUT **/

/**
 * Some GPU require vertex data to be initialized even if you don't use them.
 * If you encounter message error such as:
 * "The fragment shader uses varying _I;OIV_VERTEX_DATA;xxx, but previous shader does not write to it.",
 * you should call this method at the beginning of your vertex shader main.
 */
void OivInitVertexData()
{
}

#if (OIV_SHADER_STAGE != OIV_FRAGMENT_SHADER)
/**
 * Set the front color of the output vertex for the next shader stage.
 * Available in:
 *   - Vertex Shader
 *   - Tessellation Control Shader
 *   - Tessellation Evaluation Shader
 *   - Geometry Shader
 */
void OivSetFrontColor( in vec4 color )
{
#if (OIV_SHADER_STAGE == OIV_TESSELLATION_CONTROL_SHADER)
  gl_out[gl_InvocationID].gl_FrontColor = color;
#else
  gl_FrontColor = color;
#endif
}

/**
 * Set the back color of the output vertex for the next shader stage.
 * Available in:
 *   - Vertex Shader
 *   - Tessellation Control Shader
 *   - Tessellation Evaluation Shader
 *   - Geometry Shader
 */
void OivSetBackColor( in vec4 color )
{
#if (OIV_SHADER_STAGE == OIV_TESSELLATION_CONTROL_SHADER)
  gl_out[gl_InvocationID].gl_BackColor = color;
#else
  gl_BackColor = color;
#endif
}

/**
 * Set the texture coordinates of the output vertex for the next shader stage.
 * Available in:
 *   - Vertex Shader
 *   - Tessellation Control Shader
 *   - Tessellation Evaluation Shader
 *   - Geometry Shader
 */
void OivSetTexCoord( in int unit, in vec4 coord )
{
#if (OIV_SHADER_STAGE == OIV_TESSELLATION_CONTROL_SHADER)
# if !defined(OIV_USE_NVIDIA)
  switch (unit)
  {
    default:
    case 0:
      gl_out[gl_InvocationID].gl_TexCoord[0] = coord;
    case 1:
      gl_out[gl_InvocationID].gl_TexCoord[1] = coord;
    case 2:
      gl_out[gl_InvocationID].gl_TexCoord[2] = coord;
    case 3:
      gl_out[gl_InvocationID].gl_TexCoord[3] = coord;
    case 4:
      gl_out[gl_InvocationID].gl_TexCoord[4] = coord;
    case 5:
      gl_out[gl_InvocationID].gl_TexCoord[5] = coord;
    case 6:
      gl_out[gl_InvocationID].gl_TexCoord[6] = coord;
    case 7:
      gl_out[gl_InvocationID].gl_TexCoord[7] = coord;
  }
# else
  gl_out[gl_InvocationID].gl_TexCoord[unit] = coord;
# endif
#else
# if !defined(OIV_USE_NVIDIA)
  switch (unit)
  {
    default:
    case 0:
      gl_TexCoord[0] = coord;
    case 1:
      gl_TexCoord[1] = coord;
    case 2:
      gl_TexCoord[2] = coord;
    case 3:
      gl_TexCoord[3] = coord;
    case 4:
      gl_TexCoord[4] = coord;
    case 5:
      gl_TexCoord[5] = coord;
    case 6:
      gl_TexCoord[6] = coord;
    case 7:
      gl_TexCoord[7] = coord;
  }
# else
  gl_TexCoord[unit] = coord;
# endif
#endif
}

/**
 * Set the fog frag coord of the output vertex for the next shader stage.
 * Available in:
 *   - Vertex Shader
 *   - Tessellation Control Shader
 *   - Tessellation Evaluation Shader
 *   - Geometry Shader
 */
void OivSetFogFragCoord( in float coord )
{
#if (OIV_SHADER_STAGE == OIV_TESSELLATION_CONTROL_SHADER)
  gl_out[gl_InvocationID].gl_FogFragCoord = coord;
#else
  gl_FogFragCoord = coord;
#endif
}

#endif // OIV_SHADER_STAGE

#if (OIV_SHADER_STAGE == OIV_FRAGMENT_SHADER)

/**
 * Set the output value of the current fragment in the buffer specified by bufferIndex.
 * Available in:
 *   - Fragment Shader
 */
void OivFragmentOutput( in int bufferIndex, in vec4 outputValue )
{
  gl_FragData[bufferIndex] = outputValue;
}

/**
 * Set the output value of the current fragment.
 * This is a convenient function that calls OivFragmentOutput(0, outputValue)
 * Available in:
 *   - Fragment Shader
 */
void OivFragmentOutput( in vec4 outputValue )
{
  OivFragmentOutput(0, outputValue);
}

#endif // OIV_SHADER_STAGE

#endif /* OIV_SHADER_VARIABLES_H */
