package com.openinventor.inventor.viewercomponents.awt;

import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;

/**
 * Interface for interactive render area.<br>
 * An interactive render area has a {@code SceneInteractor} node as root of the
 * scene graph.
 *
 * @see SceneInteractor
 *
 */
public interface IRenderAreaInteractive extends IRenderArea
{

  /**
   * Camera clipping planes adjustment mode.<br>
   * When adjustment mode is set to {@code AUTO}, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.<br>
   * When adjustment mode is set to {@code MANUAL}, the user is expected to
   * manually set the camera near and far planes.
   *
   */
  public enum ClippingPlanesAdjustMode
  {
    AUTO, MANUAL
  }

  /**
   * Get the camera clipping planes adjustment mode.
   */
  public ClippingPlanesAdjustMode getClippingPlanesAdjustMode();

  /**
   * Set the camera clipping planes adjustment mode. <br>
   * When adjustment mode is set to {@code AUTO}, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.<br>
   * When adjustment mode is set to {@code MANUAL}, the user is expected to
   * manually set those planes. Updating clipping planes after a camera move is
   * not enough, if a dragger or a rendered shape is moved, they can disappear
   * or become partially clipped.<br>
   * Default is {@code AUTO}.
   */
  public void setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode);

  /**
   * Gets the root scene graph of this render area.
   *
   * @return the root scene graph
   */
  public SceneInteractor getRootSceneGraph();

  /**
   * Set the camera to view all the scene.
   *
   * @param viewport
   *          the viewport region used to compute the bounding box of the scene.
   */
  public void viewAll(SbViewportRegion viewport);

  /**
   * Moves the camera to be aligned with the given direction vector while
   * keeping the "up" direction of the camera parallel to the specified up
   * vector.
   *
   * @param direction
   *          the new view direction vector
   * @param up
   *          the new "up" direction
   */
  public void viewAxis(SbVec3f direction, SbVec3f up);

  /**
   * Save current camera. The values of the camera fields are saved and can be
   * restored by calling {@link #restoreCamera()}.
   */
  public void saveCamera();

  /**
   * Restore saved camera. The values of the camera fields are restored to the
   * last saved values.
   */
  public void restoreCamera();

  /**
   * Set camera mode to perspective or orthographic.
   *
   * @param mode
   *          the camera mode: perspective or orthographic
   */
  public void setCameraType(CameraMode mode);

  /**
   * Gets stereo support.
   *
   * @return true if stereo is supported
   */
  public boolean isStereoSupported();

  /**
   * Sets the stereo offset (the distance of each eye from the camera position).
   * The right eye is moved plus offset and the left eye is moved minus offset.
   *
   * @param offset
   *          the stereo offset
   */
  public void setStereoCameraOffset(float offset);

  /**
   * Sets the stereo balance (the position of the zero parallax plane).
   *
   * @param balance
   *          the stereo balance
   */
  public void setStereoCameraBalance(float balance);

}
