package com.openinventor.inventor.viewercomponents.awt.glcanvas.renderareas;

import java.awt.Canvas;
import java.awt.GraphicsDevice;
import java.util.List;

import com.jogamp.nativewindow.CapabilitiesImmutable;
import com.jogamp.nativewindow.NativeWindowException;
import com.jogamp.opengl.DefaultGLCapabilitiesChooser;
import com.jogamp.opengl.GLAutoDrawable;
import com.jogamp.opengl.GLCapabilities;
import com.jogamp.opengl.GLCapabilitiesChooser;
import com.jogamp.opengl.GLCapabilitiesImmutable;
import com.jogamp.opengl.GLEventListener;
import com.jogamp.opengl.awt.GLCanvas;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.devices.SoGLContext;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.viewercomponents.SoRawStereoParameters;
import com.openinventor.inventor.viewercomponents.SoRenderAreaCore;
import com.openinventor.inventor.viewercomponents.awt.IRenderArea;
import com.openinventor.inventor.viewercomponents.awt.Timer;

public class RenderArea extends GLCanvas implements GLEventListener, IRenderArea
{
  protected SoRenderAreaCore m_renderAreaCore;
  protected SoNode m_sceneGraph;
  protected SoGLRenderAction.TransparencyTypes m_transparencyType;
  private SoGLRenderAction m_glRenderAction;

  public RenderArea()
  {
    this(null, new OptionalStereoGLCapabilitiesChooser(), null);
  }

  public RenderArea(GLCapabilities caps)
  {
    this(caps, new OptionalStereoGLCapabilitiesChooser(), null);
  }

  private RenderArea(final GLCapabilitiesImmutable capsReqUser, final GLCapabilitiesChooser chooser,
      final GraphicsDevice device)
  {
    super(capsReqUser, chooser, device);

    m_renderAreaCore = null;
    m_sceneGraph = null;

    // init the system timer with a javax.swing.Timer
    SoDB.setSystemTimer(new Timer());

    addGLEventListener(this);

    m_transparencyType = SoGLRenderAction.TransparencyTypes.BLEND;
    m_glRenderAction = null;
  }

  @Override
  public void setTransparencyType(SoGLRenderAction.TransparencyTypes type)
  {
    m_transparencyType = type;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setTransparencyType(m_transparencyType);
  }

  @Override
  public SoGLRenderAction.TransparencyTypes getTransparencyType()
  {
    return m_transparencyType;
  }

  @Override
  public void setGLRenderAction(SoGLRenderAction ra)
  {
    m_glRenderAction = ra;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.getSceneManager().setGLRenderAction(ra);
  }

  @Override
  public SoGLRenderAction getGLRenderAction()
  {
    return m_glRenderAction;
  }

  @Override
  public void activateStereo(boolean activated)
  {
    if ( activated )
    {
      // activate raw stereo
      if ( !isRawStereoAvailable() )
        throw new UnsupportedOperationException("Stereo buffers are not enabled");

      SoRawStereoParameters params = new SoRawStereoParameters();
      m_renderAreaCore.setStereoParameters(params);
      m_renderAreaCore.activateStereo(true);
    }
    else
      m_renderAreaCore.activateStereo(false);
  }

  @Override
  public boolean isRawStereoAvailable()
  {
    return getChosenGLCapabilities().getStereo();
  }

  @Override
  public void setSceneGraph(SoNode sceneGraph)
  {
    m_sceneGraph = sceneGraph;

    if ( m_renderAreaCore != null )
      m_renderAreaCore.setSceneGraph(sceneGraph);
  }

  @Override
  public Canvas getComponent()
  {
    return this;
  }

  @Override
  public void scheduleRedraw()
  {
    if ( m_renderAreaCore != null )
      m_renderAreaCore.getSceneManager().scheduleRedraw();
  }

  protected void render()
  {
    m_renderAreaCore.render();
  }

  @Override
  public void init(GLAutoDrawable drawable)
  {
    // let OpenInventor be aware of the current context created by JOGL
    SoGLContext context = SoGLContext.getCurrent(true);
    m_renderAreaCore = new SoRenderAreaCore(context);
    m_renderAreaCore.setSceneGraph(m_sceneGraph);
    m_renderAreaCore.setTransparencyType(m_transparencyType);
    if ( m_glRenderAction == null )
      m_glRenderAction = m_renderAreaCore.getSceneManager().getGLRenderAction();
    else
      m_renderAreaCore.getSceneManager().setGLRenderAction(m_glRenderAction);
  }

  @Override
  public void dispose(GLAutoDrawable drawable)
  {
    m_renderAreaCore.dispose();
  }

  @Override
  public void dispose()
  {
    destroy();
  }

  @Override
  public void display(GLAutoDrawable drawable)
  {
    // render Open Inventor scene graph
    render();
  }

  @Override
  public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height)
  {
    if ( width < 0 || height < 0 )
      return;
    m_renderAreaCore.setSize(new SbVec2i32(width, height));
  }

}

/**
 * Custom implementation of GLCapabilitiesChooser.<br>
 * The DefaultGLCapabilitiesChooser does not allow to define an optional but not
 * mandatory GL capability. In particular, stereo capability is considered as a
 * mandatory capability.<br>
 * This implementation allows selecting an optional stereo capability: if no
 * stereo capability is available, it continues to search other requested
 * (mandatory) capabilities.
 */
class OptionalStereoGLCapabilitiesChooser extends DefaultGLCapabilitiesChooser
{
  @Override
  public int chooseCapabilities(CapabilitiesImmutable desired, List<? extends CapabilitiesImmutable> available,
      int windowSystemRecommendedChoice)
  {
    int chosen = -1;
    try
    {
      // launch default chooser
      chosen = super.chooseCapabilities(desired, available, windowSystemRecommendedChoice);
    }
    catch (NativeWindowException exc)
    {
      // chosen index is not valid, check if stereo capability has been
      // requested
      final GLCapabilities gldes = (GLCapabilities) desired.cloneMutable();
      if ( gldes.getStereo() )
      {
        // stereo capability has been requested but it is not available.
        System.err.println("Could not enable stereo buffers");

        // try to get the best config with no stereo
        // if we don't do that, the first available config will be chosen
        // without paying attention to others requested GL capabilities.
        gldes.setStereo(false);
        chosen = super.chooseCapabilities(gldes, available, windowSystemRecommendedChoice);
      }
      else
        // let JOGL do its job...
        throw exc;
    }

    return chosen;
  }
}
