package com.openinventor.inventor.viewercomponents.awt.newt.renderareas;

import java.awt.Canvas;
import java.util.List;

import javax.swing.SwingUtilities;

import com.jogamp.nativewindow.CapabilitiesImmutable;
import com.jogamp.nativewindow.NativeWindowException;
import com.jogamp.newt.awt.NewtCanvasAWT;
import com.jogamp.newt.opengl.GLWindow;
import com.jogamp.opengl.DefaultGLCapabilitiesChooser;
import com.jogamp.opengl.GLAutoDrawable;
import com.jogamp.opengl.GLCapabilities;
import com.jogamp.opengl.GLEventListener;
import com.jogamp.opengl.GLProfile;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.devices.SoGLContext;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.viewercomponents.SoRawStereoParameters;
import com.openinventor.inventor.viewercomponents.SoRenderAreaCore;
import com.openinventor.inventor.viewercomponents.awt.IRenderArea;
import com.openinventor.inventor.viewercomponents.awt.Timer;

public class RenderArea extends NewtCanvasAWT implements GLEventListener, IRenderArea
{

  protected SoRenderAreaCore m_renderAreaCore;
  protected SoNode m_sceneGraph;
  protected SoGLRenderAction.TransparencyTypes m_transparencyType;
  private SoGLRenderAction m_glRenderAction;

  // JOGL
  protected GLWindow m_glWindow;

  public RenderArea()
  {
    this(false);
  }

  public RenderArea(boolean enableStereo)
  {
    super();

    m_renderAreaCore = null;
    m_sceneGraph = null;

    // init the system timer with a javax.swing.Timer
    SoDB.setSystemTimer(new Timer());

    // default configuration
    GLProfile profile = GLProfile.getDefault();

    // define requested GL capabilities
    final GLCapabilities glCapabilities = new GLCapabilities(profile);
    glCapabilities.setDoubleBuffered(true);
    glCapabilities.setHardwareAccelerated(true);
    glCapabilities.setStereo(enableStereo);
    glCapabilities.setDepthBits(24);
    glCapabilities.setStencilBits(8);
    glCapabilities.setSampleBuffers(false);
    glCapabilities.setNumSamples(0);
    glCapabilities.setAlphaBits(8);

    // custom GL capabilities chooser
    OptionalStereoGLCapabilitiesChooser chooser = new OptionalStereoGLCapabilitiesChooser();

    // create a new GLWindow with default profile
    m_glWindow = GLWindow.create(glCapabilities);
    m_glWindow.setCapabilitiesChooser(chooser);
    m_glWindow.addGLEventListener(this);

    m_transparencyType = SoGLRenderAction.TransparencyTypes.BLEND;
    m_glRenderAction = null;

    setNEWTChild(m_glWindow);
  }

  @Override
  public void setTransparencyType(SoGLRenderAction.TransparencyTypes type)
  {
    m_transparencyType = type;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.setTransparencyType(m_transparencyType);
  }

  @Override
  public SoGLRenderAction.TransparencyTypes getTransparencyType()
  {
    return m_transparencyType;
  }

  @Override
  public void setGLRenderAction(SoGLRenderAction ra)
  {
    m_glRenderAction = ra;
    if ( m_renderAreaCore != null )
      m_renderAreaCore.getSceneManager().setGLRenderAction(ra);
  }

  @Override
  public SoGLRenderAction getGLRenderAction()
  {
    return m_glRenderAction;
  }

  @Override
  public Canvas getComponent()
  {
    return this;
  }

  @Override
  public void activateStereo(boolean activated)
  {
    if ( activated )
    {
      // activate raw stereo
      if ( !isRawStereoAvailable() )
        throw new UnsupportedOperationException("Stereo buffers are not enabled");

      SoRawStereoParameters params = new SoRawStereoParameters();
      m_renderAreaCore.setStereoParameters(params);
      m_renderAreaCore.activateStereo(true);
    }
    else
      m_renderAreaCore.activateStereo(false);
  }

  @Override
  public boolean isRawStereoAvailable()
  {
    return m_glWindow.getChosenGLCapabilities().getStereo();
  }

  @Override
  public void setSceneGraph(SoNode sceneGraph)
  {
    m_sceneGraph = sceneGraph;

    if ( m_renderAreaCore != null )
      m_renderAreaCore.setSceneGraph(sceneGraph);
  }

  @Override
  public void scheduleRedraw()
  {
    if ( m_renderAreaCore != null )
      m_renderAreaCore.getSceneManager().scheduleRedraw();
  }

  protected void render()
  {
    m_renderAreaCore.render();
  }

  @Override
  public void init(GLAutoDrawable drawable)
  {
    // let OpenInventor be aware of the current context created by JOGL
    // Note: context returned by SoGLContext.getCurrent() is considered bound.
    // So it must be manually unbound when no more used
    SoGLContext context = SoGLContext.getCurrent(true);
    m_renderAreaCore = new SoRenderAreaCore(context);
    m_renderAreaCore.setSceneGraph(m_sceneGraph);
    m_renderAreaCore.setTransparencyType(m_transparencyType);
    if ( m_glRenderAction == null )
      m_glRenderAction = m_renderAreaCore.getSceneManager().getGLRenderAction();
    else
      m_renderAreaCore.getSceneManager().setGLRenderAction(m_glRenderAction);
    context.unbind();
  }

  @Override
  public void dispose(GLAutoDrawable drawable)
  {
    m_renderAreaCore.dispose();
  }

  @Override
  public void dispose()
  {
    destroy();
  }

  @Override
  public void display(GLAutoDrawable drawable)
  {
    // Render scene in EDT
    if ( SwingUtilities.isEventDispatchThread() )
    {
      // render Open Inventor scene graph
      render();
    }
    else
      SwingUtilities.invokeLater(new Runnable()
      {

        @Override
        public void run()
        {
          m_glWindow.display();
        }
      });
  }

  @Override
  public void reshape(GLAutoDrawable drawable, int x, int y, final int width, final int height)
  {
    // Update viewport region in EDT
    if ( SwingUtilities.isEventDispatchThread() )
      m_renderAreaCore.setSize(new SbVec2i32(width, height));
    else
      SwingUtilities.invokeLater(new Runnable()
      {

        @Override
        public void run()
        {
          m_renderAreaCore.setSize(new SbVec2i32(width, height));
        }
      });

  }

  /**
   * Custom implementation of GLCapabilitiesChooser.<br>
   * The DefaultGLCapabilitiesChooser does not allow to define an optional but
   * not mandatory GL capability. In particular, stereo capability is considered
   * as a mandatory capability.<br>
   * This implementation allows selecting an optional stereo capability: if no
   * stereo capability is available, it continues to search other requested
   * (mandatory) capabilities.
   */
  class OptionalStereoGLCapabilitiesChooser extends DefaultGLCapabilitiesChooser
  {
    @Override
    public int chooseCapabilities(CapabilitiesImmutable desired, List<? extends CapabilitiesImmutable> available,
        int windowSystemRecommendedChoice)
    {
      int chosen = -1;
      try
      {
        // launch default chooser
        chosen = super.chooseCapabilities(desired, available, windowSystemRecommendedChoice);
      }
      catch (NativeWindowException exc)
      {
        // chosen index is not valid, check if stereo capability has been
        // requested
        final GLCapabilities gldes = (GLCapabilities) desired.cloneMutable();
        if ( gldes.getStereo() )
        {
          // stereo capability has been requested but it is not available.
          System.err.println("Could not enable stereo buffers");

          // try to get the best config with no stereo
          // if we don't do that, the first available config will be chosen
          // without paying attention to others requested GL capabilities.
          gldes.setStereo(false);
          chosen = super.chooseCapabilities(gldes, available, windowSystemRecommendedChoice);
        }
        else
          // let JOGL do its job...
          throw exc;
      }

      return chosen;
    }
  }
}
