package com.openinventor.inventor.viewercomponents.awt.newt.renderareas;

import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.io.IOException;
import java.net.URLConnection;

import com.jogamp.common.util.IOUtil;
import com.jogamp.newt.Display;
import com.jogamp.newt.Display.PointerIcon;
import com.jogamp.opengl.util.PNGPixelRect;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionMode;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionModeListener;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.NavigationMode;

public class RenderAreaExaminer extends RenderAreaInteractive implements IRenderAreaExaminer
{
  private SceneExaminer m_examinerRootSceneGraph;
  private PointerIcon m_seekIcon;
  private PointerIcon m_viewingIcon;

  public RenderAreaExaminer()
  {
    this(false);
  }

  public RenderAreaExaminer(boolean enableStereo)
  {
    super(false, enableStereo);

    // Disable SceneExaminer automatic adjustment of clipping planes,
    // adjustment is managed by the render area.
    SoPreferences.setValue("OIV_SCENE_EXAMINER_AUTO_CLIPPING_PLANES", "false");
    m_rootSceneGraph = m_examinerRootSceneGraph = new SceneExaminer();

    buildSceneGraph();

    m_seekIcon = null;
    m_viewingIcon = null;

    m_examinerRootSceneGraph.addInteractionModeListener(new InteractionModeListener()
    {
      @Override
      public void seekModeChanged(boolean onOrOff)
      {
        updateSeekCursor(onOrOff);
      }

      @Override
      public void interactionModeChanged(InteractionMode newMode)
      {
        updateInteractionCursor();
      }
    });
    addHierarchyListener(new HierarchyListener()
    {

      @Override
      public void hierarchyChanged(HierarchyEvent e)
      {
        // Mouse's cursor can be updated only when the native display is valid
        // and ready to operate, i.e. when the render area is visible.
        if ( (e.getChangeFlags() & HierarchyEvent.SHOWING_CHANGED) != 0 && isShowing() )
          updateInteractionCursor();
      }
    });
  }

  @Override
  public void setSeekMode(boolean onOrOff)
  {
    m_examinerRootSceneGraph.setSeekMode(onOrOff);
  }

  @Override
  public void setNavigationMode(NavigationMode mode)
  {
    m_examinerRootSceneGraph.setNavigationMode(mode);
  }

  @Override
  public NavigationMode getNavigationMode()
  {
    return m_examinerRootSceneGraph.getNavigationMode();
  }

  @Override
  public void setInteractionMode(InteractionMode mode)
  {
    m_examinerRootSceneGraph.setInteractionMode(mode);
  }

  @Override
  public InteractionMode getInteractionMode()
  {
    return m_examinerRootSceneGraph.getInteractionMode();
  }

  @Override
  public void addInteractionModeListener(InteractionModeListener listener)
  {
    m_examinerRootSceneGraph.addInteractionModeListener(listener);
  }

  @Override
  public void removeInteractionModeListener(InteractionModeListener listener)
  {
    m_examinerRootSceneGraph.removeInteractionModeListener(listener);
  }

  @Override
  public SceneExaminer getRootSceneGraph()
  {
    return m_examinerRootSceneGraph;
  }

  private void updateSeekCursor(boolean seek)
  {
    try
    {
      if ( seek )
      {
        if ( m_seekIcon == null )
          m_seekIcon = createSeekCursor();
        m_glWindow.setPointerIcon(m_seekIcon);
      }
      else
        updateInteractionCursor();
    }
    catch (IllegalStateException e)
    {
      // Mouse's cursor can be updated only when the native display is valid
      // and ready to operate, i.e. when the render area is visible.
      // Here, the render area is not ready, cursor will be updated later (see
      // hierarchy listener) so don't print the exception trace.
    }
  }

  private void updateInteractionCursor()
  {
    try
    {
      if ( getInteractionMode() == InteractionMode.NAVIGATION )
      {
        if ( m_viewingIcon == null )
          m_viewingIcon = createViewingCursor();
        m_glWindow.setPointerIcon(m_viewingIcon);
      }
      else
        m_glWindow.setPointerIcon(null);
    }
    catch (IllegalStateException e)
    {
      // Mouse's cursor can be updated only when the native display is valid
      // and ready to operate, i.e. when the render area is visible.
      // Here, the render area is not ready, cursor will be updated later (see
      // hierarchy listener) so don't print the exception trace.
    }
  }

  private PointerIcon createSeekCursor()
  {
    return createCursor("/com/openinventor/inventor/viewercomponents/icons/CursorSeek.png");
  }

  private PointerIcon createViewingCursor()
  {
    return createCursor("/com/openinventor/inventor/viewercomponents/icons/CursorCurvedHand.png");
  }

  private PointerIcon createCursor(String filepath)
  {
    final Display disp = m_glWindow.getScreen().getDisplay();
    if ( !disp.isNativeValid() )
      throw new IllegalStateException("Can't create cursor: Display invalid");

    final IOUtil.ClassResources res =
        new IOUtil.ClassResources(new String[] { filepath }, getClass().getClassLoader(), getClass());
    final URLConnection urlConn = res.resolve(0);
    PNGPixelRect image;
    try
    {
      image = PNGPixelRect.read(urlConn.getInputStream(), null, false, 0, false);
      return disp.createPointerIcon(image, 16, 16);
    }
    catch (IOException e)
    {
      e.printStackTrace();
    }

    return null;
  }
}
