package com.openinventor.inventor.viewercomponents.awt.tools;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.*;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.NavigationMode;

public class PreferencesDialog extends JDialog
{

  public static interface StereoListener
  {
    public void activateRawStereo(boolean activated);

    public void balanceValueChanged(float value);

    public void offsetValueChanged(float value);
  }

  public static interface GeneralPreferencesListener
  {
    public void navigationModeChanged(NavigationMode newMode);

    public void clippingModeChanged(String newMode);
  }

  private static PreferencesDialog m_instance = null;

  private StereoListener stereoListener;
  private GeneralPreferencesListener generalPreferencesListener;

  private JCheckBox rawStereoCheckBox;
  private SliderPanel stereoBalanceSlider;
  private SliderPanel stereoOffsetSlider;

  private final ButtonGroup navButtonGroup = new ButtonGroup();
  private JRadioButton orbitButton;
  private JRadioButton planeButton;

  private JComboBox<String> clippingPlanesComboBox;

  private PreferencesDialog()
  {
    super((JDialog) null, "Preferences");
    setIconImage(new ImageIcon(
        PreferencesDialog.class.getResource("/com/openinventor/inventor/viewercomponents/icons/preferencesMenu.png"))
            .getImage());

    stereoListener = null;
    generalPreferencesListener = null;
    buildUI();
  }

  public static PreferencesDialog getInstance()
  {
    if ( m_instance == null )
      m_instance = new PreferencesDialog();

    return m_instance;
  }

  public void setStereoListener(StereoListener listener)
  {
    stereoListener = listener;
  }

  public StereoListener getStereoListener()
  {
    return stereoListener;
  }

  public void setGeneralPreferencesListener(GeneralPreferencesListener listener)
  {
    generalPreferencesListener = listener;
  }

  public GeneralPreferencesListener getGeneralPreferencesListener()
  {
    return generalPreferencesListener;
  }

  public void enableStereoPreferences(boolean enabled)
  {
    if ( !enabled && rawStereoCheckBox.isSelected() )
    {
      // deactivate raw stereo
      rawStereoCheckBox.doClick();
    }
    rawStereoCheckBox.setEnabled(enabled);
    stereoBalanceSlider.setEnabled(enabled);
    stereoOffsetSlider.setEnabled(enabled);
  }

  public void setNavigationMode(NavigationMode mode)
  {
    switch ( mode )
    {
    case ORBIT :
      orbitButton.setSelected(true);
      break;
    case PLANE :
      planeButton.setSelected(true);
      break;
    default:
      break;
    }
  }

  public void setClippingPlanesAdjustMode(String mode)
  {
    clippingPlanesComboBox.setSelectedItem(mode);
  }

  private void buildUI()
  {
    JPanel stereoPanel = buildStereoPanel();
    JPanel generalPanel = buildGeneralPanel();

    JTabbedPane tabbedPane = new JTabbedPane(JTabbedPane.TOP);
    tabbedPane.addTab("General", null, generalPanel, "");
    tabbedPane.addTab("Stereo",
        new ImageIcon(PreferencesDialog.class
            .getResource("/com/openinventor/inventor/viewercomponents/icons/ViewerStereo24.png")),
        stereoPanel, "Stereo preferences");

    JPanel mainPanel = new JPanel(new BorderLayout());
    mainPanel.add(tabbedPane, BorderLayout.CENTER);
    setContentPane(mainPanel);
    pack();
  }

  private JPanel buildGeneralPanel()
  {
    JPanel generalPanel = new JPanel();
    GridBagLayout gbl_generalPanel = new GridBagLayout();
    gbl_generalPanel.columnWeights = new double[] { 0.0, 0.0, 0.0 };
    gbl_generalPanel.rowWeights = new double[] { 0.0, 0.0 };
    generalPanel.setLayout(gbl_generalPanel);

    JLabel navigationModeLabel = new JLabel("Navigation mode");
    GridBagConstraints gbc_navigationModeLabel = new GridBagConstraints();
    gbc_navigationModeLabel.anchor = GridBagConstraints.WEST;
    gbc_navigationModeLabel.insets = new Insets(0, 0, 5, 5);
    gbc_navigationModeLabel.gridx = 0;
    gbc_navigationModeLabel.gridy = 0;
    generalPanel.add(navigationModeLabel, gbc_navigationModeLabel);

    orbitButton = new JRadioButton("Orbit");
    GridBagConstraints gbc_orbitButton = new GridBagConstraints();
    gbc_orbitButton.anchor = GridBagConstraints.WEST;
    gbc_orbitButton.insets = new Insets(0, 0, 5, 5);
    gbc_orbitButton.gridx = 1;
    gbc_orbitButton.gridy = 0;
    generalPanel.add(orbitButton, gbc_orbitButton);
    orbitButton.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( generalPreferencesListener != null )
          generalPreferencesListener.navigationModeChanged(NavigationMode.ORBIT);
      }
    });
    navButtonGroup.add(orbitButton);
    orbitButton.setHorizontalAlignment(SwingConstants.LEFT);

    planeButton = new JRadioButton("Plane");
    GridBagConstraints gbc_planeButton = new GridBagConstraints();
    gbc_planeButton.anchor = GridBagConstraints.WEST;
    gbc_planeButton.weightx = 1.0;
    gbc_planeButton.insets = new Insets(0, 0, 5, 0);
    gbc_planeButton.gridx = 2;
    gbc_planeButton.gridy = 0;
    generalPanel.add(planeButton, gbc_planeButton);
    planeButton.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( generalPreferencesListener != null )
          generalPreferencesListener.navigationModeChanged(NavigationMode.PLANE);
      }
    });
    navButtonGroup.add(planeButton);
    planeButton.setHorizontalAlignment(SwingConstants.LEFT);

    JLabel clippingModeLabel = new JLabel("Clipping planes mode");
    GridBagConstraints gbc_clippingModeLabel = new GridBagConstraints();
    gbc_clippingModeLabel.anchor = GridBagConstraints.NORTH;
    gbc_clippingModeLabel.weighty = 1.0;
    gbc_clippingModeLabel.insets = new Insets(0, 0, 5, 5);
    gbc_clippingModeLabel.gridx = 0;
    gbc_clippingModeLabel.gridy = 1;
    generalPanel.add(clippingModeLabel, gbc_clippingModeLabel);
    clippingModeLabel.setHorizontalAlignment(SwingConstants.LEFT);

    clippingPlanesComboBox = new JComboBox<String>(new String[] { "AUTO", "MANUAL" });
    GridBagConstraints gbc_clippingPlanesComboBox = new GridBagConstraints();
    gbc_clippingPlanesComboBox.gridwidth = 2;
    gbc_clippingPlanesComboBox.anchor = GridBagConstraints.NORTHWEST;
    gbc_clippingPlanesComboBox.insets = new Insets(0, 0, 5, 5);
    gbc_clippingPlanesComboBox.gridx = 1;
    gbc_clippingPlanesComboBox.gridy = 1;
    generalPanel.add(clippingPlanesComboBox, gbc_clippingPlanesComboBox);
    clippingPlanesComboBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( generalPreferencesListener != null )
          generalPreferencesListener.clippingModeChanged((String) clippingPlanesComboBox.getSelectedItem());
      }
    });

    return generalPanel;
  }

  private JPanel buildStereoPanel()
  {
    rawStereoCheckBox = new JCheckBox("Activate raw stereo");
    rawStereoCheckBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( stereoListener != null )
          stereoListener.activateRawStereo(rawStereoCheckBox.isSelected());
      }
    });

    stereoBalanceSlider = new SliderPanel(-3.f, 3.f, 1.f, 2);
    stereoBalanceSlider.addInfoText("Zero Parallax Balance : ");
    stereoBalanceSlider.setSliderSize(new Dimension(200, 20));
    stereoBalanceSlider.setInfoTextSize(new Dimension(140, 20));
    stereoBalanceSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        if ( stereoListener != null && stereoBalanceSlider.isEnabled() )
          stereoListener.balanceValueChanged(value);
      }
    });

    stereoOffsetSlider = new SliderPanel(0.f, 2.f, 0.7f, 2);
    stereoOffsetSlider.addInfoText("Camera Offset : ");
    stereoOffsetSlider.setSliderSize(new Dimension(200, 20));
    stereoOffsetSlider.setInfoTextSize(new Dimension(140, 20));
    stereoOffsetSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        if ( stereoListener != null && stereoOffsetSlider.isEnabled() )
          stereoListener.offsetValueChanged(value);
      }
    });

    JPanel stereoAdjustmentPanel = new JPanel();
    stereoAdjustmentPanel
        .setBorder(new TitledBorder(null, "Stereo Adjustment", TitledBorder.LEFT, TitledBorder.TOP, null, null));
    stereoAdjustmentPanel.setLayout(new BorderLayout());
    stereoAdjustmentPanel.add(stereoBalanceSlider, BorderLayout.NORTH);
    stereoAdjustmentPanel.add(stereoOffsetSlider, BorderLayout.CENTER);

    JPanel stereoPanel = new JPanel();
    stereoPanel.setLayout(new BorderLayout());
    stereoPanel.add(rawStereoCheckBox, BorderLayout.NORTH);
    stereoPanel.add(stereoAdjustmentPanel, BorderLayout.CENTER);

    return stereoPanel;
  }

}
