package com.openinventor.inventor.viewercomponents.nodes;

import java.util.ArrayList;

import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoHandleEventAction;
import com.openinventor.inventor.actions.SoRayPickAction;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.events.SoMouseWheelEvent;
import com.openinventor.inventor.gestures.events.SoDoubleTapGestureEvent;
import com.openinventor.inventor.gestures.events.SoLongTapGestureEvent;
import com.openinventor.inventor.gestures.events.SoRotateGestureEvent;
import com.openinventor.inventor.gestures.events.SoScaleGestureEvent;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.touch.events.SoTouchEvent;
import com.openinventor.inventor.viewercomponents.SoCameraInteractor;

/**
 * Tool class for easily building a basic interactive OpenInventor application
 * without using existing viewer classes.
 * <p>
 * The SceneExaminer is an extension of the {@link SceneInteractor} node that
 * allows providing the camera and headlight manipulations like panning, zooming
 * and orbiting similar to the behavior of the classic Open Inventor viewer
 * class {@link com.openinventor.inventor.awt.SwSimpleViewer} with an examiner
 * area ({@link NavigationMode#ORBIT}) or a plane area
 * ({@link NavigationMode#PLANE}). Similar behavior includes a 'headlight', i.e.
 * an SoDirectionalLight node automatically aligned with the camera's view
 * direction. <br>
 * The SceneExaminer is not directly comparable with a classic OpenInventor
 * viewer as it does not provide any GUI (no button, no popup menu) and fewer
 * interactive features (no animation). However it does provide a touch event
 * handler that allows manipulating a scene on a touch device.
 * <p>
 * See parent class {@link SceneInteractor} for more details about the structure
 * of the internal scene graph. <br>
 * The SceneExaminer uses an instance of {@link SoCameraInteractor} to
 * manipulate the camera in response to OpenInventor events.
 *
 * <p>
 * <b>Notes:</b>
 * <ul>
 * <li>Window system integration<br>
 * The SceneExaminer needs a component that integrates the Open Inventor 3D
 * rendering window with the native window system. System dependent tasks
 * include creating a window, placing the window in the application's user
 * interface, initializing OpenGL and processing the native event/message loop.
 * System independent support for this is provided by the
 * {@link com.openinventor.inventor.viewercomponents.SoRenderAreaCore} class.
 * Example components are provided for AWT and SWT toolkits.
 * <li>Event handling<br>
 * The SceneExaminer needs a component that builds OpenInventor events (SoEvent)
 * from native system events. System independent support for this is provided by
 * the {@link com.openinventor.inventor.viewercomponents.SoEventBuilder} class.
 * Example components are provided for AWT and SWT toolkits: AWTEventToSoEvent
 * and SWTEventToSoEvent.
 * <li>Library<br>
 * A basic version of SceneExaminer is a supported part of the Open Inventor
 * API and a prebuilt jar is provided.
 * <li>Source code<br>
 * The basic version of SceneExaminer is also provided as source code in the
 * sources folder to allow applications to customize and build their own
 * interactive tool class.<br>
 * See $OIVJHOME/source/com/openinventor/inventor/viewercomponents/nodes.
 * <li>Interaction Modes<br>
 * Similar to the classic Open Inventor viewer classes, SceneExaminer is either
 * in NAVIGATION mode (the default, similar to viewing mode) or SELECTION mode.
 * The user must press the ESC key to toggle between interaction modes. (Of
 * course this behavior can be modified or replaced by implementing an alternate
 * version of SceneExaminer.) In navigation mode, Open Inventor events are
 * automatically handled to modify the camera as defined in the Usage section
 * below. Events that are defined in this section are not sent to the
 * application scene graph, but all other events are sent to the application
 * scene graph. Also specific viewing behaviors can be disabled as needed (see
 * for example {@link #enableZoom(boolean)}). In selection mode, all events are
 * sent to the application scene graph.
 * <li>Scene graph<br>
 * The application scene graph should be the last child of the SceneExaminer.
 * The initial application scene graph can be added by simply calling the
 * inherited method addChild(). But note that if you need to replace the
 * application scene graph, for example loading a new data set, do <i>not</i>
 * call removeAllChildren(). That would also remove the SceneExaminer's camera,
 * headlight and event handler nodes. Add an SoSeparator to the SceneExaminer to
 * serve as the "scene graph holder", then add and remove the application scene
 * graph from this node.
 * <li>Clip planes<br>
 * SceneExaminer automatically adjusts the 'near' and 'far' clipping planes when
 * events modifying the camera are handled. This adjustment, based on the
 * bounding box of the scene, ensures that shapes will not be clipped as the
 * camera orbits and also that depth buffer precision is maximized. This
 * adjustment is only done in InteractionMode NAVIGATION and can be disabled by
 * setting the environment variable "OIV_SCENE_EXAMINER_AUTO_CLIPPING_PLANES" to
 * false.<br>
 * <b>Note</b>: Updating clipping planes after a camera move can be not
 * sufficient. If the scene graph is modified or if a dragger or a rendered
 * shape is moved, they can disappear or become partially clipped. A classic
 * implementation of a render area must adjust clipping planes before each
 * rendering by calling the provided method
 * {@link #adjustClippingPlanes(SbViewportRegion)}. See render area's
 * implementations available in $OIVJHOME/examples/inventor/viewercomponents/awt
 * and $OIVJHOME/examples/inventor/viewercomponents/swt folders for examples of
 * {@code adjustClippingPlanes} use.
 * </ul>
 * <p>
 *
 * Compatibility<br>
 * Please note that some interaction behaviors are different than the classic
 *   Open Inventor viewer classes :
 * <ul>
 * <li>In @B Orbit@b mode: @B Left Mouse + Shift: @b now : Zoom in/out.</li>
 * <li>In @B Plane@b mode: @B Left Mouse + Middle Mouse or Left Mouse + Shift: @b now : Roll the scene.</li>
 * <li> @B Mouse wheel@b in both modes performs a dolly relative to the @I cursor position@i,
 *      not the center of the viewport.</li>
 * <br>
 * <li>The classic <b>Alt key</b> behavior is not implemented.
 *     This key is reserved for application use.</li>
 * <li>The <b>Right Mouse</b> button does not display a popup menu.
 *     This button is reserved for application use.</li>
 * </ul>
 * <p>
 * <b>Usage:</b>
 * <ul>
 * <li><b>Orbit</b> mode:</li>
 * <ul>
 * <li>With a mouse</li>
 * <ul>
 * <li><b>Left Mouse:</b> Rotate the scene or seek to point if seek mode is
 * activated.</li>
 * <li><b>Middle Mouse or Left Mouse + Ctrl:</b> Pan the scene.</li>
 * <li><b>Left Mouse + Middle Mouse or Middle Mouse + Ctrl or Left Mouse + Shift:</b> Zoom in/out the scene.</li>
 * <li><b>Mouse Wheel:</b> Zoom in/out (zoom center is the mouse cursor location).</li>
 * <li><b>Escape key:</b>  Switch between navigation mode and selection mode.</li>
 * <li><b>S key:</b> Activate/Deactivate seek mode.</li>
 * </ul>
 * <li>With a touchscreen</li>
 * <ul>
 * <li><b>1 finger:</b> Rotate the scene.</li>
 * <li><b>2 fingers:</b> Rotate the scene on the screen plan, zoom in/out and
 * pan (rotation and zoom center are located between the two fingers).</li>
 * <li><b>Double tap:</b> Seek to the point located by the finger.</li>
 * <li><b>Long tap:</b> Enable/Disable selection mode.</li>
 * </ul>
 * </ul>
 * </ul>
 * <ul>
 * <li><b>Plane</b> mode:</li>
 * <ul>
 * <li>With a mouse</li>
 * <ul>
 * <li><b>Left Mouse: </b> Zoom in/out or seek to point if seek mode is activated.</li>
 * <li><b>Middle Mouse or Left Mouse + Ctrl: </b> Pan the scene.</li>
 * <li><b>Left Mouse + Middle Mouse or Middle Mouse + Ctrl or Left Mouse + Shift: </b> Roll the scene.</li>
 * <li><b>Mouse Wheel:</b> Zoom in/out (zoom center is the mouse cursor location).</li>
 * <li><b>Escape key:</b>  Switch between navigation mode and selection mode.</li>
 * <li><b>S key:</b> Activate/Deactivate seek mode.</li>
 * </ul>
 * <li>With a touchscreen</li>
 * <ul>
 * <li><b>1 finger:</b> Pan the scene.</li>
 * <li><b>2 fingers:</b> Rotate the scene on the screen plan, zoom in/out and
 * pan (rotation and zoom center are located between the two fingers).</li>
 * <li><b>Double tap:</b> Seek to the point located by the finger.</li>
 * <li><b>Long tap:</b> Enable/Disable selection mode.</li>
 * </ul>
 * </ul>
 * </ul>
 *
 * Note: On MacOS, some GUI frameworks or viewers may match the Control (Ctrl) value to the Command (CMD) key on the keyboard.
 */
public class SceneExaminer extends SceneInteractor
{

  /**
   * Listener interface to receive notifications of interaction mode changes.
   *
   */
  public interface InteractionModeListener
  {
    /**
     * Invoked when seek mode has changed
     */
    public void seekModeChanged(boolean onOrOff);

    /**
     * Invoked when interaction mode has changed
     */
    public void interactionModeChanged(InteractionMode newMode);

  }

  /**
   * Interaction Mode (navigation or selection)
   */
  public enum InteractionMode
  {
    NAVIGATION, SELECTION
  }

  /**
   * Navigation Mode
   */
  public enum NavigationMode
  {
    ORBIT, PLANE
  }

  private NavigationInteraction m_navigation;
  private SelectionInteraction m_selection;
  private BaseInteraction m_currentInteraction;
  private ArrayList<InteractionModeListener> m_modeListeners;
  private boolean m_isSelectionEnabled;

  public SceneExaminer()
  {
    super();

    m_currentInteraction = m_navigation = new OrbitInteraction(m_cameraInteractor);
    m_selection = new SelectionInteraction();
    m_modeListeners = new ArrayList<InteractionModeListener>();
    m_isSelectionEnabled = true;
  }

  @Override
  public void setCameraMode(CameraMode mode)
  {
    super.setCameraMode(mode);

    if ( m_navigation != null )
      m_navigation.setCameraInteractor(m_cameraInteractor);
  }

  /**
   * Get the current navigation mode.
   */
  public NavigationMode getNavigationMode()
  {
    if ( m_navigation instanceof OrbitInteraction )
      return NavigationMode.ORBIT;
    else
      return NavigationMode.PLANE;
  }

  /**
   * Set navigation mode.
   */
  public void setNavigationMode(NavigationMode mode)
  {
    switch ( mode )
    {
    case ORBIT :
      m_navigation = new OrbitInteraction(m_navigation);
      break;
    case PLANE :
      m_navigation = new PlaneInteraction(m_navigation);
      break;
    default:
      break;
    }
    if ( getInteractionMode() == InteractionMode.NAVIGATION )
      m_currentInteraction = m_navigation;
  }

  /**
   * Get the current interaction mode.
   */
  public InteractionMode getInteractionMode()
  {
    if ( m_currentInteraction instanceof SelectionInteraction )
      return InteractionMode.SELECTION;

    return InteractionMode.NAVIGATION;
  }

  /**
   * Set interaction mode to navigation or selection.
   */
  public void setInteractionMode(InteractionMode mode)
  {
    if ( getInteractionMode() != mode )
    {
      switch ( mode )
      {
      case SELECTION :
        m_navigation.reset();
        m_currentInteraction = m_selection;
        break;
      case NAVIGATION :
        m_currentInteraction = m_navigation;
        break;
      default:
        break;
      }

      for ( InteractionModeListener listener : m_modeListeners )
        listener.interactionModeChanged(mode);
    }
  }

  /**
   * Adds a listener to receive notifications of interaction mode changes.
   */
  public void addInteractionModeListener(InteractionModeListener listener)
  {
    m_modeListeners.add(listener);
    m_navigation.m_modeListeners.add(listener);
  }

  /**
   * Removes the specified listener so that it no longer receives notifications
   * of interaction mode changes.
   */
  public void removeInteractionModeListener(InteractionModeListener listener)
  {
    m_modeListeners.remove(listener);
    m_navigation.m_modeListeners.add(listener);
  }

  /**
   * Enable or disable selection mode.
   */
  public void enableSelection(boolean enabled)
  {
    m_isSelectionEnabled = enabled;
  }

  /**
   * Return if selection is enabled.
   */
  public boolean isSelectionEnabled()
  {
    return m_isSelectionEnabled;
  }

  /**
   * Enable or disable zoom.
   */
  public void enableZoom(boolean enabled)
  {
    m_navigation.m_isZoomEnabled = enabled;
  }

  /**
   * Return if zoom is enabled.
   */
  public boolean isZoomEnabled()
  {
    return m_navigation.m_isZoomEnabled;
  }

  /**
   * Enable or disable camera rotate.
   */
  public void enableRotate(boolean enabled)
  {
    m_navigation.m_isRotateEnabled = enabled;
  }

  /**
   * Return if rotate is enabled.
   */
  public boolean isRotateEnabled()
  {
    return m_navigation.m_isRotateEnabled;
  }

  /**
   * Enable or disable seek.
   */
  public void enableSeek(boolean enabled)
  {
    m_navigation.m_isSeekEnabled = enabled;
  }

  /**
   * Return if seek is enabled.
   */
  public boolean isSeekEnabled()
  {
    return m_navigation.m_isSeekEnabled;
  }

  /**
   * Enable or disable camera panning.
   */
  public void enablePan(boolean enabled)
  {
    m_navigation.m_isPanEnabled = enabled;
  }

  /**
   * Return if camera panning is enabled.
   */
  public boolean isPanEnabled()
  {
    return m_navigation.m_isPanEnabled;
  }

  /**
   * Enable or disable camera orbiting.
   */
  public void enableOrbit(boolean enabled)
  {
    m_navigation.m_isOrbitEnabled = enabled;
  }

  /**
   * Return if camera orbiting is enabled.
   */
  public boolean isOrbitEnabled()
  {
    return m_navigation.m_isOrbitEnabled;
  }

  /**
   * Set the interaction into or out off seek mode (default is off).
   */
  public void setSeekMode(boolean onOrOff)
  {
    if ( getInteractionMode() == InteractionMode.NAVIGATION )
      m_navigation.setSeekMode(onOrOff);
  }

  @Override
  protected void keyPressed(SoKeyboardEvent event, SoHandleEventAction action)
  {
    if ( event.getKey() == SoKeyboardEvent.Keys.ESCAPE )
    {
      switchInteractionMode();
      action.setHandled();
    }
    else
    {
      m_currentInteraction.beginAction(event, action);
    }
  }

  @Override
  protected void keyReleased(SoKeyboardEvent event, SoHandleEventAction action)
  {
    m_currentInteraction.beginAction(event, action);
  }

  @Override
  protected void mousePressed(SoMouseButtonEvent event, SoHandleEventAction action)
  {
    m_currentInteraction.beginAction(event, action);
  }

  @Override
  protected void mouseReleased(SoMouseButtonEvent event, SoHandleEventAction action)
  {
    m_currentInteraction.beginAction(event, action);
  }

  @Override
  protected void mouseWheelMoved(SoMouseWheelEvent event, SoHandleEventAction action)
  {
    m_currentInteraction.doAction(event, action);
  }

  @Override
  protected void mouseMoved(SoLocation2Event event, SoHandleEventAction action)
  {
    m_currentInteraction.doAction(event, action);
  }

  @Override
  protected void touch(SoTouchEvent event, SoHandleEventAction action)
  {
    m_currentInteraction.doAction(event, action);
  }

  @Override
  protected void zoom(SoScaleGestureEvent event, SoHandleEventAction action)
  {
    m_currentInteraction.doAction(event, action);
  }

  @Override
  protected void rotate(SoRotateGestureEvent event, SoHandleEventAction action)
  {
    m_currentInteraction.doAction(event, action);
  }

  @Override
  protected void doubleTap(SoDoubleTapGestureEvent event, SoHandleEventAction action)
  {
    m_currentInteraction.doAction(event, action);
  }

  @Override
  protected void longTap(SoLongTapGestureEvent event, SoHandleEventAction action)
  {
    switchInteractionMode();
    action.setHandled();
  }

  private void switchInteractionMode()
  {
    if ( getInteractionMode() == InteractionMode.SELECTION )
      setInteractionMode(InteractionMode.NAVIGATION);
    else if ( m_isSelectionEnabled )
      setInteractionMode(InteractionMode.SELECTION);
  }
}

abstract class BaseInteraction
{

  public abstract boolean propageEvents();

  public abstract void beginAction(SoMouseButtonEvent event, SoHandleEventAction action);

  public abstract void beginAction(SoKeyboardEvent event, SoHandleEventAction action);

  public abstract void doAction(SoLocation2Event event, SoHandleEventAction action);

  public abstract void doAction(SoTouchEvent event, SoHandleEventAction action);

  public abstract void doAction(SoMouseWheelEvent event, SoHandleEventAction action);

  public abstract void doAction(SoScaleGestureEvent event, SoHandleEventAction action);

  public abstract void doAction(SoRotateGestureEvent event, SoHandleEventAction action);

  public abstract void doAction(SoDoubleTapGestureEvent event, SoHandleEventAction action);

  public abstract void doAction(SoLongTapGestureEvent event, SoHandleEventAction action);

}

abstract class NavigationInteraction extends BaseInteraction
{
  private SoCameraInteractor m_cameraInteractor;

  public boolean m_isZoomEnabled;
  public boolean m_isPanEnabled;
  public boolean m_isOrbitEnabled;
  public boolean m_isRotateEnabled;

  private final boolean m_isAutoClippingPlanes;

  // Seek
  public ArrayList<SceneExaminer.InteractionModeListener> m_modeListeners;
  public boolean m_isSeekEnabled;
  private boolean m_isSeekMode;
  private SeekAnimator m_seekAnimator;

  protected boolean m_isButton1Down;
  protected boolean m_isButton2Down;
  protected SbVec2f m_mouseNormPosition;
  protected int m_mouseWheelDelta;

  public NavigationInteraction(SoCameraInteractor cameraInteractor)
  {
    m_cameraInteractor = cameraInteractor;

    m_isZoomEnabled = true;
    m_isPanEnabled = true;
    m_isOrbitEnabled = true;
    m_isRotateEnabled = true;

    m_isAutoClippingPlanes = SoPreferences.getBoolean("OIV_SCENE_EXAMINER_AUTO_CLIPPING_PLANES", true);

    m_modeListeners = new ArrayList<SceneExaminer.InteractionModeListener>();
    m_isSeekEnabled = true;
    m_isSeekMode = false;
    m_seekAnimator = new SeekAnimator(cameraInteractor, null);
    m_seekAnimator.setListener(new AnimatorListener()
    {
      @Override
      public void animationStarted()
      {}

      @Override
      public void animationStopped()
      {
        setSeekMode(false);
      }
    });

    m_isButton1Down = false;
    m_isButton2Down = false;
    m_mouseNormPosition = new SbVec2f();
    m_mouseWheelDelta = SoPreferences.getInteger("OIV_WHEEL_DELTA", 120);
  }

  public NavigationInteraction(NavigationInteraction copyFrom)
  {
    m_cameraInteractor = copyFrom.m_cameraInteractor;

    m_isZoomEnabled = copyFrom.m_isZoomEnabled;
    m_isPanEnabled = copyFrom.m_isPanEnabled;
    m_isOrbitEnabled = copyFrom.m_isOrbitEnabled;
    m_isRotateEnabled = copyFrom.m_isRotateEnabled;

    m_isAutoClippingPlanes = copyFrom.m_isAutoClippingPlanes;

    m_modeListeners = copyFrom.m_modeListeners;
    m_isSeekEnabled = copyFrom.m_isSeekEnabled;
    m_isSeekMode = copyFrom.m_isSeekMode;
    m_seekAnimator = copyFrom.m_seekAnimator;
    m_seekAnimator.setListener(new AnimatorListener()
    {
      @Override
      public void animationStarted()
      {}

      @Override
      public void animationStopped()
      {
        setSeekMode(false);
      }
    });

    m_isButton1Down = copyFrom.m_isButton1Down;
    m_isButton2Down = copyFrom.m_isButton2Down;
    m_mouseNormPosition = new SbVec2f(copyFrom.m_mouseNormPosition);
    m_mouseWheelDelta = copyFrom.m_mouseWheelDelta;
  }

  public void reset()
  {
    // reset mouse buttons state
    m_isButton1Down = false;
    m_isButton2Down = false;
    // stop seek
    setSeekMode(false);
  }

  public void setCameraInteractor(SoCameraInteractor cameraInteractor)
  {
    m_cameraInteractor = cameraInteractor;
    m_seekAnimator.setCameraInteractor(cameraInteractor);
  }

  /**
   * Set the interaction into or out off seek mode (default is off).
   */
  public void setSeekMode(boolean onOrOff)
  {
    // check if seek is being turned off while seek animation is happening
    if ( !onOrOff )
      m_seekAnimator.stop();

    m_isSeekMode = onOrOff;

    for ( SceneExaminer.InteractionModeListener listener : m_modeListeners )
      listener.seekModeChanged(onOrOff);
  }

  private void doSeek(SoEvent event, SoHandleEventAction action)
  {
    SbViewportRegion vpRegion = action.getViewportRegion();
    SoNode sceneGraph = action.getNodeAppliedTo();

    // do the picking
    SoRayPickAction pick = new SoRayPickAction(vpRegion);
    pick.setSceneManager(action.getSceneManager());
    pick.setNormalizedPoint(event.getNormalizedPosition(vpRegion));
    pick.setRadius(1);
    pick.setPickAll(false); // pick only the closest object
    pick.apply(sceneGraph);

    // makes sure something got picked
    SoPickedPoint pp = pick.getPickedPoint();
    if ( pp == null )
      setSeekMode(false);
    else
    {
      // set up and start animation
      m_seekAnimator.setSceneGraph(sceneGraph);
      m_seekAnimator.setUp(pp.getPoint(), vpRegion);
      m_seekAnimator.start();
    }

    action.setHandled();
  }

  @Override
  public void beginAction(SoMouseButtonEvent event, SoHandleEventAction action)
  {
    SoMouseButtonEvent.Buttons button = event.getButton();
    SoMouseButtonEvent.States state = event.getState();

    if ( button == SoMouseButtonEvent.Buttons.BUTTON1 )
      m_isButton1Down = state == SoMouseButtonEvent.States.DOWN;
    else if ( button == SoMouseButtonEvent.Buttons.BUTTON2 )
      m_isButton2Down = state == SoMouseButtonEvent.States.DOWN;

    if ( m_isButton1Down && m_isSeekMode )
    {
      doSeek(event, action);
    }
    else
    {
      SbViewportRegion vpRegion = action.getViewportRegion();
      m_mouseNormPosition = event.getNormalizedPosition(vpRegion);
      beginAction(action, event.wasCtrlDown(), event.wasShiftDown());
    }

    if (!propageEvents())
      action.setHandled();
  }

  @Override
  public void beginAction(SoKeyboardEvent event, SoHandleEventAction action)
  {
    SoKeyboardEvent.Keys key = event.getKey();
    SoKeyboardEvent.States state = event.getState();

    switch ( key )
    {
    case S :
      if ( m_isSeekEnabled && state == SoKeyboardEvent.States.DOWN )
      {
        setSeekMode(!m_isSeekMode);
        action.setHandled();
      }
      break;
    case LEFT_CONTROL :
      beginAction(action, state == SoKeyboardEvent.States.DOWN, event.wasShiftDown());
      break;
    case LEFT_SHIFT :
      beginAction(action, event.wasCtrlDown(), state == SoKeyboardEvent.States.DOWN);
      break;
    default:
      break;
    }

    if (!propageEvents())
      action.setHandled();
  }

  protected abstract void beginAction(SoHandleEventAction action, boolean ctrlDown, boolean shiftDown);

  @Override
  public void doAction(SoMouseWheelEvent event, SoHandleEventAction action)
  {
    // Zoom
    int wheelDelta = event.getDelta() / m_mouseWheelDelta;
    float scaleFactor = (float) Math.pow(2., wheelDelta * Math.PI / 180.);

    doDollyWithCenter(m_mouseNormPosition, scaleFactor, action);

    if (!propageEvents())
      action.setHandled();
  }

  @Override
  public void doAction(SoScaleGestureEvent event, SoHandleEventAction action)
  {
    float delta = event.getDeltaScaleFactor();
    SbViewportRegion region = action.getViewportRegion();
    SbVec2f normPosition = region.normalize(event.getPosition());

    doDollyWithCenter(normPosition, 1.0f / delta, action);

    if (!propageEvents())
      action.setHandled();
  }

  @Override
  public void doAction(SoRotateGestureEvent event, SoHandleEventAction action)
  {
    doRotate(event, action);

    if (!propageEvents())
      action.setHandled();
  }

  @Override
  public void doAction(SoDoubleTapGestureEvent event, SoHandleEventAction action)
  {
    if ( m_isSeekEnabled )
    {
      setSeekMode(true);
      doSeek(event, action);
    }

    if (!propageEvents())
      action.setHandled();
  }

  @Override
  public void doAction(SoLongTapGestureEvent event, SoHandleEventAction action)
  {
    if (!propageEvents())
      action.setHandled();
  }

  protected void beginPan(SoHandleEventAction action)
  {
    if ( m_isPanEnabled )
    {
      m_cameraInteractor.activatePanning(m_mouseNormPosition, action.getViewportRegion());
      action.setHandled();
    }
  }

  protected void beginOrbit(SoHandleEventAction action)
  {
    if ( m_isOrbitEnabled )
    {
      m_cameraInteractor.activateOrbiting(m_mouseNormPosition);
      m_cameraInteractor.setRotationCenter(m_cameraInteractor.getFocalPoint());
      action.setHandled();
    }
  }

  protected void doDolly(SoEvent event, SoHandleEventAction action)
  {
    if ( m_isZoomEnabled )
    {
      SbViewportRegion vpRegion = action.getViewportRegion();
      SbVec2f newLocator = event.getNormalizedPosition(vpRegion);
      float dollyScaleFactor = (float) Math.pow(2, 10 * (newLocator.getY() - m_mouseNormPosition.getY()));
      m_cameraInteractor.dolly(dollyScaleFactor);

      if ( m_isAutoClippingPlanes )
        m_cameraInteractor.adjustClippingPlanes(action.getNodeAppliedTo(), vpRegion);
      m_mouseNormPosition = newLocator;
      action.setHandled();
    }
  }

  protected void doDollyWithCenter(SbVec2f center, float scaleFactor, SoHandleEventAction action)
  {
    if ( m_isZoomEnabled )
    {
      SbViewportRegion vpRegion = action.getViewportRegion();
      m_cameraInteractor.dollyWithZoomCenter(center, scaleFactor, vpRegion);

      if ( m_isAutoClippingPlanes )
        m_cameraInteractor.adjustClippingPlanes(action.getNodeAppliedTo(), vpRegion);
      action.setHandled();
    }
  }

  protected void doOrbit(SoEvent event, SoHandleEventAction action)
  {
    if ( m_isOrbitEnabled )
    {
      SbViewportRegion vpRegion = action.getViewportRegion();
      m_mouseNormPosition = event.getNormalizedPosition(vpRegion);
      m_cameraInteractor.orbit(m_mouseNormPosition);

      if ( m_isAutoClippingPlanes )
        m_cameraInteractor.adjustClippingPlanes(action.getNodeAppliedTo(), vpRegion);
      action.setHandled();
    }
  }

  protected void doPan(SoEvent event, SoHandleEventAction action)
  {
    if ( m_isPanEnabled )
    {
      SbViewportRegion vpRegion = action.getViewportRegion();
      m_mouseNormPosition = event.getNormalizedPosition(vpRegion);
      m_cameraInteractor.pan(m_mouseNormPosition, vpRegion);
      action.setHandled();
    }
  }

  protected void doTranslate(SoTouchEvent event, SoHandleEventAction action)
  {
    if ( m_isPanEnabled )
    {
      // NOTE: We are normalizing a distance, not a position. Cannot use vpRegion.normalize!
      SbViewportRegion vpRegion = action.getViewportRegion();
      SbVec2i32 vpSize = vpRegion.getViewportSizePixelsi32();
      SbVec2f displacement = event.getDisplacement().over(2);
      m_cameraInteractor.translate(new SbVec2f(displacement.getX()/vpSize.getX(), displacement.getY()/vpSize.getY()), vpRegion);
      action.setHandled();
    }
  }

  protected void doRotate(SoRotateGestureEvent event, SoHandleEventAction action)
  {
    if ( m_isRotateEnabled )
    {
      SbViewportRegion vpRegion = action.getViewportRegion();
      SbVec2f eventNormPosition = event.getNormalizedPosition(vpRegion);
      float distFromEye = m_cameraInteractor.getCamera().getViewVolume().getNearDist();
      SbVec3f rotCenter = m_cameraInteractor.projectToPlane(eventNormPosition, distFromEye, vpRegion);

      m_cameraInteractor.setRotationAxis(new SbVec3f(0, 0, 1.0f));
      m_cameraInteractor.setRotationCenter(rotCenter);
      m_cameraInteractor.rotate(-event.getDeltaRotation() * 0.5f);
      action.setHandled();
    }
  }

  protected void doRoll(SoEvent event, SoHandleEventAction action)
  {
    if ( m_isRotateEnabled )
    {
      SbViewportRegion vpRegion = action.getViewportRegion();

      SbVec2f newLocator = event.getNormalizedPosition(vpRegion);
      SbVec2f center = new SbVec2f(0.5f, 0.5f);
      SbVec2f oldPosCenterDist = m_mouseNormPosition.minus(center);
      m_mouseNormPosition = newLocator;

      // compute roll angle
      SbVec2f newPosCenterDist = m_mouseNormPosition.minus(center);
      float rollAngle = newPosCenterDist.getX() == 0 && newPosCenterDist.getY() == 0 ? 0
          : (float) Math.atan2(newPosCenterDist.getY(), newPosCenterDist.getX());
      rollAngle -= oldPosCenterDist.getX() == 0 && oldPosCenterDist.getY() == 0 ? 0
          : (float) Math.atan2(oldPosCenterDist.getY(), oldPosCenterDist.getX());
      m_cameraInteractor.roll(rollAngle);

      if ( m_isAutoClippingPlanes )
        m_cameraInteractor.adjustClippingPlanes(action.getNodeAppliedTo(), vpRegion);
      action.setHandled();
    }
  }

}

class OrbitInteraction extends NavigationInteraction
{
  private boolean m_isTouchOrbitActivated;

  public OrbitInteraction(SoCameraInteractor cameraInteractor)
  {
    super(cameraInteractor);

    m_isTouchOrbitActivated = false;
  }

  public OrbitInteraction(NavigationInteraction copyFrom)
  {
    super(copyFrom);

    m_isTouchOrbitActivated = false;
  }

  @Override
  public boolean propageEvents()
  {
    return false;
  }

  @Override
  protected void beginAction(SoHandleEventAction action, boolean ctrlDown, boolean shiftDown)
  {
    if ( m_isButton1Down && ctrlDown )
    {
      // BUTTON 1 + CTRL = pan
      beginPan(action);
    }
    else if ( m_isButton2Down )
    {
      // BUTTON 2 without modifier = pan
      beginPan(action);
    }
    else if ( m_isButton1Down )
    {
      // BUTTON 1 without modifier = orbit
      beginOrbit(action);
    }

    if (!propageEvents())
      action.setHandled();
  }

  @Override
  public void doAction(SoLocation2Event event, SoHandleEventAction action)
  {
    boolean ctrlDown = event.wasCtrlDown();
    boolean shiftDown = event.wasShiftDown();

    if ( m_isButton1Down && m_isButton2Down )
    {
      // BUTTON 1 + BUTTON 2 = dolly
      doDolly(event, action);
    }
    else if ( m_isButton1Down && ctrlDown )
    {
      // BUTTON 1 + CTRL = pan
      doPan(event, action);
    }
    else if ( m_isButton2Down && ctrlDown )
    {
      // BUTTON 2 + CTRL = dolly
      doDolly(event, action);
    }
    else if ( m_isButton1Down && shiftDown )
    {
      // BUTTON 1 + SHIFT = dolly
      doDolly(event, action);
    }
    else if ( m_isButton1Down )
    {
      // BUTTON 1 without modifier = orbit
      doOrbit(event, action);
    }
    else if ( m_isButton2Down )
    {
      // BUTTON 2 without modifier = pan
      doPan(event, action);
    }

    if ( !action.isHandled() )
    {
      // update mouse cursor's position
      m_mouseNormPosition = event.getNormalizedPosition(action.getViewportRegion());

      if (!propageEvents())
        action.setHandled();
    }
  }

  @Override
  public void doAction(SoTouchEvent event, SoHandleEventAction action)
  {
    SoTouchEvent.States state = event.getState();

    int numFinger = event.getTouchManager().getFingerNumber();

    if ( numFinger == 1 && state == SoTouchEvent.States.DOWN )
    {
      // one finger down
      SbViewportRegion vpRegion = action.getViewportRegion();
      m_mouseNormPosition = event.getNormalizedPosition(vpRegion);
      beginOrbit(action);

      if ( action.isHandled() )
        m_isTouchOrbitActivated = true;
    }
    else if ( numFinger == 1 && state == SoTouchEvent.States.MOVE && m_isTouchOrbitActivated )
    {
      // one finger moved
      doOrbit(event, action);
    }
    else if ( numFinger == 2 && (state == SoTouchEvent.States.DOWN || state == SoTouchEvent.States.MOVE) )
    {
      // 2 fingers down or moved
      doTranslate(event, action);
    }
    else if ( numFinger == 2 && state == SoTouchEvent.States.UP )
    {
      // one finger is on the screen but one has been lifted,
      // orbiting is temporarily disabled until the next touch down event.
      m_isTouchOrbitActivated = false;
    }

    if (!propageEvents())
      action.setHandled();
  }
}

class PlaneInteraction extends NavigationInteraction
{

  public PlaneInteraction(SoCameraInteractor cameraInteractor)
  {
    super(cameraInteractor);
  }

  public PlaneInteraction(NavigationInteraction copyFrom)
  {
    super(copyFrom);
  }

  @Override
  public boolean propageEvents()
  {
    return false;
  }

  @Override
  protected void beginAction(SoHandleEventAction action, boolean ctrlDown, boolean shiftDown)
  {
    if ( m_isButton1Down && ctrlDown )
    {
      // BUTTON 1 + CTRL = pan
      beginPan(action);
    }
    else if ( m_isButton2Down )
    {
      // BUTTON 2 without modifier = pan
      beginPan(action);
    }

    if (!propageEvents())
      action.setHandled();
  }

  @Override
  public void doAction(SoLocation2Event event, SoHandleEventAction action)
  {
    boolean ctrlDown = event.wasCtrlDown();
    boolean shiftDown = event.wasShiftDown();

    if ( m_isButton1Down && m_isButton2Down )
    {
      // BUTTON 1 + BUTTON 2 = roll
      doRoll(event, action);
    }
    else if ( m_isButton1Down && ctrlDown )
    {
      // BUTTON 1 + CTRL = pan
      doPan(event, action);
    }
    else if ( m_isButton2Down && ctrlDown )
    {
      // BUTTON 2 + CTRL = roll
      doRoll(event, action);
    }
    else if ( m_isButton1Down && shiftDown )
    {
      // BUTTON 1 + SHIFT = roll
      doRoll(event, action);
    }
    else if ( m_isButton1Down )
    {
      // BUTTON 1 without modifier = dolly
      doDolly(event, action);
    }
    else if ( m_isButton2Down )
    {
      // BUTTON 2 without modifier = pan
      doPan(event, action);
    }

    if ( !action.isHandled() )
    {
      // update mouse cursor's position
      m_mouseNormPosition = event.getNormalizedPosition(action.getViewportRegion());

      if (!propageEvents())
        action.setHandled();
    }
  }

  @Override
  public void doAction(SoTouchEvent event, SoHandleEventAction action)
  {
    SoTouchEvent.States state = event.getState();

    int numFinger = event.getTouchManager().getFingerNumber();

    if ( numFinger == 1 && (state == SoTouchEvent.States.DOWN || state == SoTouchEvent.States.MOVE) )
    {
      // 1 finger down or moved
      doTranslate(event, action);
    }
    else if ( numFinger == 2 && (state == SoTouchEvent.States.DOWN || state == SoTouchEvent.States.MOVE) )
    {
      // 2 fingers down or moved
      doTranslate(event, action);
    }

    if (!propageEvents())
      action.setHandled();
  }
}

class SelectionInteraction extends BaseInteraction
{

  public SelectionInteraction()
  {}

  @Override
  public void doAction(SoTouchEvent event, SoHandleEventAction action)
  {
    SoEvent eventOut = convertTouchEvent(event);
    action.setEvent(eventOut);

    if (!propageEvents())
      action.setHandled();
  }

  @Override
  public boolean propageEvents()
  {
    return true;
  }

  private SoEvent convertTouchEvent(SoTouchEvent event)
  {
    SoMouseButtonEvent mbe = new SoMouseButtonEvent();
    SoLocation2Event lct = new SoLocation2Event();
    SoTouchEvent.States state = event.getState();

    if ( state == SoTouchEvent.States.DOWN )
    {
      mbe.setTime(event.getTime());
      mbe.setPosition(event.getPosition());
      mbe.setButton(SoMouseButtonEvent.Buttons.BUTTON1);
      mbe.setState(SoMouseButtonEvent.States.DOWN);
      return mbe;
    }

    if ( state == SoTouchEvent.States.MOVE )
    {
      lct.setTime(event.getTime());
      lct.setPosition(event.getPosition());
      lct.setEventSource(SoLocation2Event.EventSources.MOUSE_MOVE);
      return lct;
    }

    if ( state == SoTouchEvent.States.UP )
    {
      mbe.setTime(event.getTime());
      mbe.setPosition(event.getPosition());
      mbe.setButton(SoMouseButtonEvent.Buttons.BUTTON1);
      mbe.setState(SoMouseButtonEvent.States.UP);
      return mbe;
    }

    return null;
  }

  @Override
  public void beginAction(SoMouseButtonEvent event, SoHandleEventAction action)
  {
    if (!propageEvents())
      action.setHandled();
  }

  @Override
  public void beginAction(SoKeyboardEvent event, SoHandleEventAction action)
  {
    if (!propageEvents())
      action.setHandled();
  }


  @Override
  public void doAction(SoLocation2Event event, SoHandleEventAction action)
  {
    if (!propageEvents())
      action.setHandled();
  }


  @Override
  public void doAction(SoMouseWheelEvent event, SoHandleEventAction action)
  {
    if (!propageEvents())
      action.setHandled();
  }


  @Override
  public void doAction(SoScaleGestureEvent event, SoHandleEventAction action)
  {
    if (!propageEvents())
      action.setHandled();
  }


  @Override
  public void doAction(SoRotateGestureEvent event, SoHandleEventAction action)
  {
    if (!propageEvents())
      action.setHandled();
  }


  @Override
  public void doAction(SoDoubleTapGestureEvent event, SoHandleEventAction action)
  {
    if (!propageEvents())
      action.setHandled();
  }


  @Override
  public void doAction(SoLongTapGestureEvent event, SoHandleEventAction action)
  {
    if (!propageEvents())
      action.setHandled();
  }


}
