package com.openinventor.inventor.viewercomponents.nodes;

import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.actions.SoHandleEventAction;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.events.SoMouseWheelEvent;
import com.openinventor.inventor.gestures.events.SoDoubleTapGestureEvent;
import com.openinventor.inventor.gestures.events.SoLongTapGestureEvent;
import com.openinventor.inventor.gestures.events.SoRotateGestureEvent;
import com.openinventor.inventor.gestures.events.SoScaleGestureEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.touch.events.SoTouchEvent;
import com.openinventor.inventor.viewercomponents.SoCameraInteractor;

/**
 * Tool class for easily building a basic OpenInventor application without using
 * existing viewer classes.
 * <p>
 * The SceneInteractor is a simple extension of the {@link SoSeparator} node
 * that allows handling of Open Inventor events. This class should be overridden
 * as it provides only empty event handlers. <br>
 * <p>
 * This node is intended to be used as the root of a scene graph. The
 * SceneInteractor is a custom SoSeparator whose children are:
 * <ul>
 * <li>an SoEventCallback node that handles keyboard, mouse and touch events
 * <li>a camera (switch between perspective and orthographic camera)
 * <li>a headlight
 * <li>the application's scene graph, which should be the last child.
 * </ul>
 *
 * The SceneInteractor uses an instance of {@link SoCameraInteractor} in order
 * to manipulate the camera in response to OpenInventor events.
 * <p>
 * Class diagram of the SceneInteractor showing the relationship between the
 * SoEventCallback, the SoCamera, the SoDirectionalLight (used as headlight) and
 * the SoCameraInteractor.
 * <p>
 * <img src={@docRoot}/docFiles/SceneInteractor.png alt="Class diagram">
 * <p>
 * Detail of the scene graph rooted by a SceneInteractor:
 * <p>
 * <img src={@docRoot}/docFiles/SceneInteractorSceneGraph.png alt="Detail of the
 * scene graph">
 * <p>
 * <b>Notes:</b>
 * <ul>
 * <li>A basic version of SceneInteractor is a supported part of the
 * OpenInventor API, the prebuilt jar is located in the usual folder
 * $OIVJHOME/jre/lib/ext.
 * <li>The basic version of SceneInteractor is also provided as source code in
 * the sources folder to allow applications to customize and build their own
 * interactive tool class. See
 * $OIVJHOME/sources/com/openinventor/inventor/viewercomponents.
 * </ul>
 */
public class SceneInteractor extends SoSeparator
{

  public enum CameraMode
  {
    PERSPECTIVE, ORTHOGRAPHIC
  }

  protected SoCameraInteractor m_cameraInteractor;
  private SoCameraInteractor m_perspInteractor;
  private SoCameraInteractor m_orthoInteractor;

  private KeyboardEventCallbackCB m_keyboardCB;
  private Location2EventCallbackCB m_locationCB;
  private MouseButtonEventCallbackCB m_mouseButtonCB;
  private MouseWheelEventCallbackCB m_mouseWheelCB;
  private TouchEventCallbackCB m_touchCB;
  private ScaleGestureEventCallbackCB m_scaleGestureCB;
  private RotateGestureEventCallbackCB m_rotateGestureCB;
  private DoubleTapGestureEventCallbackCB m_doubleTapGestureCB;
  private LongTapGestureEventCallbackCB m_longTapGestureCB;

  protected SoEventCallback m_eventCallback;
  private SoSwitch m_cameraSwitch;
  private SoSwitch m_headlightSwitch;
  private SoRotation m_headlightRot;

  public SceneInteractor()
  {
    // Stereo camera
    SoStereoCamera stereoCamera = new SoStereoCamera();
    m_perspInteractor = SoCameraInteractor.getNewInstance(stereoCamera);

    // Orthographic camera
    SoOrthographicCamera orthoCamera = new SoOrthographicCamera();
    m_orthoInteractor = SoCameraInteractor.getNewInstance(orthoCamera);

    m_cameraSwitch = new SoSwitch();
    {
      m_cameraSwitch.addChild(stereoCamera);
      m_cameraSwitch.addChild(orthoCamera);
    }

    m_headlightRot = new SoRotation();
    SoTransformSeparator transformSep = new SoTransformSeparator();
    {
      transformSep.addChild(m_headlightRot);
      transformSep.addChild(new SoDirectionalLight());
    }
    m_headlightSwitch = new SoSwitch();
    {
      m_headlightSwitch.addChild(transformSep);
    }
    // enable headlight by default
    m_headlightSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

    m_cameraInteractor = null;
    // perspective camera by default
    setCameraMode(CameraMode.PERSPECTIVE);

    // events listeners
    m_eventCallback = new SoEventCallback();

    m_keyboardCB = new KeyboardEventCallbackCB();
    m_locationCB = new Location2EventCallbackCB();
    m_mouseButtonCB = new MouseButtonEventCallbackCB();
    m_mouseWheelCB = new MouseWheelEventCallbackCB();
    m_touchCB = new TouchEventCallbackCB();
    m_scaleGestureCB = new ScaleGestureEventCallbackCB();
    m_rotateGestureCB = new RotateGestureEventCallbackCB();
    m_doubleTapGestureCB = new DoubleTapGestureEventCallbackCB();
    m_longTapGestureCB = new LongTapGestureEventCallbackCB();

    m_eventCallback.addEventCallback(SoKeyboardEvent.class, m_keyboardCB);
    m_eventCallback.addEventCallback(SoLocation2Event.class, m_locationCB);
    m_eventCallback.addEventCallback(SoMouseButtonEvent.class, m_mouseButtonCB);
    m_eventCallback.addEventCallback(SoMouseWheelEvent.class, m_mouseWheelCB);
    m_eventCallback.addEventCallback(SoTouchEvent.class, m_touchCB);
    m_eventCallback.addEventCallback(SoScaleGestureEvent.class, m_scaleGestureCB);
    m_eventCallback.addEventCallback(SoRotateGestureEvent.class, m_rotateGestureCB);
    m_eventCallback.addEventCallback(SoDoubleTapGestureEvent.class, m_doubleTapGestureCB);
    m_eventCallback.addEventCallback(SoLongTapGestureEvent.class, m_longTapGestureCB);
    {
      addChild(m_eventCallback);
      addChild(m_cameraSwitch);
      addChild(m_headlightSwitch);
    }
  }

  @Override
  public boolean dispose()
  {
    m_keyboardCB.dispose();
    m_locationCB.dispose();
    m_mouseButtonCB.dispose();
    m_mouseWheelCB.dispose();
    m_touchCB.dispose();
    m_scaleGestureCB.dispose();
    m_rotateGestureCB.dispose();
    m_doubleTapGestureCB.dispose();
    m_longTapGestureCB.dispose();

    return super.dispose();
  }

  /**
   * Set camera mode to perspective or orthographic.
   */
  public void setCameraMode(CameraMode mode)
  {
    if ( mode == CameraMode.PERSPECTIVE )
    {
      if ( m_cameraInteractor != null )
        // synchronize old and new cameras
        m_perspInteractor.synchronize(m_cameraInteractor.getCamera());
      m_cameraInteractor = m_perspInteractor;
      m_cameraSwitch.whichChild.setValue(0);
    }
    else if ( mode == CameraMode.ORTHOGRAPHIC )
    {
      if ( m_cameraInteractor != null )
        // synchronize old and new cameras
        m_orthoInteractor.synchronize(m_cameraInteractor.getCamera());
      m_cameraInteractor = m_orthoInteractor;
      m_cameraSwitch.whichChild.setValue(1);
    }
    m_headlightRot.rotation.connectFrom(m_cameraInteractor.getCamera().orientation);
  }

  /**
   * Get the current camera mode.
   */
  public CameraMode getCameraMode()
  {
    int activatedCamera = m_cameraSwitch.whichChild.getValue();

    if ( activatedCamera == 0 )
      return CameraMode.PERSPECTIVE;
    else
      return CameraMode.ORTHOGRAPHIC;
  }

  /**
   * @return the current camera interactor
   */
  public SoCameraInteractor getCameraInteractor()
  {
    return m_cameraInteractor;
  }

  /**
   * Returns the current camera.
   *
   * @return the current camera
   */
  public SoCamera getCamera()
  {
    return m_cameraInteractor.getCamera();
  }

  /**
   * Adjust near and far clipping planes to minimize clipping of objects in the
   * scene. This adjustment, based on the bounding box of the scene, ensures
   * that shapes will not be clipped and also that depth buffer precision is
   * maximized. This method should be called before each render traversal.
   */
  public void adjustClippingPlanes(SbViewportRegion vpRegion)
  {
    m_cameraInteractor.adjustClippingPlanes(this, vpRegion);
  }

  /**
   * Set the camera to view all the scene.
   */
  public void viewAll(SbViewportRegion vpRegion)
  {
    m_cameraInteractor.viewAll(this, vpRegion);
  }

  /**
   * Moves the camera to be aligned with the given direction vector while
   * keeping the "up" direction of the camera parallel to the specified up
   * vector.
   *
   * @param direction
   *          the new view direction vector
   * @param up
   *          the new "up" direction
   */
  public void viewAxis(SbVec3f direction, SbVec3f up)
  {
    m_cameraInteractor.viewAxis(direction, up);
  }

  /**
   * Enable or disable headlight.
   */
  public void enableHeadlight(boolean enabled)
  {
    if ( enabled )
      m_headlightSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    else
      m_headlightSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
  }

  /**
   * Return if headlight is enabled.
   */
  public boolean isHeadlightEnabled()
  {
    return m_headlightSwitch.whichChild.getValue() == SoSwitch.SO_SWITCH_ALL;
  }

  /**
   * Invoked when a key has been pressed.
   */
  protected void keyPressed(SoKeyboardEvent event, SoHandleEventAction action)
  {}

  /**
   * Invoked when a key has been released.
   */
  protected void keyReleased(SoKeyboardEvent evt, SoHandleEventAction action)
  {}

  /**
   * Invoked when a mouse button has been pressed on this component.
   */
  protected void mousePressed(SoMouseButtonEvent event, SoHandleEventAction action)
  {}

  /**
   * Invoked when a mouse button has been released on this component.
   */
  protected void mouseReleased(SoMouseButtonEvent event, SoHandleEventAction action)
  {}

  /**
   * Invoked when the mouse wheel is rotated.
   */
  protected void mouseWheelMoved(SoMouseWheelEvent event, SoHandleEventAction action)
  {}

  /**
   * Invoked when the mouse cursor has been moved onto this component.
   */
  protected void mouseMoved(SoLocation2Event event, SoHandleEventAction action)
  {}

  /**
   * Invoked when the finger is on/off/moving on this component.
   */
  protected void touch(SoTouchEvent event, SoHandleEventAction action)
  {}

  /**
   * Invoked when a pinch gesture is detected on this component.
   */
  protected void zoom(SoScaleGestureEvent event, SoHandleEventAction action)
  {}

  /**
   * Invoked when a two finger rotate gesture is detected on this component.
   */
  protected void rotate(SoRotateGestureEvent event, SoHandleEventAction action)
  {}

  /**
   * Invoked when a double-tap gesture is detected on this component.
   */
  protected void doubleTap(SoDoubleTapGestureEvent event, SoHandleEventAction action)
  {}

  /**
   * Invoked when a long-tap gesture is detected on this component.
   */
  protected void longTap(SoLongTapGestureEvent event, SoHandleEventAction action)
  {}
  // EVENTS CALLBACKS

  private class KeyboardEventCallbackCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();

      if ( evt != null )
      {
        if ( SoKeyboardEvent.isKeyPressEvent(evt, SoKeyboardEvent.Keys.ANY) )
          keyPressed((SoKeyboardEvent) evt, node.getAction());
        else if ( SoKeyboardEvent.isKeyReleaseEvent(evt, SoKeyboardEvent.Keys.ANY) )
          keyReleased((SoKeyboardEvent) evt, node.getAction());
      }
    }
  }

  private class MouseButtonEventCallbackCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();

      if ( evt != null )
      {
        if ( SoMouseButtonEvent.isButtonPressEvent(evt, SoMouseButtonEvent.Buttons.ANY) )
          mousePressed((SoMouseButtonEvent) evt, node.getAction());
        else if ( SoMouseButtonEvent.isButtonReleaseEvent(evt, SoMouseButtonEvent.Buttons.ANY) )
          mouseReleased((SoMouseButtonEvent) evt, node.getAction());
      }
    }
  }

  private class MouseWheelEventCallbackCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();

      if ( evt != null )
        mouseWheelMoved((SoMouseWheelEvent) evt, node.getAction());
    }
  }

  private class Location2EventCallbackCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();

      if ( evt != null )
        mouseMoved((SoLocation2Event) evt, node.getAction());
    }
  }

  private class TouchEventCallbackCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();

      if ( evt != null )
        touch((SoTouchEvent) evt, node.getAction());
    }
  }

  private class ScaleGestureEventCallbackCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();

      if ( evt != null )
        zoom((SoScaleGestureEvent) evt, node.getAction());
    }
  }

  private class RotateGestureEventCallbackCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();

      if ( evt != null )
        rotate((SoRotateGestureEvent) evt, node.getAction());
    }
  }

  private class DoubleTapGestureEventCallbackCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();

      if ( evt != null )
        doubleTap((SoDoubleTapGestureEvent) evt, node.getAction());
    }
  }

  private class LongTapGestureEventCallbackCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();

      if ( evt != null )
        longTap((SoLongTapGestureEvent) evt, node.getAction());
    }
  }

}
