package com.openinventor.inventor.viewercomponents.swt.glcanvas.renderareas;

import java.util.List;

import org.eclipse.swt.widgets.Composite;

import com.jogamp.nativewindow.CapabilitiesImmutable;
import com.jogamp.nativewindow.NativeWindowException;
import com.jogamp.opengl.DefaultGLCapabilitiesChooser;
import com.jogamp.opengl.GLAutoDrawable;
import com.jogamp.opengl.GLCapabilities;
import com.jogamp.opengl.GLCapabilitiesChooser;
import com.jogamp.opengl.GLCapabilitiesImmutable;
import com.jogamp.opengl.GLEventListener;
import com.jogamp.opengl.swt.GLCanvas;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.devices.SoGLContext;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.viewercomponents.SoRawStereoParameters;
import com.openinventor.inventor.viewercomponents.SoRenderAreaCore;
import com.openinventor.inventor.viewercomponents.swt.Timer;

public class RenderArea extends GLCanvas implements GLEventListener
{

  protected SoRenderAreaCore m_renderAreaCore;
  protected SoNode m_sceneGraph;

  public RenderArea(Composite parent, int style)
  {
    this(parent, style, null, new OptionalStereoGLCapabilitiesChooser());
  }

  public RenderArea(Composite parent, int style, GLCapabilities caps)
  {
    this(parent, style, caps, new OptionalStereoGLCapabilitiesChooser());
  }

  private RenderArea(final Composite parent, final int style, final GLCapabilitiesImmutable capsReqUser,
      final GLCapabilitiesChooser capsChooser)
  {
    super(parent, style, capsReqUser, capsChooser);

    m_renderAreaCore = null;
    m_sceneGraph = null;

    // init the system timer with an SWT timer
    SoDB.setSystemTimer(new Timer());

    addGLEventListener(this);
  }

  public void activateStereo(boolean activated)
  {
    if ( activated )
    {
      // activate raw stereo
      if ( !isRawStereoAvailable() )
        throw new UnsupportedOperationException("Stereo buffers are not enabled");

      SoRawStereoParameters params = new SoRawStereoParameters();
      m_renderAreaCore.setStereoParameters(params);
      m_renderAreaCore.activateStereo(true);
    }
    else
      m_renderAreaCore.activateStereo(false);
  }

  public boolean isRawStereoAvailable()
  {
    return getChosenGLCapabilities().getStereo();
  }

  public void setSceneGraph(SoNode sceneGraph)
  {
    m_sceneGraph = sceneGraph;

    if ( m_renderAreaCore != null )
      m_renderAreaCore.setSceneGraph(sceneGraph);
  }

  protected void render()
  {
    m_renderAreaCore.render();
  }

  @Override
  public void init(GLAutoDrawable drawable)
  {
    // let OpenInventor be aware of the current context created by JOGL
    SoGLContext context = SoGLContext.getCurrent(true);
    m_renderAreaCore = new SoRenderAreaCore(context);
    m_renderAreaCore.setSceneGraph(m_sceneGraph);
  }

  @Override
  public void dispose(GLAutoDrawable drawable)
  {
    m_renderAreaCore.dispose();
  }

  @Override
  public void display(GLAutoDrawable drawable)
  {
    // render Open Inventor scene graph
    render();
  }

  @Override
  public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height)
  {
    m_renderAreaCore.setSize(new SbVec2i32(width, height));
  }
}

/**
 * Custom implementation of GLCapabilitiesChooser.<br>
 * The DefaultGLCapabilitiesChooser does not allow to define an optional but not
 * mandatory GL capability. In particular, stereo capability is considered as a
 * mandatory capability.<br>
 * This implementation allows selecting an optional stereo capability: if no
 * stereo capability is available, it continues to search other requested
 * (mandatory) capabilities.
 */
class OptionalStereoGLCapabilitiesChooser extends DefaultGLCapabilitiesChooser
{
  @Override
  public int chooseCapabilities(CapabilitiesImmutable desired, List<? extends CapabilitiesImmutable> available,
      int windowSystemRecommendedChoice)
  {
    int chosen = -1;
    try
    {
      // launch default chooser
      chosen = super.chooseCapabilities(desired, available, windowSystemRecommendedChoice);
    }
    catch (NativeWindowException exc)
    {
      // chosen index is not valid, check if stereo capability has been
      // requested
      final GLCapabilities gldes = (GLCapabilities) desired.cloneMutable();
      if ( gldes.getStereo() )
      {
        // stereo capability has been requested but it is not available.
        System.err.println("Could not enable stereo buffers");

        // try to get the best config with no stereo
        // if we don't do that, the first available config will be chosen
        // without paying attention to others requested GL capabilities.
        gldes.setStereo(false);
        chosen = super.chooseCapabilities(gldes, available, windowSystemRecommendedChoice);
      }
      else
        // let JOGL do its job...
        throw exc;
    }

    return chosen;
  }
}
