package com.openinventor.inventor.viewercomponents.swt.glcanvas.renderareas;

import com.jogamp.opengl.GLAutoDrawable;
import com.jogamp.opengl.GLCapabilities;

import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackListener;
import org.eclipse.swt.events.MouseWheelListener;
import org.eclipse.swt.widgets.Composite;

import com.openinventor.inventor.SbEventListener;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.events.SoMouseWheelEvent;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoGroup;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoStereoCamera;
import com.openinventor.inventor.viewercomponents.SiRenderArea;
import com.openinventor.inventor.viewercomponents.SiRenderArea.RenderEventArg;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;

public class RenderAreaInteractive extends RenderArea
    implements MouseListener, MouseMoveListener, MouseWheelListener, MouseTrackListener, KeyListener
{

  /**
   * Camera clipping planes adjustment mode.<br>
   * When adjustment mode is set to {@code AUTO}, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.<br>
   * When adjustment mode is set to {@code MANUAL}, the user is expected to
   * manually set the camera near and far planes.
   *
   */
  public enum ClippingPlanesAdjustMode
  {
    AUTO, MANUAL
  }

  protected SceneInteractor m_rootSceneGraph;
  private SoGroup m_appSceneGraph;

  private ClippingPlanesAdjustMode m_clippingMode;

  public RenderAreaInteractive(Composite parent, int style)
  {
    super(parent, style);

    init(true);
  }

  public RenderAreaInteractive(Composite parent, int style, GLCapabilities caps)
  {
    super(parent, style, caps);

    init(true);
  }

  protected RenderAreaInteractive(Composite parent, int style, boolean buildRootSceneGraph)
  {
    super(parent, style);

    init(buildRootSceneGraph);
  }

  protected RenderAreaInteractive(Composite parent, int style, boolean buildRootSceneGraph, GLCapabilities caps)
  {
    super(parent, style, caps);

    init(buildRootSceneGraph);
  }

  private void init(boolean buildRootSceneGraph)
  {
    m_clippingMode = ClippingPlanesAdjustMode.AUTO;

    if ( buildRootSceneGraph )
    {
      m_rootSceneGraph = new SceneInteractor();
      buildSceneGraph();
    }

    addMouseListener(this);
    addMouseMoveListener(this);
    addMouseWheelListener(this);
    addKeyListener(this);
  }

  protected void buildSceneGraph()
  {
    m_appSceneGraph = new SoGroup();
    m_rootSceneGraph.addChild(m_appSceneGraph);
    super.setSceneGraph(m_rootSceneGraph);
  }

  @Override
  public void setSceneGraph(SoNode sceneGraph)
  {
    m_appSceneGraph.removeAllChildren();
    if ( sceneGraph != null )
      m_appSceneGraph.addChild(sceneGraph);
  }

  @Override
  protected void render()
  {
    if ( m_clippingMode == ClippingPlanesAdjustMode.AUTO )
    {
      // Update camera clipping planes before each rendering.
      m_rootSceneGraph.adjustClippingPlanes(m_renderAreaCore.getSceneManager().getViewportRegion());
    }

    // render scene
    super.render();
  }

  @Override
  public void init(GLAutoDrawable drawable)
  {
    super.init(drawable);

    m_renderAreaCore.onStartRender().addEventListener(new SbEventListener<SiRenderArea.RenderEventArg>()
    {
      @Override
      public void onEvent(RenderEventArg arg0)
      {
        display();
      }
    });
  }

  /**
   * Get the camera clipping planes adjustment mode.
   */
  public ClippingPlanesAdjustMode getClippingPlanesAdjustMode()
  {
    return m_clippingMode;
  }

  /**
   * Set the camera clipping planes adjustment mode. <br>
   * When adjustment mode is set to {@code AUTO}, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.<br>
   * When adjustment mode is set to {@code MANUAL}, the user is expected to
   * manually set those planes. Updating clipping planes after a camera move is
   * not enough, if a dragger or a rendered shape is moved, they can disappear
   * or become partially clipped.<br>
   * Default is {@code AUTO}.
   */
  public void setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode)
  {
    if ( m_clippingMode != mode )
    {
      m_clippingMode = mode;
      render();
    }
  }

  public SceneInteractor getRootSceneGraph()
  {
    return m_rootSceneGraph;
  }

  public void viewAll(SbViewportRegion viewport)
  {
    m_rootSceneGraph.viewAll(viewport);
  }

  public void viewAxis(SbVec3f direction, SbVec3f up)
  {
    m_rootSceneGraph.viewAxis(direction, up);
  }

  public void saveCamera()
  {
    m_rootSceneGraph.getCameraInteractor().pushCamera();
  }

  public void restoreCamera()
  {
    m_rootSceneGraph.getCameraInteractor().popCamera();
  }

  public void setCameraType(CameraMode mode)
  {
    m_rootSceneGraph.setCameraMode(mode);
  }

  @Override
  public void activateStereo(boolean activated)
  {
    if ( activated )
    {
      // check camera
      if ( !isStereoSupported() )
        throw new UnsupportedOperationException("Cannot activate stereo: current camera is not a SoStereoCamera!");
    }
    super.activateStereo(activated);
  }

  public boolean isStereoSupported()
  {
    // check camera
    SoCamera camera = m_rootSceneGraph.getCamera();
    return camera instanceof SoStereoCamera;
  }

  public void setStereoCameraOffset(float offset)
  {
    SoCamera camera = m_rootSceneGraph.getCamera();
    try
    {
      SoStereoCamera stereoCamera = (SoStereoCamera) camera;
      stereoCamera.offset.setValue(offset);
    }
    catch (ClassCastException e)
    {
      throw new UnsupportedOperationException("Current camera is not a SoStereoCamera!");
    }
  }

  public void setStereoCameraBalance(float balance)
  {
    SoCamera camera = m_rootSceneGraph.getCamera();
    try
    {
      SoStereoCamera stereoCamera = (SoStereoCamera) camera;
      stereoCamera.balance.setValue(balance);
    }
    catch (ClassCastException e)
    {
      throw new UnsupportedOperationException("Current camera is not a SoStereoCamera!");
    }
  }

  @Override
  public void mouseDown(MouseEvent e)
  {
    if ( m_renderAreaCore != null )
    {
      SoMouseButtonEvent evt = SWTEventToSoEvent.getMousePressEvent(e, this);
      m_renderAreaCore.processEvent(evt);
    }
  }

  @Override
  public void mouseScrolled(MouseEvent e)
  {
    if ( m_renderAreaCore != null )
    {
      SoMouseWheelEvent evt = SWTEventToSoEvent.getMouseWheelEvent(e, this);
      m_renderAreaCore.processEvent(evt);
    }
  }

  @Override
  public void mouseMove(MouseEvent e)
  {
    if ( m_renderAreaCore != null )
    {
      SoLocation2Event evt = SWTEventToSoEvent.getMouseMoveEvent(e, this);
      m_renderAreaCore.processEvent(evt);
    }
  }

  @Override
  public void mouseDoubleClick(MouseEvent e)
  {
    if ( m_renderAreaCore != null )
    {
      SoMouseButtonEvent evt = SWTEventToSoEvent.getMouseDoubleClickEvent(e, this);
      m_renderAreaCore.processEvent(evt);
    }
  }

  @Override
  public void mouseUp(MouseEvent e)
  {
    if ( m_renderAreaCore != null )
    {
      SoMouseButtonEvent evt = SWTEventToSoEvent.getMouseReleaseEvent(e, this);
      m_renderAreaCore.processEvent(evt);
    }
  }

  @Override
  public void mouseEnter(MouseEvent e)
  {
    if ( m_renderAreaCore != null )
    {
      SoLocation2Event evt = SWTEventToSoEvent.getMouseEnterEvent(e, this);
      m_renderAreaCore.processEvent(evt);
    }
  }

  @Override
  public void mouseExit(MouseEvent e)
  {
    if ( m_renderAreaCore != null )
    {
      SoLocation2Event evt = SWTEventToSoEvent.getMouseLeaveEvent(e, this);
      m_renderAreaCore.processEvent(evt);
    }
  }

  @Override
  public void mouseHover(MouseEvent e)
  {}

  @Override
  public void keyPressed(KeyEvent e)
  {
    if ( m_renderAreaCore != null )
    {
      SoKeyboardEvent evt = SWTEventToSoEvent.getKeyPressEvent(e, this);
      m_renderAreaCore.processEvent(evt);
    }
  }

  @Override
  public void keyReleased(KeyEvent e)
  {
    if ( m_renderAreaCore != null )
    {
      SoKeyboardEvent evt = SWTEventToSoEvent.getKeyReleaseEvent(e, this);
      m_renderAreaCore.processEvent(evt);
    }
  }

}
