///////////////////////////////////////////////////////////////////////////////
//
// This code is part of the Open Inventor Medical Edition utility library.
//
// Open Inventor customers may use this source code to create or enhance Open
// Inventor-based applications.
//
// The medical utility classes are provided as a jar named com.openinventor.medical.jar,
// that can be used directly in an Open Inventor application. The classes in this jar
// are documented and supported by FEI. These classes are also provided as source code.
//
///////////////////////////////////////////////////////////////////////////////
package com.openinventor.medical.helpers;

import java.io.File;
import java.io.FileNotFoundException;

import com.openinventor.inventor.*;
import com.openinventor.inventor.actions.SoGetBoundingBoxAction;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.errors.SoError;
import com.openinventor.inventor.helpers.SbFileHelper;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.SoCameraInteractor;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoLDMResourceParameters;
import com.openinventor.ldm.readers.SoLDMReader;
import com.openinventor.ldm.readers.SoVolumeReader;
import com.openinventor.medical.nodes.DicomInfo;
import com.openinventor.medical.nodes.SliceOrientationMarkers;
import com.openinventor.medical.nodes.SliceScaleBar;
import com.openinventor.medical.nodes.TextBox;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.readers.SoVRDicomData;
import com.openinventor.volumeviz.readers.SoVRDicomFileReader;
import com.openinventor.volumeviz.readers.dicom.SiDicomDataSet;
import com.openinventor.volumeviz.readers.dicom.SiDicomElement;
import com.openinventor.volumeviz.readers.dicom.SiDicomValue;
import com.openinventor.volumeviz.readers.dicom.SoDicomTag;

//@formatter:off
/**
 * Utility class for medical applications using Open Inventor.
 *
 * Open Inventor includes a general purpose 3D graphics core library plus
 * optimized extensions for specific data types and markets, including medical
 * visualization. We recommend using these utility functions (and the Medical
 * utility nodes) to get the best performance and productivity from the Open
 * Inventor Medical Edition.
 * <p>
 * <b> Summary:</b>
 * <ul>
 * <li> {@link #dicomAdjustVolume dicomAdjustVolume()} <br>
 * Call this method to adjust VolumeViz
 * parameters for optimal performance with medical data and to position a DICOM
 * volume correctly in 3D space. </li>
 * <p>
 * <li> {@link #dicomAdjustDataRange dicomAdjustDataRange()} <br>
 * Call this method to set the VolumeViz
 * data range based on the Window Center and Window Width tags in a DICOM file.
 * If those tags are not present, automatically sets the data range to the
 * actual range of values in the data. </li>
 * <p>
 * <li> {@link #dicomSetWindowCenterWidth dicomSetWindowCenterWidth()} and
 * {@link #dicomGetWindowCenterWidth dicomGetWindowCenterWidth()} <br>
 * In Open Inventor the range of voxel values that map onto the color
 * map are specified as the 'min' and 'max' values of the range (see SoDataRange).
 * These methods provide a convenient way to set and query the data range using
 * the medical convention of 'window
 * center' and 'window width'. </li>
 * <p>
 * <li> {@link #dicomFindFilesbySeries dicomFindFilesbySeries()} <br>
 * Call this method to build a list of
 * DICOM image files in a directory that belong to the same series as the
 * specified file. This list can be passed to the DICOM volume reader using the
 * setFileList() method. </li>
 * <p>
 * <li> {@link #orientView orientView()} <br>
 * This method is useful for orienting the camera
 * to view a single image (a slice in VolumeViz jargon) along one of the
 * standard axes (Axial, Coronal or Sagittal). But remember that the volume is
 * always a 3D object to Open Inventor and you must also set the 'axis' field on
 * the SoOrthoSlice node. See the dicomImageViewer example. </li>
 * </ul>
 *
 * @see com.openinventor.medical.nodes.DicomInfo
 * @see com.openinventor.medical.nodes.Gnomon
 * @see com.openinventor.medical.nodes.Magnifier
 * @see com.openinventor.medical.nodes.OrthoSliceBorder
 * @see com.openinventor.medical.nodes.Ruler
 * @see com.openinventor.medical.nodes.SliceOrientationMarkers
 * @see com.openinventor.medical.nodes.SliceScaleBar
 * @see com.openinventor.medical.nodes.TextBox
 */
//@formatter:on
public class MedicalHelper
{
  public static final String EXAMPLE_LOGO = "$OIVJHOME/source/com/openinventor/medical/data/oiv_logo_white_test.png";
  public static final short WINDOW_HEIGHT = 633;
  public static final short WINDOW_WIDTH = 1024;

  /**
   * Medical axis names. These names are mapped to the corresponding XYZ enum
   * value in Open Inventor.
   */

  public enum Axis implements IntegerValuedEnum
  {
    /** Axial (Z axiz) */
    AXIAL(2),
    /** Transverse (synonym for Axial) */
    TRANSVERSE(2),
    /** Coronal (Y axis) */
    CORONAL(1),
    /** Sagittal (X axis) */
    SAGITTAL(0);

    private final int axis;

    @Override
    public int getValue()
    {
      return axis;
    }

    private Axis(int s)
    {
      axis = s;
    }
  };

  /**
   * Returns a collection of DicomInfo annotation nodes for Open Inventor
   * medical examples. This method is not important for customer applications,
   * but the technique can be useful for adding annotation text to an
   * application window.
   *
   * @param filename
   *          Path to a DICOM file from which info will be extracted
   * @return Root of the annotation scene graph.
   */
  public static SoNode exampleDicomAnnotation(String filename)
  {
    SoSeparator mainSep = new SoSeparator();
    if ( filename.length() <= 0 )
      return mainSep;

    // Top left DICOM annotation
    DicomInfo topLeftDicomAnnotation = new DicomInfo();
    topLeftDicomAnnotation.fileName.setValue(filename);
    topLeftDicomAnnotation.position.setValue(new SbVec3f(-0.99f, 0.99f, 0));
    // Add text to annotation
    topLeftDicomAnnotation.displayDicomInfo("Patient Name : ", (short) 0x10, (short) 0x0010);
    topLeftDicomAnnotation.displayDicomInfo("Patient Id : ", (short) 0x10, (short) 0x0020);
    topLeftDicomAnnotation.displayDicomInfo("Institution Name : ", (short) 0x08, (short) 0x0080);
    topLeftDicomAnnotation.displayDicomInfo("Modality : ", (short) 0x08, (short) 0x0060);
    topLeftDicomAnnotation.displayDicomInfo("Patient Pos : ", (short) 0x18, (short) 0x5100);
    topLeftDicomAnnotation.displayDicomInfo("Patient orientation : ", (short) 0x20, (short) 0x0020);
    topLeftDicomAnnotation.displayDicomInfo("samples per pixel : ", (short) 0x28, (short) 0x0002);
    topLeftDicomAnnotation.displayDicomInfo("Photometric interp : ", (short) 0x28, (short) 0x0004);
    topLeftDicomAnnotation.displayDicomInfo("Image Orientation : ", (short) 0x20, (short) 0x0037);
    topLeftDicomAnnotation.displayDicomInfo("Image rows (height) : ", (short) 0x28, (short) 0x0010);
    topLeftDicomAnnotation.displayDicomInfo("Image columns (width) : ", (short) 0x28, (short) 0x0011);

    mainSep.addChild(topLeftDicomAnnotation);

    // Bottom right DICOM annotation
    DicomInfo bottomRightDicomAnnotation = new DicomInfo();
    bottomRightDicomAnnotation.alignmentH.setValue(DicomInfo.AlignmentH.RIGHT);
    bottomRightDicomAnnotation.alignmentV.setValue(DicomInfo.AlignmentV.BOTTOM);
    bottomRightDicomAnnotation.fileName.setValue(filename);
    bottomRightDicomAnnotation.position.setValue(new SbVec3f(0.99f, -0.99f, 0));
    // Add text to annotation
    bottomRightDicomAnnotation.displayDicomInfo("Convolution Kernel : ", (short) 0x0018, (short) 0x1210);
    bottomRightDicomAnnotation.displayDicomInfo("Thickness : ", (short) 0x0018, (short) 0x0050);
    bottomRightDicomAnnotation.displayDicomInfo("kVp : ", (short) 0x0018, (short) 0x0060);

    mainSep.addChild(bottomRightDicomAnnotation);

    // Top right DICOM annotation
    DicomInfo topRightDicomAnnotation = new DicomInfo();
    topRightDicomAnnotation.alignmentH.setValue(DicomInfo.AlignmentH.RIGHT);
    topRightDicomAnnotation.fileName.setValue(filename);
    topRightDicomAnnotation.position.setValue(new SbVec3f(0.99f, 0.99f, 0));
    // Add text to annotation
    topRightDicomAnnotation.displayDicomInfo("Patient Birthday : ", (short) 0x0010, (short) 0x0030);
    topRightDicomAnnotation.displayDicomInfo("Aquisition date : ", (short) 0x0008, (short) 0x0022);
    topRightDicomAnnotation.displayDicomInfo("Patient's Age : ", (short) 0x0010, (short) 0x1010);
    topRightDicomAnnotation.displayDicomInfo("Sex : ", (short) 0x0010, (short) 0x0040);

    mainSep.addChild(topRightDicomAnnotation);

    return mainSep;
  }

  //@formatter:off
  /**
   * Make the scene appear larger or smaller.
   * <p>
   * Provided as a convenience to simplify application code. Values greater than
   * 1 make the scene appear larger. Values less than 1 make the scene appear
   * smaller. For example a of 2 will make the scene approximately 2 times
   * larger in the rendering window. The specific effect on the camera is a
   * 'dolly' or a 'zoom' depending on the type of camera.
   * <p>
   * This is often useful after calling viewAll(). That method sets the camera
   * parameters based on a bounding sphere around the scene which often leaves
   * the scene not "filling" the viewport.
   * <p>
   * Details:
   * <ul>
   * <li> SoPerspectiveCamera: <br>
   * changes the camera 'position' field. For
   * example, values greater than 1 move the camera closer to the focal point
   * (divide the camera's distance from the focal point by the given value),
   * which makes the scene appear larger. This is a 'dolly'.</li>
   * <p>
   * <li> SoOrthographicCamera: <br>
   * changes the camera 'height' field. For example,
   * values greater than 1 decrease the view volume height (scale the height by
   * the inverse of the given value), which makes the scene appear larger. This
   * is a 'zoom' operation.</li>
   * <p>
   * <li> No effect if value is <= 0 or camera is null.
   * </ul>
   * <p>
   * Note that the value expected by SoCameraInteractor.dolly is the inverse,
   * which is less intuitive.
   *
   * @param value
   *          The dolly value (ignored if zero).
   * @param cameraInteractor
   *          The interactor used to modify the camera (ignored if null).
   */
  //@formatter:on
  public static void dollyZoom(float value, SoCameraInteractor cameraInteractor)
  {
    if ( cameraInteractor != null && value != 0 )
      cameraInteractor.dolly(1 / value);
  }

  /**
   * Returns the example logo node
   *
   * @return Example logo node
   * @throws FileNotFoundException
   *           If the example logo resource is not found
   */
  public static SoNode getExampleLogoNode() throws FileNotFoundException
  {
    File logoFile = new File(SbFileHelper.expandString(EXAMPLE_LOGO));
    if ( !logoFile.exists() )
    {
      throw new FileNotFoundException("Resource " + EXAMPLE_LOGO + " not found");
    }

    SoImageBackground imgBg = new SoImageBackground();
    imgBg.style.setValue(SoImageBackground.Styles.LOWER_LEFT);
    imgBg.filename.setValue(EXAMPLE_LOGO);
    return imgBg;
  }

  /**
   * Adjust data range based on values in the DICOM file, i.e. the window center
   * (0028,1050) and window width (0028,1051) values. If there are multiple
   * slices, values are taken from the first slice. Volume is needed to get
   * DICOM attributes, but is not modified.
   *
   * @return true if successful. Assigns default values if necessary.
   */
  public static boolean dicomAdjustDataRange(SoDataRange rangeNode, SoVolumeData volume)
  {
    if ( rangeNode == null )
      return false;

    SbVec2f winCW = new SbVec2f(0, 0);
    boolean rc = MedicalHelper.dicomGetWindowCenterWidth(volume, winCW);
    rangeNode.min.setValue(winCW.getValueAt(0) - 0.5f * winCW.getValueAt(1));
    rangeNode.max.setValue(winCW.getValueAt(0) + 0.5f * winCW.getValueAt(1));
    return rc;
  }

  //@formatter:off
  /**
   * Get the window center (level) and width values from a DICOM volume. If the
   * query fails, returns false and sets 'winCW' to 0,0.
   * <p>
   * Uses the Window Center (0028,1050) and Window Width (0028,1051) tags from
   * the first image in the stack. If these tags do not exist in the volume,
   * then window center and width are computed from the actual data min and max
   * values. (Note that querying the actual min and max values may take some
   * time because every voxel must be loaded.) This method may be called with a
   * non-DICOM volume. In that case the actual min and max data values are used.
   * @return false if the query fails
   */
  //@formatter:on
  public static boolean dicomGetWindowCenterWidth(SoVolumeData volume, SbVec2f winCW)
  {
    boolean rc = false;
    float center = 0;
    float width = 0;

    if ( volume != null )
    {
      SoVolumeReader reader = volume.getReader();
      if ( reader != null && reader instanceof SoVRDicomFileReader )
      {
        SoVRDicomFileReader dicomReader = (SoVRDicomFileReader) (reader);
        SoVRDicomData dicomData = dicomReader.getDicomData();
        String str = dicomData.getDicomInfo((short) 0x0028, (short) 0x1050);

        if ( str.length() > 0 )
        {
          center = Float.parseFloat(str);
          str = dicomData.getDicomInfo((short) 0x0028, (short) 0x1051);

          if ( str.length() > 0 )
          {
            width = Float.parseFloat(str);
            rc = true;
          }
        }
      }

      // Tags were not found or this is not a DICOM volume.
      if ( !rc )
      {
        double volmin, volmax;
        long[] minMax = volume.getMinMax();
        volmin = minMax[0];
        volmax = minMax[1];
        width = (float) (volmax - volmin);
        center = (float) (volmin + 0.5f * width);
      }
    }
    winCW.setValue(center, width);
    return rc;
  }

  /**
   * Build a scene graph to display slice orientation markers.
   *
   * This is just a convenience method and helps keep the demo/example programs
   * consistent. Applications can use SliceOrientationMarkers directly.
   */
  public static SoSeparator buildSliceOrientationMarkers(SoOrthoSlice orthoSlice)
  {
    SoSeparator markerSep = new SoSeparator();

    // Color for slice orientation marker text.
    SoMaterial textMat = new SoMaterial();
    textMat.diffuseColor.setValue(1.0f, 1.0f, 0.25f);
    markerSep.addChild(textMat);

    // Enable background for text so it's visible against the image.
    SoTextProperty textProp = new SoTextProperty();
    textProp.margin.setValue(0.1f);
    textProp.style.setValue(SoTextProperty.Styles.BACK_FRAME);
    textProp.styleColors.set1Value(3, new SbColorRGBA(0.0f, 0.1f, 0.1f, 0.25f));
    markerSep.addChild(textProp);

    // Slice orientation markers.
    // Connect the axis field to the slice so the node will update
    // automatically.
    SliceOrientationMarkers markers = new SliceOrientationMarkers();
    if ( orthoSlice != null )
      markers.axis.connectFrom(orthoSlice.axis);
    markerSep.addChild(markers);

    return markerSep;
  }

  //@formatter:off
  /**
   * Build a scene graph to display dynamic scale bars for slice viewing.
   * <p>
   * This is just a convenience method and helps keep the demo/example programs
   * consistent. Applications can use SliceScaleBar directly.
   * <p>
   * Note that a typical length, 10 cm, is assumed.
   */
  //@formatter:on
  public static SoSeparator buildSliceScaleBars(SoCamera camera)
  {
    SoSeparator scaleSep = new SoSeparator();

    // Color for scale bars
    SoMaterial scaleColor = new SoMaterial();
    scaleColor.diffuseColor.setValue(1.0f, 0.25f, 0.25f);
    scaleSep.addChild(scaleColor);

    // Wider lines for better visibility
    SoDrawStyle scaleStyle = new SoDrawStyle();
    scaleStyle.lineWidth.setValue(2.0f);
    scaleSep.addChild(scaleStyle);

    // Horizontal scale bar
    SliceScaleBar scale1 = new SliceScaleBar();
    scale1.position.setValue(0, -0.99f);
    scale1.length.setValue(100); // 10 cm
    if ( camera != null )
      scale1.trackedCamera.setValue(camera);
    scale1.label.setValue("10cm");
    scaleSep.addChild(scale1);

    // Vertical scale bar
    SliceScaleBar scale2 = new SliceScaleBar();
    scale2.position.setValue(-0.99f, 0);
    scale2.length.setValue(100); // 10 cm
    scale2.orientation.setValue(SliceScaleBar.Orientation.VERTICAL);
    if ( camera != null )
      scale2.trackedCamera.setValue(camera);
    scale2.label.setValue("10cm");
    scaleSep.addChild(scale2);

    return scaleSep;
  }

  /**
   * Slice viewer annotations.
   *
   * This is just a convenience method and helps keep the demo/example programs
   * consistent. Applications can use SliceScaleBar (etc) directly.
   */
  public static SoSeparator buildSliceAnnotation(SoCamera camera, SoOrthoSlice sliceNode, String filename)
  {
    SoSeparator root = new SoSeparator();

    if ( camera == null || sliceNode == null )
      return root;

    // Slice orientation markers.
    root.addChild(MedicalHelper.buildSliceOrientationMarkers(sliceNode));

    // Dynamic scale bars
    root.addChild(MedicalHelper.buildSliceScaleBars(camera));

    // DICOM slice annotation
    if ( !filename.isEmpty() )
    {
      SoMaterial dicomMatl = new SoMaterial();
      dicomMatl.diffuseColor.setValue(0.8f, 0.8f, 0.5f);
      root.addChild(dicomMatl);

      DicomInfo upperLeft = new DicomInfo();
      upperLeft.fileName.setValue(filename);
      upperLeft.position.setValue(-0.99f, 0.99f, 0.0f);
      upperLeft.alignmentV.setValue(DicomInfo.AlignmentV.TOP);
      upperLeft.displayDicomInfo("", (short) 0x0010, (short) 0x0010); // Patient
                                                                      // Name
      upperLeft.displayDicomInfo("", (short) 0x0010, (short) 0x0030); // Patient
                                                                      // Birth
                                                                      // Date
      upperLeft.displayDicomInfo("", (short) 0x0008, (short) 0x1030); // Study
                                                                      // Description
      upperLeft.displayDicomInfo("", (short) 0x0008, (short) 0x103E); // Series
                                                                      // Description
      root.addChild(upperLeft);

      DicomInfo upperRight = new DicomInfo();
      upperRight.fileName.setValue(filename);
      upperRight.position.setValue(0.99f, 0.99f, 0.0f);
      upperRight.alignmentH.setValue(DicomInfo.AlignmentH.RIGHT);
      upperRight.alignmentV.setValue(DicomInfo.AlignmentV.TOP);
      upperRight.textAlignH.setValue(DicomInfo.AlignmentH.RIGHT);
      upperRight.displayDicomInfo("", (short) 0x0008, (short) 0x0080); // Institution
      upperRight.displayDicomInfo("", (short) 0x0008, (short) 0x0090); // Physician
      upperRight.displayDicomInfo("", (short) 0x0008, (short) 0x1090); // Model
                                                                       // name
      root.addChild(upperRight);

      DicomInfo lowerRight = new DicomInfo();
      lowerRight.fileName.setValue(filename);
      lowerRight.position.setValue(0.99f, -0.99f, 0.0f);
      lowerRight.alignmentH.setValue(DicomInfo.AlignmentH.RIGHT);
      lowerRight.alignmentV.setValue(DicomInfo.AlignmentV.BOTTOM);
      lowerRight.textAlignH.setValue(DicomInfo.AlignmentH.RIGHT);
      lowerRight.displayDicomInfo("", (short) 0x0008, (short) 0x0060); // Modality
      lowerRight.displayDicomInfo("mA: ", (short) 0x0018, (short) 0x1151); // X-Ray
                                                                           // Tube
                                                                           // Current
      lowerRight.displayDicomInfo("kV: ", (short) 0x0018, (short) 0x0060); // KVP
                                                                           // (Kilo
                                                                           // Voltage
      // Peak)
      lowerRight.displayDicomInfo("", (short) 0x0008, (short) 0x0022); // Acquisition
                                                                       // date
      root.addChild(lowerRight);

      TextBox lowerLeft = new TextBox();
      lowerLeft.position.setValue(-0.99f, -0.94f, 0.0f); // Leave room for OIV
                                                         // logo
      lowerLeft.alignmentV.setValue(TextBox.AlignmentV.BOTTOM);
      lowerLeft.addLine("Image: " + sliceNode.sliceNumber.getValue());
      root.addChild(lowerLeft);
    }

    return root;
  }

  //@formatter:off
  /**
   * Optimize volume data node for DICOM volumes.
   * <p>
   * The volume properties are modified (but not the data values). Do not call
   * this method before setting the volume data node's fileName field or calling
   * setReader.
   * <p>
   * <ul>
   * <li>First, adjusts the LDM tile size to optimize data loading. Note this
   * adjustment is valid for any data format that is not already tiled, i.e.
   * most formats except .lda and .ldm. </li>
   * <p>
   * <li>Second, adjusts the volume extent in 3D to be the actual physical extent
   * of the data set measured from outside edge to outside edge of the voxels.
   * This is a DICOM specific adjustment.</li>
   * <p>
   * <li>Third, if 'useImagePosition' is true, adjusts the volume extent so the
   * center of the first voxel is positioned at the coordinates specified in the
   * data set's "Image position" (0020,0032) attribute. By default the DICOM
   * volume reader sets the volume extent so the center of the volume is
   * positioned at 0,0,0 in 3D space. This is a DICOM specific adjustment.</li>
   * </ul>
   * @return true if successful
   */
  //@formatter:on
  public static boolean dicomAdjustVolume(SoVolumeData volume, boolean useImagePosition)
  {
    if ( volume == null )
      return false;

    // Get the reader being used to load data for this volume.
    SoVolumeReader reader = volume.getReader();

    // Part 1
    // -------------------------------------------------------------------
    // Not specific to DICOM files.
    //
    // Set tile size to contain the whole volume if possible.
    // The default LDM tile size is quite small and results in inefficient
    // handling
    // of typical DICOM volumes. On the other hand, for a typical DICOM volume
    // with 512x512 images, a tile size of 512 means that each slice of a tile
    // can
    // be loaded by efficiently loading an entire image. This adjustment works
    // for
    // other data formats as long as they are not already tiled, i.e. it does
    // not
    // work for .lda or .ldm files (trying to change tile size gives an error).
    //
    // Note this does NOT work for .ldm format data files (tile size is fixed).
    //
    // We recommend not making the tile size larger than 512. For a DICOM volume
    // that contains 12 or 16 bit data values, a tile of size 1024 would require
    // allocating 2 GB of contiguous memory. That request could fail even on the
    // CPU, let alone on the GPU.
    //
    // For very large volumes we still get the benefit of LDM data management
    // and
    // can visualize a volume of almost any size.
    if ( reader != null && !(reader instanceof SoLDMReader) )
    {
      // Get the largest dimension of the volume
      SbVec3i32 volDim = volume.data.getSize();
      int largestDim = Math.max(volDim.getX(), Math.max(volDim.getY(), volDim.getZ()));

      // If largestDim is zero, we haven't actually loaded any data yet!
      // There's nothing useful we can do here...
      if ( largestDim == 0 )
        return false;

      // Limit the tile size to 512 or less and force to be a power of 2
      // (VolumeViz requirement).
      largestDim = Math.min(largestDim, 512);
      largestDim = nextPowerOf2(largestDim);

      // Tiles can be asymmetric in the limited sense that the Z dimension can
      // be smaller than the X and Y dimension.
      // Also force power of 2 to be safe.
      int zDim = Math.min(largestDim, volDim.getZ());
      zDim = nextPowerOf2(zDim);

      // Update the volume properties.
      SoLDMResourceParameters resParms = volume.ldmResourceParameters.getValue();
      resParms.tileDimension.setValue(largestDim, largestDim, zDim);
    }

    // Part 2
    // -------------------------------------------------------------------
    // Not specific to DICOM files.
    //
    // Unless you take specific action, Open Inventor will load data on the GPU
    // using 1 byte per voxel regardless of the actual voxel data type. There
    // is no issue if the voxels are 1 byte (0..255) values. Otherwise the data
    // values may be scaled to fit in 1 byte values on the GPU. This effectively
    // reduces the precision of the data on the GPU because multiple actual
    // values
    // could be scaled to the same value. However note that when the application
    // asks for the value of a specific voxel, Open Inventor returns the value
    // from the original actual data, so these values are always correct. The
    // issue (if any) comes when you want to manipulate the data range very
    // precisely. DICOM data values are usually 12 or more bits per value and
    // require 2 bytes to store on the CPU. We can request that Open Inventor
    // store actual values on the GPU as well, using the texturePrecision field.
    //
    // This does use additional memory on the GPU, so there may be cases where
    // it's better to use the default setting on very low end hardware.
    //
    // Get the number of bytes per voxel.
    int bytesPerVoxel = volume.getDatumSize();
    if ( bytesPerVoxel == 2 ) // Store actual values on GPU
      volume.texturePrecision.setValue((short) 16);

    // Part 3
    // -------------------------------------------------------------------
    // DICOM specific
    //
    // There is a known bug that the physical extent of the volume is slightly
    // incorrect. VolumeViz considers the extent of the volume in 3D to span
    // from
    // the "outside edge" of the first voxel to the "outside edge" of the last
    // voxel. The DICOM file defines the size of a voxel and the position of the
    // _center_ of the first voxel.
    //
    // We can compute the correct physical extent for the X and Y axes because
    // we
    // can get the X and Y pixel size in physical units (mm) from the DICOM
    // file.
    // For some volumes it is possible to do the same thing for the Z axis using
    // the value of the Slice Spacing tag, but... this tag is not required and
    // may
    // not exist in every volume. Also some experts consider it unreliable. The
    // reliable way to compute the Z extent is by using the Image Position
    // values
    // from the first and last slices of the volume. And this is what the volume
    // reader does already, except that it multiplies by zDim-1 instead of zDim.
    // We just need to fix that computation.
    if ( reader != null && reader instanceof SoVRDicomFileReader )
    {
      SoVRDicomFileReader dicomReader = (SoVRDicomFileReader) reader;
      SoVRDicomData dicomData = dicomReader.getDicomData();

      // Get the X and Y physical voxel size from the DICOM data
      float xVoxelSize = dicomData.getXPixelSize();
      float yVoxelSize = dicomData.getYPixelSize();

      // There is no reliable tag to get this info for Z, so we just adjust the
      // reader's (slightly) incorrect calculation.
      SbVec3i32 volDim = volume.data.getSize();
      SbVec3f volSize = volume.extent.getValue().getSize();
      float zVoxelSize;
      if ( volDim.getZ() <= 1 ) // But we must not divide by zero...
        zVoxelSize = volSize.getZ();
      else
        zVoxelSize = volSize.getZ() / (volDim.getZ() - 1);

      // The adjustment we need to make is plus and minus 1/2 voxel.
      SbVec3f halfVoxel = new SbVec3f(xVoxelSize * 0.5f, yVoxelSize * 0.5f, zVoxelSize * 0.5f);

      if ( useImagePosition )
      {
        // Compute the physical extent of the volume.
        // For VolumeViz this means from outside edge to outside edge.
        SbVec3f volPhysicalSize =
            new SbVec3f(volDim.getX() * xVoxelSize, volDim.getY() * yVoxelSize, volDim.getZ() * zVoxelSize);

        // Get Image Position value from first image in DICOM stack.
        SbVec3f imagePos = dicomGetImagePosition(volume);

        // Compute volume extent in 3D so the first voxel is positioned
        // at the image position specified in the DICOM data.
        // Remember that VolumeViz considers the volume extent to completely
        // include every voxel. So given the position of the center of the
        // first voxel, the volume extent begins 1/2 voxel before that.
        SbVec3f newMin = imagePos.minus(halfVoxel); // Corners of volume extent
        SbVec3f newMax = newMin.plus(volPhysicalSize);
        SbBox3f newExtent = new SbBox3f(newMin, newMax);

        // Update the volume extent to get correct position/size in 3D space.
        volume.extent.setValue(newExtent);

        // Sanity check. useImagePosition gives correct position only for axis
        // aligned acquisitions.
        // For oblique acquisition, the more generic method
        // dicomAdjustVolume( SoVolumeData volume, SoMatrixTransform
        // imgToPatient )
        // should be used.
        if ( !dicomData.getOrientation().equals(SbMatrix.identity()) )
        {
          SoError.post(
              "Given DICOM volume doesn't seems to be axis aligned. You should use dicomAdjustVolume(SoVolumeData, SoMatrixTransform) method instead.");
          return false;
        }
      }
    }
    return true;
  }

  //@formatter:off
  /**
   * Similar to dicomAdjustVolume( SoVolumeData, boolean ) above but returns an SoMatrixTransform
   * that can be used to properly locate VolumeData in patient space in case of non axis aligned
   * acquisition:
   * <pre>
   * ...
   * SoMatrixTransform matrixTransform = new SoMatrixTransfrom();
   * dicomAdjustVolume(volumeData, matrixTransfrom)
   * root.addChild(matrixTransform);
   * root.addChild(volumeData);
   * ...
   * </pre>
   * The computed matrix embeds the image position and image orientation as described here:
   * https://dicom.innolitics.com/ciods/ct-image/image-plane/00200037
   * Scaling part is embedded in VolumeData extent.
   * <p>
   *  If SoVolumeData is not a DICOM volume, imgToPatient will be set to identity.
   *  </p>
   * @return true if successful
   */
  //@formatter:on
  public static boolean dicomAdjustVolume(SoVolumeData volume, SoMatrixTransform imgToPatient)
  {
    if ( imgToPatient == null )
    {
      SoError.post("null SoMatrixTransform specified.");
      return false;
    }

    if ( volume == null )
    {
      SoError.post("null volume specified.");
      imgToPatient.matrix.setValue(SbMatrix.identity());
      return false;
    }

    if ( !dicomAdjustVolume(volume, false) )
    {
      SoError.post("Error while adjusting volume extent.");
      imgToPatient.matrix.setValue(SbMatrix.identity());
      return false;
    }

    if ( !(volume.getReader() instanceof SoVRDicomFileReader) )
    {
      SoError.post("Cannot retrive DICOM reader.");
      imgToPatient.matrix.setValue(SbMatrix.identity());
      return false;
    }

    SbBox3f extent = volume.extent.getValue();
    SbVec3i32 dims = volume.data.getSize();

    if ( extent.getSize().getX() == 0.0f || extent.getSize().getY() == 0.0f )
    {
      SoError.post("Volume extent is empty. Cannot define spaces.");
      imgToPatient.matrix.setValue(SbMatrix.identity());
      return false;
    }

    if ( dims.getX() == 0 || dims.getY() == 0 || dims.getZ() == 0 )
    {
      SoError.post("Image has no dimension. Cannot define spaces.");
      imgToPatient.matrix.setValue(SbMatrix.identity());
      return false;
    }

    // Locate volume in DICOM image space: Center of top left voxel should be at (0, 0, 0)
    // By default, VolumeViz locate center of volume in origin.
    SbVec3f halfVoxelSize = new SbVec3f(0.5f * extent.getSize().getX() / dims.getX(),
      0.5f * extent.getSize().getX() / dims.getY(),
      0.5f * extent.getSize().getY() / dims.getZ());

    SbVec3f volumeSize = extent.getSize();
    volume.extent.setValue(-halfVoxelSize.getX(), -halfVoxelSize.getY(), -halfVoxelSize.getZ(),
        -halfVoxelSize.getX() + volumeSize.getX(), -halfVoxelSize.getY() + volumeSize.getY(),
        -halfVoxelSize.getZ() + volumeSize.getZ());

    SoVRDicomFileReader reader = (SoVRDicomFileReader) (volume.getReader());

    // Origin of center of top left voxel expressed in Patient space as defined in DICOM spec Patient Position (0020, 0032)
    SbVec3f OImgInPatient = reader.getDicomData().getImagePosition();

    // Build matrix of Image space expressed in patient space.
    // warning: OIV user row vector convention.
    // ImageOrientX x y z 0
    // ImageOrientY x y z 0
    // ImageOrientZ x y z 0
    // PatientPos   x y z 1
    SbMatrix ImgToPatientMatrix = reader.getDicomData().getOrientation();
    ImgToPatientMatrix.setElement(0, 3, OImgInPatient.getX());
    ImgToPatientMatrix.setElement(1, 3, OImgInPatient.getY());
    ImgToPatientMatrix.setElement(2, 3, OImgInPatient.getZ());

    imgToPatient.matrix.setValue(ImgToPatientMatrix);

    return true;
  }

  private static int nextPowerOf2(int n)
  {
    int result = 1;
    while ( result < n )
      result <<= 1;
    return result;
  }

  //@formatter:off
  /**
   * Get the "Image Position (Patient)" attribute (0020,0032) from a DICOM
   * volume.
   * <p>
   * DICOM calls this the upper left hand corner of the image, but more
   * precisely it's the <i>center</i> of the first voxel, in millimeters (mm).
   * For VolumeViz the upper left hand corner of the image is literally the <i>
   * corner</i> of the voxel, one-half voxel different from Image Position.
   * <p>
   * Note the SoVRDicomData getPosition() method does <i>not</i> return the
   * value of the Image Position attribute. It returns a value computed from
   * Image Position. If there are multiple slices, value is taken from the first
   * slice.
   *
   * @return 0,0,0 if not successful.
   */
  //@formatter:on
  public static SbVec3f dicomGetImagePosition(SoVolumeData volume)
  {
    SbVec3f imagePos = new SbVec3f(0, 0, 0);
    if ( volume != null )
    {
      SoVolumeReader reader = volume.getReader();
      if ( reader != null && reader instanceof SoVRDicomFileReader )
      {
        SoVRDicomFileReader dicomReader = (SoVRDicomFileReader) reader;
        SoVRDicomData dicomData = dicomReader.getDicomData();
        SiDicomDataSet dicomDataSet = dicomData.getDicomDataSet();

        // "Image position" tags
        SoDicomTag tag = new SoDicomTag((short) 0x0020, (short) 0x0032);
        SiDicomElement elt = dicomDataSet.getElement(tag);
        SiDicomValue value = elt.getValue();

        if ( value.getLength() == 3 )
          imagePos = new SbVec3f((float) value.getDouble(0), (float) value.getDouble(1), (float) value.getDouble(2));
      }
    }
    return imagePos;
  }

  /**
   * Convenience function to draw a specified bounding box as a wireframe
   */
  public static SoSeparator createBoundingBox(SbBox3f bbox, SbColor color)
  {
    SoSeparator bboxSep = new SoSeparator();

    // Position and size the Box
    SoTransform transform = new SoTransform();
    transform.translation.setValue(bbox.getCenter());
    transform.scaleFactor.setValue(bbox.getSize().times(0.5f));
    bboxSep.addChild(transform);

    // Box should not interfere with picking other geometry
    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
    bboxSep.addChild(pickStyle);

    // The box will be easier to see without lighting and with wide lines
    SoLightModel cubeModel = new SoLightModel();
    cubeModel.model.setValue(SoLightModel.Models.BASE_COLOR);
    bboxSep.addChild(cubeModel);

    SoBaseColor cubeColor = new SoBaseColor();
    cubeColor.rgb.setValue((color == null) ? new SbColor(1, 0, 0) : color);
    bboxSep.addChild(cubeColor);

    SoDrawStyle cubeStyle = new SoDrawStyle();
    cubeStyle.lineWidth.setValue(1);
    cubeStyle.style.setValue(SoDrawStyle.Styles.LINES);
    bboxSep.addChild(cubeStyle);

    SoCube cube = new SoCube();
    bboxSep.addChild(cube);

    return bboxSep;
  }

  /**
   * Convenience function to retrieve the bounding box of the specified node. If
   * node contains SoVolumeData, use SoVolumeData extent.
   */
  public static SbBox3f getBoundingBox(SoNode node)
  {
    // retrieve root node of volume data and compute its boundingBox.
    // Try to find a SoVolumeData in specified root node.
    // If SoVolumeData found, use its extend as bbox.
    // Else, apply an SoGetBoundingBoxAction on specified root node.
    SoVolumeData volumeData = find(node, SoVolumeData.class);
    if ( volumeData != null )
    {
      return volumeData.extent.getValue();
    }
    else
    {
      // Use fake viewport for SoGetBoundingBoxAction. see
      // SoGetBoundingBoxAction constructor for details.
      SbViewportRegion fakeViewportRegion = new SbViewportRegion(new SbVec2i32(1, 1));
      SoGetBoundingBoxAction gba = new SoGetBoundingBoxAction(fakeViewportRegion);
      gba.apply(node);

      return gba.getBoundingBox();
    }
  }

  /**
   * Convenience function to retrieve the cube corresponding to the specified
   * bounding box.
   */
  public static SoSeparator createCube(SbBox3f bbox)
  {
    SoSeparator bboxSep = new SoSeparator();
    SoTransform transform = new SoTransform();
    transform.translation.setValue(bbox.getCenter());
    SbVec3f size = bbox.getSize();
    size.multiply(0.5f);
    transform.scaleFactor.setValue(size);
    bboxSep.addChild(transform);

    SoCube cube = new SoCube();
    bboxSep.addChild(cube);

    return bboxSep;
  }

  /**
   * Equivalent to {@link #orientView(Axis, SoCamera, SoVolumeData, float)
   * orientView(axis, camera, volume, 1.01f)}
   */
  public static boolean orientView(MedicalHelper.Axis axis, SoCamera camera, SoVolumeData volume)
  {
    return orientView(axis, camera, volume, 1.01f);
  }

  //@formatter:off
  /**
   * Adjusts the camera to view a slice or volume along the specified axis.
   * <p>
   * The camera is rotated to give the conventional orientation for the specified
   * axis. If the volume data is specified, then viewAll() is called to center
   * the volume in the view volume.
   * <p>
   * If the camera is an SoOrthographicCamera (the usual case for viewing a
   * slice) and the volume data node is specified, then the camera (specifically
   * the view volume height) is adjusted to make the slice image fill the
   * viewport. This is convenient because the viewAll() method gives a very
   * loose fit around the slice image.
   * <p>
   * If the camera is an SoPerspectiveCamera (commonly used for viewing a 3D
   * volume) or the volume data node is specified, then viewAll() is called, but
   * no additional adjustment is done to the camera. 'slack' is ignored in this
   * case.
   * <p>
   * Axial -> "Feet" view. Coronal -> "Anterior" view. Sagittal -> "Left" view.
   *
   * @param axis
   *          Axial, Coronal or Sagittal
   * @param camera
   *          The camera node that will be modified (fails if null).
   * @param volume
   *          The data volume that is being viewed (used to compute view volume,
   *          not modified)
   * @param slack
   *          Values greater than 1 will leave additional space between image and window edge.
   *
   * @return true if the operation succeeded.
   */
  //@formatter:on
  public static boolean orientView(MedicalHelper.Axis axis, SoCamera camera, SoVolumeData volume, float slack)
  {
    if ( camera == null )
      return false;

    // We don't actually modify the data volume, but we can't query the
    // extent if it is constant.
    SbVec3f size = new SbVec3f(1.0f, 1.0f, 1.0f);
    SbBox3f bbox = new SbBox3f(-1.0f, -1.0f, -1.0f, 1.0f, 1.0f, 1.0f);
    if ( volume != null )
    {
      bbox = volume.extent.getValue();
      size = bbox.getSize();
    }
    float width  = 1;
    float height = 1;

    SbRotation orientation = new SbRotation();
    if ( axis == Axis.SAGITTAL ) // X axis / Z up
    {
      width = size.getY();
      height = size.getZ();
      orientation = new SbRotation(new SbVec3f(0, 0, 1), (float) (Math.PI / 2.0));
      SbRotation rot2 = new SbRotation(new SbVec3f(0, 1, 0), (float) (Math.PI / 2.0));
      orientation.multiply(rot2);
    }
    else if ( axis == Axis.CORONAL ) // Y axis / Z up
    {
      width = size.getX();
      height = size.getZ();
      orientation = new SbRotation(new SbVec3f(1, 0, 0), (float) (Math.PI / 2.0));
    }
    else // Z axis - Axial / Y up
    {
      width = size.getX();
      height = size.getY();
      orientation = new SbRotation(new SbVec3f(1, 0, 0), (float) (Math.PI));
    }

    // Notes:
    // - We use viewAll() to get the camera position, near/far clip planes, etc
    // set.
    // - We use a fake viewport here to do the viewAll. Shouldn't be a problem.
    // - Then we adjust the view volume height to get a "tight" fit around the
    // slice.
    SbViewportRegion vport = new SbViewportRegion((short) 500, (short) 500);
    camera.orientation.setValue(orientation);
    if ( camera instanceof SoPerspectiveCamera )
    {
      // TODO: Compute camera position to fill the viewport.
      if ( volume != null )
        camera.viewAll(bbox, vport);
    }
    else
    {
      // Orthographic Camera
      SoOrthographicCamera orthoCam = (SoOrthographicCamera) camera;
      if ( volume != null )
      {
        orthoCam.viewAll(bbox, vport);
        height = (height >= width) ? height : width;
        orthoCam.height.setValue(height * slack);
      }
    }

    return true;
  }

  /**
   * Convenience function to find first specified node in the scene graph.
   *
   * @param root
   *          root of scene graph you want to search
   * @param nodeClass
   *          class of the node you want to find
   * @param nodeName
   *          Name of the node you want to find
   * @return The found node or null if nothing found
   */
  @SuppressWarnings("unchecked")
  public static <NodeClass extends SoNode> NodeClass find(SoNode root, Class<NodeClass> nodeClass, String nodeName)
  {
    if ( root == null )
      return null;

    SoSearchAction sa = new SoSearchAction();
    if ( nodeName.length() > 0 )
      sa.setName(nodeName);

    sa.setSearchingAll(true);
    sa.setNodeClass(nodeClass);
    sa.setInterest(SoSearchAction.Interests.FIRST);
    sa.apply(root);

    SoPath path = sa.getPath();
    if ( path != null )
      return (NodeClass) path.regular.getTail();

    return null;
  }

  /**
   * Convenience function to find first specified node in the scene graph.
   *
   * @param root
   *          root of scene graph you want to search
   * @param nodeClass
   *          class of the node you want to find
   * @return The found node or null if nothing found
   */
  public static <NodeClass extends SoNode> NodeClass find(SoNode root, Class<NodeClass> nodeClass)
  {
    return find(root, nodeClass, "");
  }

  /**
   * Convenience function to read a .iv file and return pointer to the root of
   * the created scene graph.
   *
   * @param filename
   *          of file containing a scene graph.
   * @return Root of the read scene graph or null if the file cannot be read.
   */
  public static SoSeparator readFile(String filename)
  {
    // Open the input file
    SoInput mySceneInput = new SoInput();
    if ( !mySceneInput.openFile(filename) )
    {
      SoError.post("Cannot open file " + filename);
      return null;
    }

    // Read the whole file into the database
    SoSeparator myGraph = SoDB.readAll(mySceneInput);
    if ( myGraph == null )
    {
      SoError.post("Problem reading file " + filename);
      return null;
    }

    mySceneInput.closeFile();
    return myGraph;
  }

}
