///////////////////////////////////////////////////////////////////////////////
//
// This code is part of the Open Inventor Medical Edition utility library.
//
// Open Inventor customers may use this source code to create or enhance Open
// Inventor-based applications.
//
// The medical utility classes are provided as a jar named com.openinventor.medical.jar,
// that can be used directly in an Open Inventor application. The classes in this jar
// are documented and supported by FEI. These classes are also provided as source code.
//
///////////////////////////////////////////////////////////////////////////////
package com.openinventor.medical.nodes;

import com.openinventor.inventor.SbMatrix;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.actions.SoAction;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.elements.SoViewingMatrixElement;
import com.openinventor.inventor.elements.SoViewportRegionElement;
import com.openinventor.inventor.fields.SoField;
import com.openinventor.inventor.fields.SoSFBool;
import com.openinventor.inventor.fields.SoSFFloat;
import com.openinventor.inventor.fields.SoSFInt32;
import com.openinventor.inventor.fields.SoSFVec2i32;
import com.openinventor.inventor.misc.SoState;
import com.openinventor.inventor.nodes.*;

//@formatter:off
/**
 * <strong>(Preview Feature) </strong>Shape node to display a medical gnomon (compass) on the screen.
 *
 * <p style="background-color: #c7c7c7;">Preview Feature means this class is fully supported and can be used in Open Inventor applications.
 * Being tagged as a Preview Feature just means that the implementation is still
 * subject to API changes and adjustments based on feedback from early adopters.
 * Please be also aware that source compatibility might be broken regardless of
 * the Open Inventor compatibility changes policy due to our commitment to bring
 * needed changes to be sure the specifications of this Preview Feature match
 * the expectations of our customers.
 * </p>
 *
 * This node displays a medical specific "gnomon" (compass) on the screen that
 * shows the user the current orientation of the volume in 3D space.
 *
 * Visibility of the gnomon is controlled by the {@link #isDisplayed} field.
 *
 * Position and size of the gnomon are specified in pixels using the {@link #position},
 * {@link #width} and {@link #height} fields.
 * <p>
 * <b>File format/default:</b>
 * <p>
 * Gnomon {
 * <ul><table>
 * <tr><td> isDisplayed    </td><td> TRUE         </td></tr>
 * <tr><td> position       </td><td> 0 0          </td></tr>
 * <tr><td> width          </td><td> 100          </td></tr>
 * <tr><td> height         </td><td> 100          </td></tr>
 * </table></ul> }
 *
 * @see DicomInfo
 * @see Magnifier
 * @see Ruler
 *
 */
//@formatter:on
public class Gnomon extends SoAnnotation
{

  /**
   * Controls whether the gnomon is visible (default is true).
   */
  public SoSFBool isDisplayed;

  /**
   * Position of the gnomon viewport in pixels (default is 0,0).
   */
  public SoSFVec2i32 position;

  /**
   * Specifies the width of the gnomon viewport in pixels (default is 100).
   */
  public SoSFInt32 width;

  /**
   * Specifies the height of the gnomon viewport in pixels (default is 100).
   */
  public SoSFInt32 height;

  /**
   * Specifies the distance from the camera to the gnomon geometry. Default is
   * 3.5. Applications should not need to modify this value.
   */
  public SoSFFloat cameraDistance;

  /**
   * Constructor
   */
  public Gnomon()
  {
    isDisplayed = new SoSFBool(this, "isDisplayed", SoField.FieldTypes.EXPOSED_FIELD, true);
    position = new SoSFVec2i32(this, "position", SoField.FieldTypes.EXPOSED_FIELD, new SbVec2i32(0, 0));
    width = new SoSFInt32(this, "width", SoField.FieldTypes.EXPOSED_FIELD, 100);
    height = new SoSFInt32(this, "height", SoField.FieldTypes.EXPOSED_FIELD, 100);
    cameraDistance = new SoSFFloat(this, "cameraDistance", SoField.FieldTypes.EXPOSED_FIELD, 3.5f);

    // Name main node
    setName("Gnomon");

    _displaySwitch = new SoSwitch();
    _displaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE); // Off until we
                                                                 // have valid
                                                                 // geometry
    // Define Callback
    _callback = new SoCallback();
    _callback.setCallback(new GnomonRenderCB());

    // Setup gnomon camera (viewAll might not work here because we want
    // the point of rotation to be the origin of the axes (i.e. 0,0,0),
    // not the geometric center of the gnomon geometry.
    _camera = new SoOrthographicCamera();
    _camera.position.setValue(new SbVec3f(0, 0, cameraDistance.getValue()));
    _camera.nearDistance.setValue(1);
    _camera.farDistance.setValue(6);
    _camera.focalDistance.setValue(cameraDistance.getValue());

    // Holder for our geometry
    _geometry = new SoSeparator();

    buildSceneGraph();
  }

  /**
   * Builds the scene graph that represents a gnomon object.
   */
  protected void buildSceneGraph()
  {

    // Reset the bounding box so gnomon geometry will not
    // affect bbox of scene (ie. will not affect viewAll).
    SoBBox bboxNode = new SoBBox();
    bboxNode.mode.setValue(SoBBox.Modes.NO_BOUNDING_BOX);
    this.addChild(bboxNode);

    // Switch controls gnomon visibility
    this.addChild(_displaySwitch);

    // Under the switch is our callback, camera and geometry
    _displaySwitch.addChild(_callback);
    _displaySwitch.addChild(_camera);

    _gnomonScale = new SoScale();
    _displaySwitch.addChild(_gnomonScale);
    _displaySwitch.addChild(_geometry);

    // Read the gnomon geometry from a memory buffer
    SoInput inGeometry = new SoInput();
    inGeometry.setBuffer(gnomonGeometry, gnomonGeometry.length());

    SoNode node;
    node = SoDB.read(inGeometry);
    if ( node != null )
    {
      _geometry.addChild(node);
      _displaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    }
  }

  protected SoCallback _callback;
  protected SoOrthographicCamera _camera;
  protected SoSwitch _displaySwitch;
  protected SoScale _gnomonScale;
  protected SoSeparator _geometry;

  /**
   * Gnomon callback function
   *
   * This function will be called (from our Callback node) just before the
   * gnomon camera is traversed. It does some drawing setup and modifies the
   * gnomon camera, as follows:
   *
   * 1) Modify OpenGL viewport to position the gnomon in the window. 2) Clear
   * the OpenGL depth buffer in the region where the gnomon will be drawn (to
   * ensure gnomon is "on top"). 3) Modify the gnomon camera to: a) Have the
   * same orientation as the 3D scene camera. b) View the gnomon geometry at a
   * consistent size.
   *
   * @param action
   *          Action define operations to be applied as each node during
   *          traversal of scene graph
   */
  private class GnomonRenderCB implements SoCallback.CB
  {
    @Override
    public void invoke(SoAction action)
    {
      // Don't do anything if this is not a render traversal
      if ( action instanceof SoGLRenderAction )
      {
        // Suppress auto-caching above the callback node to ensure
        // that this code will actually be traversed on every render.
        // The gnomon geometry itself is under a Separator, which
        // should be cached, so there is no performance penalty.
        SoState state = action.getState();
        ((SoGLRenderAction) action).shouldAutoCache(SoGLRenderAction.AutoCacheModes.DONT_AUTO_CACHE);

        // Set the OpenGL viewport to position gnomon in the window.
        //
        // Note that we modify the Open Inventor traversal state (rather
        // than calling OpenGL directly) so that any nodes that depend on
        // knowing the viewport will function correctly.
        //
        // The default gnomon position is in the lower-left corner.
        SbViewportRegion vport = new SbViewportRegion(new SbVec2i32(width.getValue(), height.getValue()));
        SbVec2i32 vportOrig = position.getValue();
        SbVec2i32 vportSize = new SbVec2i32(width.getValue(), height.getValue());
        vport.setViewportPixels(vportOrig, vportSize);
        SoViewportRegionElement.set(state, vport);

        // Get the current camera rotation from the viewing matrix.
        //
        // Note that the viewing matrix is applied to geometry, so it's
        // actually the inverse of the matrix we need. The getTransform
        // method will return the rotation in variable cameraRotation.
        //
        // We could get the camera orientation by hiding a pointer to the
        // camera, but this approach is very "fragile". For example, if the
        // user changes the camera type by clicking the viewer button, the
        // viewer destroys the current camera and creates a new one.
        SbMatrix viewMat = SoViewingMatrixElement.get(state);

        SbRotation cameraRotation = viewMat.inverse().decompose().rotation;

        // Disable notification
        _camera.enableNotify(false);

        // Set the new orientation for the gnomon camera
        _camera.orientation.setValue(cameraRotation);

        // Get the current "focal distance" (distance to point of rotation)
        float distance = _camera.focalDistance.getValue();

        // Reposition the camera so it's looking at the pt-of-rotation
        // (which is the center of the gnomon or 0,0,0 in this case).
        //
        // We'll use an algorithm similar to what the examiner viewer uses
        // to spin the camera around the pt-of-rotation.
        // First get the rotation as a matrix. Conveniently SbMatrix has an
        // assignment operator that takes an SbRotation value.
        // Next extract the view direction vector from the rotation matrix.
        // Now move the camera radius units along the direction vector.
        SbMatrix mx = cameraRotation.getMatrix();
        SbVec3f direction = new SbVec3f(-mx.getColumn(2)[0], -mx.getColumn(2)[1], -mx.getColumn(2)[2]);
        direction.multiply(distance);
        _camera.position.setValue(new SbVec3f(0, 0, 0).minus(direction));

        // re-enable notification
        _camera.enableNotify(false);
      }
    }
  }

  /**
   * Define the gnomon geometry (simple axes in this case)
   *
   * We could define these programmatically, but using a memory stream avoids
   * including headers for nodes that we don't actually use in our program. We
   * could define these in a .iv file, but the memory stream avoids the danger
   * of not having the file available at run time.
   *
   * For convenience we define the gnomon geometry in a -1..1 space because this
   * is the default view volume for Open Inventor cameras.
   *
   * Note that, to be safe, we reset some attributes that might be inherited
   * from the scene graph and are not appropriate for our axes. But to be safe
   * you should also keep the actual scene under its own Separator, so its
   * attributes won't affect the gnomon.
   */
// @formatter:off
  final String gnomonGeometry = ""
      + "#Inventor V2.1 ascii\n"
      + "DEF GnomonGeom Separator { \n"
      + "  DepthBuffer { test TRUE write TRUE } \n"
      + "  PickStyle { style UNPICKABLE } \n"
      + "  DrawStyle { style FILLED } \n"
      + "  LightModel { model PHONG } \n"
      + "  MaterialBinding { value OVERALL } \n"
      + "  Complexity { value .2 } \n"
      + "  ShapeHints { vertexOrdering COUNTERCLOCKWISE shapeType SOLID } \n"
      + "  RotationXYZ { axis X angle 1.570796327 } \n"
      + "  Separator { \n"
      + "    Cube { \n"
      + "      width 1 \n"
      + "      height 1 \n"
      + "      depth 1 \n"
      + "    } \n"
      + "    DrawStyle { \n"
      + "      style LINES \n"
      + "    } \n"
      + "    Material { \n"
      + "    } \n"
      + "    Cube { \n"
      + "      width 1.01 \n"
      + "      height 1.01 \n"
      + "      depth 1.01 \n"
      + "    } \n"
      + "  } \n"
      + "  Font { \n"
      + "    name arial \n"
      + "  } \n"
      + "  Separator { \n"
      + "    Transform { \n"
      + "      translation 0 -0.4 0.6 \n"
      + "      rotation 0 0 1  0  \n"
      + "      scaleFactor 0.1 0.1 0.1 \n"
      + "      scaleOrientation 0 0 1  0 \n"
      + "    } \n"
      + "    Material { \n"
      + "      diffuseColor 0 0 1 \n"
      + "    } \n"
      + "    Text3 { \n"
      + "      parts ALL \n"
      + "      string A \n"
      + "      justification CENTER \n"
      + "    } \n"
      + "  } \n"
      + "  Separator { \n"
      + "    Transform { \n"
      + "      translation -0.6 -0.4 0 \n"
      + "      rotation 0 -1 0  1.55 \n"
      + "      scaleFactor 0.1 0.1 0.1 \n"
      + "      scaleOrientation 0 1 0 0 \n"
      + "    } \n"
      + "    Material { \n"
      + "      diffuseColor 0 1 0 \n"
      + "    } \n"
      + "    Text3 { \n"
      + "      string R \n"
      + "      parts ALL \n"
      + "      justification CENTER \n"
      + "    } \n"
      + "  } \n"
      + "  Separator { \n"
      + "    Transform { \n"
      + "      translation 0.6 -0.4 0 \n"
      + "      rotation 0 1 0 1.55 \n"
      + "      scaleFactor 0.1 0.1 0.1 \n"
      + "      scaleOrientation 0 1 0  0 \n"
      + "    } \n"
      + "    Material { \n"
      + "      diffuseColor 1 0 0 \n"
      + "    } \n"
      + "    Text3 { \n"
      + "      justification CENTER \n"
      + "      parts ALL \n"
      + "      string L \n"
      + "    } \n"
      + "  } \n"
      + "  Separator { \n"
      + "    Transform { \n"
      + "      translation 0 -0.4 -0.6 \n"
      + "      rotation 0 1 0  3.14 \n"
      + "      scaleFactor 0.1 0.1 0.1 \n"
      + "      scaleOrientation 0 0 1  0 \n"
      + "    } \n"
      + "    Material { \n"
      + "      diffuseColor 0 0 1 \n"
      + "    } \n"
      + "    Text3 { \n"
      + "      string P \n"
      + "      justification CENTER \n"
      + "      parts ALL \n"
      + "    } \n"
      + "  } \n"
      + "  Separator { \n"
      + "    Transform { \n"
      + "      translation 0 0.6 0.35 \n"
      + "      rotation -1 0 0  1.55 \n"
      + "      scaleFactor 0.1 0.1 0.1 \n"
      + "      scaleOrientation 0 0 1  0 \n"
      + "    } \n"
      + "    Material { \n"
      + "      diffuseColor 1 1 1 \n"
      + "    } \n"
      + "    Text3 { \n"
      + "      justification CENTER \n"
      + "      parts ALL \n"
      + "      string H \n"
      + "    } \n"
      + "  } \n"
      + "  Separator { \n"
      + "    Transform { \n"
      + "      translation 0 -0.6 -0.35 \n"
      + "      rotation 1 0 0  1.55 \n"
      + "      scaleFactor 0.1 0.1 0.1 \n"
      + "      scaleOrientation 0 0 1  0 \n"
      + "    } \n"
      + "    Material { \n"
      + "      diffuseColor 0.5 0.5 0.5 \n"
      + "    } \n"
      + "    Text3 { \n"
      + "      string F \n"
      + "      justification CENTER \n"
      + "      parts ALL \n"
      + "    } \n"
      + "  } \n"
      + "} ";
//@formatter:on
}
