///////////////////////////////////////////////////////////////////////////////
//
// This code is part of the Open Inventor Medical Edition utility library.
//
// Open Inventor customers may use this source code to create or enhance Open
// Inventor-based applications.
//
// The medical utility classes are provided as a jar named com.openinventor.medical.jar,
// that can be used directly in an Open Inventor application. The classes in this jar
// are documented and supported by FEI. These classes are also provided as source code.
//
///////////////////////////////////////////////////////////////////////////////
package com.openinventor.medical.nodes;

import com.openinventor.inventor.IntegerValuedEnum;
import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.fields.SoField;
import com.openinventor.inventor.fields.SoSFBool;
import com.openinventor.inventor.fields.SoSFColor;
import com.openinventor.inventor.fields.SoSFEnum;
import com.openinventor.inventor.fields.SoSFUInt32;
import com.openinventor.inventor.nodes.*;
import com.openinventor.volumeviz.elements.SoVolumeDataElement;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;

//@formatter:off
/**
 * <strong>(Preview Feature) </strong>Ortho slice shape node with border.
 *
 * <p style="background-color: #c7c7c7;">Preview Feature means this class is fully supported and can be used in Open Inventor applications.
 * Being tagged as a Preview Feature just means that the implementation is still
 * subject to API changes and adjustments based on feedback from early adopters.
 * Please be also aware that source compatibility might be broken regardless of
 * the Open Inventor compatibility changes policy due to our commitment to bring
 * needed changes to be sure the specifications of this Preview Feature match
 * the expectations of our customers.
 * </p>
 *
 * Even if this node does not extend the class SoOrthoSlice, it defines a custom
 * ortho (axis aligned) slice along the X, Y, or Z axis of a volume defined by
 * an SoVolumeData node.
 * <p>
 * In addition to the standard features of SoOrthoSlice, this node can render
 * a {@link #border} around the slice using the specified {@link #borderColor}.
 * <p>
 * The border color can be used, for example, to help the user see at a glance
 * which axis each slice corresponds to. Another use is to change the border
 * color when a slice is selected.
 * <p>
 * <b>File format/default:</b>
 * <p>
 * OrthoSliceBorder {
 * <ul><table>
 * <tr><td> sliceNumber       </td><td> 0            </td></tr>
 * <tr><td> axis              </td><td> Z            </td></tr>
 * <tr><td> interpolation     </td><td> LINEAR       </td></tr>
 * <tr><td> alphaUse          </td><td> ALPHA_BINARY </td></tr>
 * <tr><td> useRGBA           </td><td> FALSE        </td></tr>
 * <tr><td> clipping          </td><td> FALSE        </td></tr>
 * <tr><td> clippingSide      </td><td> BACK         </td></tr>
 * <tr><td> alternateRep      </td><td> NULL         </td></tr>
 * <tr><td> enableBumpMapping </td><td> FALSE        </td></tr>
 * <tr><td> bumpScale         </td><td> 1.0          </td></tr>
 * <tr><td> border            </td><td> TRUE         </td></tr>
 * <tr><td> borderColor       </td><td> 1 0.5 0      </td></tr>
 * </table></ul> }
 * <p>
 * <b>See also:</b>
 * {@link com.openinventor.volumeviz.nodes.SoOrthoSlice},
 * {@link com.openinventor.volumeviz.nodes.SoObliqueSlice},
 * {@link ObliqueSliceBorder}
 *
 */
//@formatter:on
public class OrthoSliceBorder extends SoNode
{
  /**
   * Enable the border. Default is true.
   */
  public SoSFBool border;

  /**
   * Border color. Default is orange (1, 0.5, 0).
   */
  public SoSFColor borderColor;

  /**
   * Slice Axis values
   */
  public SoSFEnum<AxisType> axis;

  /**
   * Slice number.
   */
  public SoSFUInt32 sliceNumber;

  /**
   * Slice Axis values
   */
  public enum AxisType implements IntegerValuedEnum
  {
    /** Left edge */
    X(SoOrthoSlice.AxisType.X.getValue()),
    /** Center */
    Y(SoOrthoSlice.AxisType.Y.getValue()),
    /** Right edge */
    Z(SoOrthoSlice.AxisType.Z.getValue());

    private AxisType(int axis)
    {
      _axis = axis;
    }

    @Override
    public int getValue()
    {
      return _axis;
    }

    private final int _axis;
  };

  /**
   * Constructor
   */
  public OrthoSliceBorder()
  {
    border = new SoSFBool(this, "Border", SoField.FieldTypes.EXPOSED_FIELD, true);
    borderColor =
        new SoSFColor(this, "BorderColor", SoField.FieldTypes.EXPOSED_FIELD, new SbColor(0.84f, 0.43f, 0.02f));
    axis = new SoSFEnum<AxisType>(this, "Axis", SoField.FieldTypes.EXPOSED_FIELD, AxisType.class, AxisType.Z);
    sliceNumber = new SoSFUInt32(this, "SliceNumber", SoField.FieldTypes.EXPOSED_FIELD, 0);
    buildSceneGraph();
  }

  /**
   * Called by SoGLRenderAction
   */
  @Override
  public void GLRender(SoGLRenderAction action)
  {
    // Update border on/off
    boolean borderEnabled = border.getValue();
    if ( borderEnabled == true )
      _switch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    else
      _switch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);

    // If border is turned off, nothing else to do.
    if ( !borderEnabled )
      return;

    // Get geometry in case we need to change geometry or color
    SoVertexProperty vprop = (SoVertexProperty) _geometry.vertexProperty.getValue();

    // Make sure color is current
    vprop.orderedRGBA.set1Value(0, borderColor.getValue().getPackedValue());

    // Get current volume data in the traversal state.
    // This should also create a dependency on the volumeDataElement so this
    // method is called again if there is any change to the volume data node.
    SoVolumeData volume = SoVolumeDataElement.getVolumeData(action.getState());

    // If there is a volume data (else nothing to do)
    if ( volume != null )
    {
      SbBox3f volExt = volume.extent.getValue();

      // Get slice axis and position
      int axisIndex = axis.getValue();
      int sliceNum = sliceNumber.getValue();

      // If parameters have changed
      if ( volExt != _volumeExtent || axisIndex != _sliceAxis || sliceNum != _sliceNumber )
      {
        // Remember current parameters
        _sliceAxis = axisIndex;
        _sliceNumber = sliceNum;
        _volumeExtent = volExt;

        // Update vertices of geometry depending on slice axis.
        SbVec3i32 volDim = volume.data.getSize();
        SbVec3f min = volExt.getMin();
        SbVec3f max = volExt.getMax();
        vprop.vertex.setNum(5);

        if ( axisIndex == 0 ) // X-axis slice
        {
          float voxelSize = (max.getX() - min.getX()) / volDim.getX();
          float x = min.getX() + ((sliceNum + 0.5f) * voxelSize);
          vprop.vertex.set1Value(0, x, min.getY(), min.getZ());
          vprop.vertex.set1Value(1, x, max.getY(), min.getZ());
          vprop.vertex.set1Value(2, x, max.getY(), max.getZ());
          vprop.vertex.set1Value(3, x, min.getY(), max.getZ());
          vprop.vertex.set1Value(4, x, min.getY(), min.getZ());
        }
        else if ( axisIndex == 1 ) // Y-axis slice
        {
          float voxelSize = (max.getY() - min.getY()) / volDim.getY();
          float y = min.getY() + ((sliceNum + 0.5f) * voxelSize);
          vprop.vertex.set1Value(0, min.getX(), y, min.getZ());
          vprop.vertex.set1Value(1, max.getX(), y, min.getZ());
          vprop.vertex.set1Value(2, max.getX(), y, max.getZ());
          vprop.vertex.set1Value(3, min.getX(), y, max.getZ());
          vprop.vertex.set1Value(4, min.getX(), y, min.getZ());
        }
        else // Z-axis slice
        {
          float voxelSize = (max.getZ() - min.getZ()) / volDim.getZ();
          float z = min.getZ() + ((sliceNum + 0.5f) * voxelSize);
          vprop.vertex.set1Value(0, min.getX(), min.getY(), z);
          vprop.vertex.set1Value(1, max.getX(), min.getY(), z);
          vprop.vertex.set1Value(2, max.getX(), max.getY(), z);
          vprop.vertex.set1Value(3, min.getX(), max.getY(), z);
          vprop.vertex.set1Value(4, min.getX(), min.getY(), z);
        }
      }
    }
    else
    { // No volume to render, so no border
      _switch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    }

    // Traverse our internal scene graph
    action.forwardTraversal(_scene);
  }

  /**
   * Builds the scene graph that represents a slice border object.
   */
  private void buildSceneGraph()
  {
    // Initially we don't know the volume extent or slice positions
    // Initialize member variables
    _sliceAxis = -1;
    _sliceNumber = -1;
    _volumeExtent = new SbBox3f();
    _volumeExtent.makeEmpty();

    // Build the internal scene graph
    _scene = new SoSeparator();

    // Switch to turn border on/off
    _switch = new SoSwitch();
    if ( border.getValue() == true )
      _switch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    else
      _switch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    _scene.addChild(_switch);

    // Pull the border slightly toward the camera to avoid Z-buffer fighting.
    // You might think that we could apply PolygonOffset to the OrthoSlice
    // geometry, but we'd have to use override because the OrthoSlice node has
    // it's own internal PolygonOffset node.
    SoPolygonOffset offset = new SoPolygonOffset();
    offset.units.setValue(-1);
    offset.factor.setValue(-1);
    _switch.addChild(offset);

    // Disable lighting for the slice border (easier to see)
    SoLightModel lighting = new SoLightModel();
    lighting.model.setValue(SoLightModel.Models.BASE_COLOR);
    _switch.addChild(lighting);

    // We don't want the border to be pickable, only the slice.
    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
    _switch.addChild(pickStyle);

    // We use a polygon to draw the border.
    // This allows us apply polygonOffset to it, but we need to set the
    // drawStyle
    // so it looks like a polyline. We also have to call ignore() on the other
    // fields so the application can modify them.
    SoDrawStyle style = new SoDrawStyle();
    style.style.setValue(SoDrawStyle.Styles.LINES);
    style.lineWidth.setValue(2);
    style.linePattern.setIgnored(true);
    style.linePatternScaleFactor.setIgnored(true);
    _switch.addChild(style);

    // Start with no vertices but default color set
    SoVertexProperty vprop = new SoVertexProperty();
    vprop.orderedRGBA.setValue(borderColor.getValue().getPackedValue());

    // The geometry
    _geometry = new SoFaceSet();
    _geometry.vertexProperty.setValue(vprop);
    _switch.addChild(_geometry);
  }

  private SoSeparator _scene; // Internal scene graph
  private SoSwitch _switch; // Turns border on/off
  private SoFaceSet _geometry; // Actual border geometry
  private SbBox3f _volumeExtent; // Last used volume extent
  private int _sliceAxis; // Last used slice axis
  private int _sliceNumber; // Last used slice number

}
