///////////////////////////////////////////////////////////////////////////////
//
// This code is part of the Open Inventor Medical Edition utility library.
//
// Open Inventor customers may use this source code to create or enhance Open
// Inventor-based applications.
//
// The medical utility classes are provided as a jar named com.openinventor.medical.jar,
// that can be used directly in an Open Inventor application. The classes in this jar
// are documented and supported by FEI. These classes are also provided as source code.
//
///////////////////////////////////////////////////////////////////////////////
package com.openinventor.medical.nodes;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbColorRGBA;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.actions.SoAction;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.actions.SoGetBoundingBoxAction;
import com.openinventor.inventor.fields.SoField;
import com.openinventor.inventor.fields.SoSFFloat;
import com.openinventor.inventor.fields.SoSFString;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoFieldSensor;
import com.openinventor.inventor.sensors.SoNodeSensor;
import com.openinventor.volumeviz.details.SoOrthoSliceDetail;

//@formatter:off
/**
 * <strong>(Preview Feature) </strong>Interactive distance measuring tool for SoOrthoSlice.
 *
 * <p style="background-color: #c7c7c7;">Preview Feature means this class is fully supported and can be used in Open Inventor applications.
 * Being tagged as a Preview Feature just means that the implementation is still
 * subject to API changes and adjustments based on feedback from early adopters.
 * Please be also aware that source compatibility might be broken regardless of
 * the Open Inventor compatibility changes policy due to our commitment to bring
 * needed changes to be sure the specifications of this Preview Feature match
 * the expectations of our customers.
 * </p>
 *
 * <p>
 * This node displays an interactive distance measuring tool on top of an
 * SoOrthoSlice. The user can click and drag to display a line and the physical
 * distance between the clicked point and the current cursor position.
 * <p>
 * This node is only intended to be used in a "2D" viewing environment using an
 * SoOrthographicCamera, for example viewing a single slice of a medical volume.
 * <p>
 * This node requires that the application handle mouse press, release and move
 * events. Mouse move events must be forwarded to the node using the
 * manageMouseMove method.
 * <p>
 * <b>File format/default:</b>
 * <p>
 * Ruler {
 * <ul><table>
 * <tr><td> globalFactor  </td><td> 1         </td></tr>
 * <tr><td> label         </td><td> mm        </td></tr>
 * </table></ul> }
 *
 * @see DicomInfo
 * @see Magnifier
 * @see Gnomon
 *
 */
//@formatter:on
public class Ruler extends SoNode
{

  /**
   * Specifies a scale factor applied to the size of the arrowheads on the line.
   * Default is 1.
   */
  public SoSFFloat globalFactor;

  /**
   * Specifies a label string appended to the measurement value. Default is
   * "mm".
   */
  public SoSFString label;

  /**
   * Override doAction function
   *
   * @param action
   *          Generic Inventor action
   */
  @Override
  public void doAction(SoAction action)
  {
    // traverse our internal scene graph
    action.forwardTraversal(_root);
  }

  /**
   * Override GLRender function
   *
   * @param action
   *          GL Render action
   */
  @Override
  public void GLRender(SoGLRenderAction action)
  {
    doAction(action);
  }

  /**
   * Override getBoundingBox function
   *
   * @param action
   *          GetBoundingBox action
   */
  @Override
  public void getBoundingBox(SoGetBoundingBoxAction action)
  {
    doAction(action);
  }

  /**
   * The application should call this method with 'true' on the mouse down event
   * that begins a measure operation, then call this method with 'false' on each
   * mouse move event during the measure operation. See the medicalRuler example
   * program for more details.
   */
  public void manageMouseMove(boolean firstClick, SoOrthoSliceDetail detail)
  {
    SbVec3f newPos;
    if ( detail != null )
      newPos = detail.getValueObjectPos();
    else
      newPos = initPos;

    _root.enableNotify(false);
    if ( firstClick )
    {
      initPos = newPos;
      _rulerVertexProp.vertex.set1Value(0, initPos);
      _leftConeTransform.translation.setValue(initPos);
      _dStyle.style.setValue(SoDrawStyle.Styles.INVISIBLE);

      _ruler.numVertices.setValue(1);
    }
    else
    {
      _rulerVertexProp.vertex.set1Value(1, newPos);
      _rightConeTransform.translation.setValue(newPos);
      _ruler.numVertices.setValue(2);
      _dStyle.style.setValue(SoDrawStyle.Styles.FILLED);
    }

    // Test to orient cones in function of the distance between 2 selected
    // points. To avoid cone overlapping.
    SbVec3f delta = initPos.minus(newPos);
    float dist = delta.length();

    if ( dist < globalFactor.getValue() * 5 )
    {
      _leftConeTransform.rotation.setValue(new SbRotation(new SbVec3f(0, -1, 0), delta));
      _rightConeTransform.rotation.setValue(new SbRotation(new SbVec3f(0, 1, 0), delta));
    }
    else
    {
      _leftConeTransform.rotation.setValue(new SbRotation(new SbVec3f(0, 1, 0), delta));
      _rightConeTransform.rotation.setValue(new SbRotation(new SbVec3f(0, -1, 0), delta));
    }

    _rulerAnnotation.string.setValue(String.valueOf(dist) + " " + label.getValue());

    delta.divide(3);
    _annotationTransform.translation.setValue(initPos.minus(delta));

    _root.enableNotify(true);
    _root.touch();
  }

  /**
   * Constructor
   */
  public Ruler()
  {
    globalFactor = new SoSFFloat(this, "globalFactor", SoField.FieldTypes.EXPOSED_FIELD, 1.0f);
    label = new SoSFString(this, "label", SoField.FieldTypes.EXPOSED_FIELD, "mm");
    initPos = new SbVec3f(0.0f, 0.0f, 0.0f);

    setName("Ruler");
    buildSceneGraph();
  }

  /**
   * Access the font property
   */
  public SoFont getFont()
  {
    return _annotationFont;
  }

  /**
   * Builds the scene graph that represents a ruler object.
   */
  private void buildSceneGraph()
  {
    _root = new SoSeparator();
    // Ruler = 2 Cones + 1 Lineset + 1 3d-text
    // Draw Style
    _dStyle = new SoDrawStyle();
    _dStyle.style.setValue(SoDrawStyle.Styles.INVISIBLE);
    _dStyle.lineWidth.setValue(2);
    _root.addChild(_dStyle);

    // Pick Style
    SoPickStyle rulerPickStyle = new SoPickStyle();
    rulerPickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
    _root.addChild(rulerPickStyle);

    // Shape hints
    SoShapeHints hints = new SoShapeHints();
    hints.vertexOrdering.setValue(SoShapeHints.VertexOrderings.COUNTERCLOCKWISE);
    _root.addChild(hints);

    // SoMaterial
    SoMaterial rulerAnnotationMat = new SoMaterial();
    rulerAnnotationMat.diffuseColor.setValue(new SbColor(1, 0, 0));
    _root.addChild(rulerAnnotationMat);

    //////////////////////////////////////////////////////////////////////////////
    // Left Cone
    SoSeparator leftConeSep = new SoSeparator();
    _root.addChild(leftConeSep);
    _leftConeTransform = new SoTransform();
    leftConeSep.addChild(_leftConeTransform);

    _initialLeftTranslation = new SoTranslation();
    _initialLeftTranslation.translation.setValue(new SbVec3f(0, -globalFactor.getValue(), 0));
    leftConeSep.addChild(_initialLeftTranslation);

    _leftCone = new SoCone();
    _leftCone.bottomRadius.setValue(globalFactor.getValue());
    _leftCone.height.setValue(2 * globalFactor.getValue());
    leftConeSep.addChild(_leftCone);

    //////////////////////////////////////////////////////////////////////////////
    // Right Cone
    SoSeparator rightConeSep = new SoSeparator();
    _root.addChild(rightConeSep);
    _rightConeTransform = new SoTransform();
    rightConeSep.addChild(_rightConeTransform);

    _initialRightTranslation = new SoTranslation();
    _initialRightTranslation.translation.setValue(new SbVec3f(0, -globalFactor.getValue(), 0));
    rightConeSep.addChild(_initialRightTranslation);

    _rightCone = new SoCone();
    _rightCone.bottomRadius.setValue(globalFactor.getValue());
    _rightCone.height.setValue(2 * globalFactor.getValue());
    rightConeSep.addChild(_rightCone);

    //////////////////////////////////////////////////////////////////////////////
    // Ruler line
    _rulerVertexProp = new SoVertexProperty();
    _ruler = new SoLineSet();
    _ruler.vertexProperty.setValue(_rulerVertexProp);
    _root.addChild(_ruler);

    //////////////////////////////////////////////////////////////////////////////
    // Ruler annotation
    SoAnnotation rulerAnnotationSep = new SoAnnotation();
    _root.addChild(rulerAnnotationSep);

    _annotationTransform = new SoTransform();
    rulerAnnotationSep.addChild(_annotationTransform);

    SoMaterial textMatl = new SoMaterial();
    textMatl.diffuseColor.setValue(0.1f, 0.1f, 0.1f);
    rulerAnnotationSep.addChild(textMatl);

    SoTextProperty textProp = new SoTextProperty();
    textProp.alignmentV.setValue(SoTextProperty.AlignmentVs.BOTTOM);
    textProp.style.setValue(SoTextProperty.Styles.BACK_FRAME);
    textProp.styleColors.set1Value(SoTextProperty.StyleColorType.BACK_FRAME_COLOR.getValue(),
        new SbColorRGBA(1, 1, 1, 0.5f));
    rulerAnnotationSep.addChild(textProp);

    // Font for the annotation.
    _annotationFont = new SoFont();

    _annotationFont.size.setValue(17); // Pixels
    rulerAnnotationSep.addChild(_annotationFont);

    SoTranslation intialTranslation = new SoTranslation();
    intialTranslation.translation.setValue(new SbVec3f(0, globalFactor.getValue(), 0));
    rulerAnnotationSep.addChild(intialTranslation);

    // Text of the ruler
    _rulerAnnotation = new SoText2();
    rulerAnnotationSep.addChild(_rulerAnnotation);

    // Detect changes to "label" field
    _labelFieldSensor = new SoFieldSensor(new Runnable()
    {
      @Override
      public void run()
      {
        SbVec3f begin = _rulerVertexProp.vertex.getValueAt(0);
        SbVec3f end = _rulerVertexProp.vertex.getValueAt(1);
        SbVec3f delta = begin.minus(end);
        float dist = delta.length();

        _rulerAnnotation.string.setValue(String.valueOf(dist) + " " + label.getValue());
      }
    });
    _labelFieldSensor.setPriority(0);
    _labelFieldSensor.attach(label);

    // Detect changes to "globalFactor" field
    _globalFactorFieldSensor = new SoFieldSensor(new Runnable()
    {
      @Override
      public void run()
      {
        updateGlobalFactor();
      }
    });
    _globalFactorFieldSensor.setPriority(0);
    _globalFactorFieldSensor.attach(globalFactor);

    // Detect changes to our internal scene graph.
    // Changes occur on mouse events, see manageMouseMoved().
    _internalRootSensor = new SoNodeSensor(new Runnable()
    {
      @Override
      public void run()
      {
        // mark ruler as modified
        touch();
      }
    });
    _internalRootSensor.setPriority(0);
    _internalRootSensor.attach(_root);
  }

  /**
   * Global Factor management
   */
  private void updateGlobalFactor()
  {
    // Update all node using globalFactor value
    _initialLeftTranslation.translation.setValue(new SbVec3f(0, -globalFactor.getValue(), 0));
    _initialRightTranslation.translation.setValue(new SbVec3f(0, -globalFactor.getValue(), 0));

    _leftCone.bottomRadius.setValue(globalFactor.getValue());
    _leftCone.height.setValue(2 * globalFactor.getValue());

    _rightCone.bottomRadius.setValue(globalFactor.getValue());
    _rightCone.height.setValue(2 * globalFactor.getValue());
  }

  /// Static member variable
  private static SbVec3f initPos;

  /// Member variables
  private SoCone _leftCone;
  private SoCone _rightCone;
  private SoFont _annotationFont;
  private SoDrawStyle _dStyle;
  private SoLineSet _ruler;
  private SoText2 _rulerAnnotation;
  private SoTransform _leftConeTransform;
  private SoTransform _rightConeTransform;
  private SoTransform _annotationTransform;
  private SoTranslation _initialLeftTranslation;
  private SoTranslation _initialRightTranslation;
  private SoVertexProperty _rulerVertexProp;
  private SoSeparator _root;

  private SoFieldSensor _labelFieldSensor;
  private SoFieldSensor _globalFactorFieldSensor;
  private SoNodeSensor _internalRootSensor;
}
