///////////////////////////////////////////////////////////////////////////////
//
// This code is part of the Open Inventor Medical Edition utility library.
//
// Open Inventor customers may use this source code to create or enhance Open
// Inventor-based applications.
//
// The medical utility classes are provided as a jar named com.openinventor.medical.jar,
// that can be used directly in an Open Inventor application. The classes in this jar
// are documented and supported by FEI. These classes are also provided as source code.
//
///////////////////////////////////////////////////////////////////////////////
package com.openinventor.medical.nodes;

import com.openinventor.inventor.fields.SoField;
import com.openinventor.inventor.fields.SoSFEnum;
import com.openinventor.inventor.fields.SoSFFloat;
import com.openinventor.inventor.fields.SoSFString;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoNodeSensor;
import com.openinventor.medical.helpers.MedicalHelper;

//@formatter:off
/**
 * <strong>(Preview Feature) </strong>Shape node to display slice orientation markers in window coordinates.
 *
 * <p style="background-color: #c7c7c7;">Preview Feature means this class is fully supported and can be used in Open Inventor applications.
 * Being tagged as a Preview Feature just means that the implementation is still
 * subject to API changes and adjustments based on feedback from early adopters.
 * Please be also aware that source compatibility might be broken regardless of
 * the Open Inventor compatibility changes policy due to our commitment to bring
 * needed changes to be sure the specifications of this Preview Feature match
 * the expectations of our customers.
 * </p>
 *
 * This node displays slice orientation markers at a fixed location in the
 * viewer window. Slice orientation markers are useful when viewing images in a
 * "2D" mode. The orientation markers are a set of four letters. Each letter is
 * displayed at the mid-point of one edge of the window and offset inward by the
 * value specified in the {@link #offset} field. The letters displayed depend on the
 * value of the {@link #axis} field and correspond to the DICOM radiological viewing
 * convention. This node is intended to be used with SoOrthoSlice and a "plane"
 * viewer.
 * <p>
 * The application can connect the {@link #axis} field from the 'axis' field of the
 * SoOrthoSlice node and the correct orientation markers will be displayed
 * automatically if the slice axis is changed.
 * <p>
 * For:
 * <ul>
 * <li> AXIAL display
 * <ul>
 *   <li> Left/Right: R(ight) / L(eft)</li>
 *   <li> Top/Bottom: A(nterior) / P(osterior) </li>
 * </ul></li>
 * <p>
 * <li>CORONAL display
 * <ul>
 *   <li>Left/Right: R(ight) / L(eft) </li>
 *   <li>Top/Bottom: S(uperior) / I(nferior)</li>
 * </ul></li>
 * <p>
 * <li>SAGITTAL display
 * <ul>
 *   <li>Left/Right: A(nterior) / P(osterior) </li>
 *   <li> Top/Bottom: S(uperior) / I(nferior) </li>
 * </ul></li>
 * </ul>
 *
 * <p>
 * <b>File format/default:</b>
 * <p>
 * SliceOrientationMarkers {
 * <ul><table>
 * <tr><td> offset    </td><td> 0.05         </td></tr>
 * <tr><td> axis      </td><td> AXIAL        </td></tr>
 * <tr><td> fontName  </td><td> Arial:Bold   </td></tr>
 * <tr><td> fontSize  </td><td> 17           </td></tr>
 * </table></ul> }
 * <p>
 *
 * @see DicomInfo
 * @see Magnifier
 * @see Gnomon
 * @see SliceScaleBar
 * @see TextBox
 */
//@formatter:on
public class SliceOrientationMarkers extends SoAnnotation
{
  /**
   * Axis (default is AXIAL). Use enum MedicalHelper.Axis. Determines which
   * letters are displayed.
   */
  public SoSFEnum<MedicalHelper.Axis> axis;

  /**
   * Distance from edge of window to letter. Specified in normalized screen
   * coordinates (-1 to 1). Default is 0.05.
   */
  public SoSFFloat offset;

  /**
   * Specify the font name (default is "Arial:Bold"). See SoFont.name for
   * details.
   */
  public SoSFString fontName;

  /**
   * Specify the font size in pixels (default is 17).
   */
  public SoSFFloat fontSize;

  /**
   * Constructor
   */
  public SliceOrientationMarkers()
  {
    // Initialize fields
    offset = new SoSFFloat(this, "offset", SoField.FieldTypes.EXPOSED_FIELD, 0.05f);
    fontName = new SoSFString(this, "fontName", SoField.FieldTypes.EXPOSED_FIELD, "Arial:Bold");
    fontSize = new SoSFFloat(this, "fontSize", SoField.FieldTypes.EXPOSED_FIELD, 17);
    axis = new SoSFEnum<MedicalHelper.Axis>(this, "axis", SoField.FieldTypes.EXPOSED_FIELD, MedicalHelper.Axis.class,
        MedicalHelper.Axis.AXIAL);

    // Initialize member vars
    _fontNode = new SoFont();
    _textNodes = new SoText2[4];
    _tranNodes = new SoTranslation[4];
    for ( int i = 0; i < 4; ++i )
    {
      _textNodes[i] = new SoText2();
      _tranNodes[i] = new SoTranslation();
    }

    // Create the internal scene graph
    buildSceneGraph();
  }

  /**
   * Builds the scene graph that represents a slice orientation markers object.
   *
   * Separator "AxisBoxRoot" +- BBox +- LightModel +- PickStyle +-
   * OrthographicCamera +- Font +- Separator "Left" | +- TextProperty | +-
   * Translation | +- Text2 +- Separator "Right" | +- TextProperty | +-
   * Translation | +- Text2 +- Separator "Top" | +- TextProperty | +-
   * Translation | +- Text2 +- Separator "Bottom" | +- TextProperty | +-
   * Translation | +- Text2
   */
  protected void buildSceneGraph()
  {
    setName("SliceOrientationMarkers");

    SoBBox bbox = new SoBBox();
    bbox.mode.setValue(SoBBox.Modes.NO_BOUNDING_BOX);
    addChild(bbox);

    SoLightModel lmodel = new SoLightModel();
    lmodel.model.setValue(SoLightModel.Models.BASE_COLOR);
    this.addChild(lmodel);

    SoPickStyle pstyle = new SoPickStyle();
    pstyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
    this.addChild(pstyle);

    SoOrthographicCamera camera = new SoOrthographicCamera();
    camera.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);
    this.addChild(camera);

    _fontNode = new SoFont();
    _fontNode.size.connectFrom(this.fontSize);
    _fontNode.name.connectFrom(this.fontName);
    _fontNode.renderStyle.setValue(SoFont.RenderStyles.TEXTURE);
    this.addChild(_fontNode);

    // Create text nodes.
    // Need separator for each to keep translations separate.
    // Each has its own position and alignment settings.
    for ( int i = 0; i < 4; ++i )
    {
      SoSeparator tempSep = new SoSeparator();
      this.addChild(tempSep);

      SoTextProperty textProp = new SoTextProperty();
      // Allow app to control these fields
      textProp.backFrameLineWidth.setIgnored(true);
      textProp.margin.setIgnored(true);
      textProp.style.setIgnored(true);
      textProp.styleColors.setIgnored(true);
      tempSep.addChild(textProp);
      switch ( i )
      {
      case 0 : // Left
        textProp.alignmentH.setValue(SoTextProperty.AlignmentHs.LEFT);
        textProp.alignmentV.setValue(SoTextProperty.AlignmentVs.HALF);
        break;
      case 1 : // Right
        textProp.alignmentH.setValue(SoTextProperty.AlignmentHs.RIGHT);
        textProp.alignmentV.setValue(SoTextProperty.AlignmentVs.HALF);
        break;
      case 2 : // Top
        textProp.alignmentH.setValue(SoTextProperty.AlignmentHs.CENTER);
        textProp.alignmentV.setValue(SoTextProperty.AlignmentVs.TOP);
        break;
      case 3 : // Bottom
        textProp.alignmentH.setValue(SoTextProperty.AlignmentHs.CENTER);
        textProp.alignmentV.setValue(SoTextProperty.AlignmentVs.BASE);
        break;
      }

      _tranNodes[i] = new SoTranslation();
      tempSep.addChild(_tranNodes[i]);

      _textNodes[i] = new SoText2();
      _textNodes[i].justification.setValue(SoText2.Justifications.INHERITED);
      tempSep.addChild(_textNodes[i]);
    }

    // Text positions
    float offsetVal = offset.getValue();
    _tranNodes[0].translation.setValue(offsetVal - 1, 0, 0); // Left
    _tranNodes[1].translation.setValue(1 - offsetVal, 0, 0); // Right
    _tranNodes[2].translation.setValue(0, 1 - offsetVal, 0); // Top
    _tranNodes[3].translation.setValue(0, offsetVal - 1, 0); // Bottom

    // Initialize letters
    updateLetters();

    // Detect changes to our fields or children
    _nodeSensor = new SoNodeSensor();
    _nodeSensor.setTask(new NodeSensorCB());
    _nodeSensor.setPriority(0);
    _nodeSensor.attach(this);
  }

  /**
   * Update letters when axis changes
   */
  protected void updateLetters()
  {
    if ( axis.getValue() == MedicalHelper.Axis.AXIAL.getValue() )
    {
      _textNodes[0].string.setValue("R"); // Left
      _textNodes[1].string.setValue("L"); // Right
      _textNodes[2].string.setValue("A"); // Top
      _textNodes[3].string.setValue("P"); // Bottom
    }
    else if ( axis.getValue() == MedicalHelper.Axis.CORONAL.getValue() )
    {
      _textNodes[0].string.setValue("R");
      _textNodes[1].string.setValue("L");
      _textNodes[2].string.setValue("S");
      _textNodes[3].string.setValue("I");
    }
    else
    {
      _textNodes[0].string.setValue("A");
      _textNodes[1].string.setValue("P");
      _textNodes[2].string.setValue("S");
      _textNodes[3].string.setValue("I");
    }
  }

  // Members
  protected SoFont _fontNode;
  protected SoText2[] _textNodes;
  protected SoTranslation[] _tranNodes;
  protected SoNodeSensor _nodeSensor;

  /**
   * Called when one of our monitored fields has changed. This is a "shotgun"
   * approach, but we wanted to allow for the possibility of the application
   * changing, for example, the font name or size.
   *
   * @param sensor
   */
  private class NodeSensorCB implements Runnable
  {
    @Override
    public void run()
    {
      // Update positions and letters.
      // Disable notification so this sensor doesn't get called again!
      float offsetVal = offset.getValue();
      enableNotify(false);
      _tranNodes[0].translation.setValue(offsetVal - 1, 0, 0); // Left
      _tranNodes[1].translation.setValue(1 - offsetVal, 0, 0); // Right
      _tranNodes[2].translation.setValue(0, 1 - offsetVal, 0); // Top
      _tranNodes[3].translation.setValue(0, offsetVal - 1, 0); // Bottom
      updateLetters();
      enableNotify(true);
    }
  }
}
