///////////////////////////////////////////////////////////////////////////////
//
// This code is part of the Open Inventor Medical Edition utility library.
//
// Open Inventor customers may use this source code to create or enhance Open
// Inventor-based applications.
//
// The medical utility classes are provided as a jar named com.openinventor.medical.jar,
// that can be used directly in an Open Inventor application. The classes in this jar
// are documented and supported by FEI. These classes are also provided as source code.
//
///////////////////////////////////////////////////////////////////////////////
package com.openinventor.medical.nodes;

import com.openinventor.inventor.IntegerValuedEnum;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewVolume;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.actions.SoAction;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.actions.SoGetBoundingBoxAction;
import com.openinventor.inventor.elements.SoViewVolumeElement;
import com.openinventor.inventor.elements.SoViewportRegionElement;
import com.openinventor.inventor.fields.SoField;
import com.openinventor.inventor.fields.SoSFEnum;
import com.openinventor.inventor.fields.SoSFFloat;
import com.openinventor.inventor.fields.SoSFInt32;
import com.openinventor.inventor.fields.SoSFNode;
import com.openinventor.inventor.fields.SoSFString;
import com.openinventor.inventor.fields.SoSFVec2f;
import com.openinventor.inventor.misc.SoState;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoFieldSensor;

//@formatter:off
/**
 * <strong>(Preview Feature) </strong>Shape node to display a dynamic scale bar in window coordinates.
 *
 * <p style="background-color: #c7c7c7;">Preview Feature means this class is fully supported and can be used in Open Inventor applications.
 * Being tagged as a Preview Feature just means that the implementation is still
 * subject to API changes and adjustments based on feedback from early adopters.
 * Please be also aware that source compatibility might be broken regardless of
 * the Open Inventor compatibility changes policy due to our commitment to bring
 * needed changes to be sure the specifications of this Preview Feature match
 * the expectations of our customers.
 * </p>
 *
 * <p>
 * This class displays a 2D scale bar. The {@link #position} of the scale bar is
 * specified in normalized screen coordinates (-1 to 1). By default, the
 * {@link #length} of the scale bar is a fixed distance in normalized screen coordinates.
 * However if the {@link #trackedCamera} field is set, then {@link #length} is a distance in 3D
 * world coordinates and the the size of the scale bar on screen is computed
 * based on that distance. For example, if the application is viewing DICOM data
 * measured in millimeters (mm), then setting {@link #length} to 100 displays a scale
 * bar that shows the distance 10 cm on screen. The scale bar size will change
 * if the camera is zoomed in or out (camera height field changes).
 * <p>
 * The distance computation is based on "horizontal" or "vertical" relative to
 * the camera, so the scale bar adjusts automatically if the camera is rotated
 * to view a different volume axis, e.g. Coronal vs Axial.
 * <p>
 * This class is not intended to replace the MeshViz axis classes for general 2D
 * data plotting. This class has only been tested with SoOrthoSlice and an
 * SoOrthographicCamera (the usual case for medical image viewing).
 *
 * <p>
 * <b>File format/default:</b>
 * <p>
 * SliceScaleBar {
 * <ul><table>
 * <tr><td> position          </td><td> 0 0 0        </td></tr>
 * <tr><td> length            </td><td> 1            </td></tr>
 * <tr><td> numTickIntervals  </td><td> 0            </td></tr>
 * <tr><td> trackedCamera     </td><td> NULL         </td></tr>
 * <tr><td> orientation       </td><td> HORIZONTAL   </td></tr>
 * <tr><td> alignment         </td><td> CENTER"      </td></tr>
 * <tr><td> label             </td><td> ""           </td></tr>
 * </table></ul> }
 *
 * @see DicomInfo
 * @see Magnifier
 * @see Gnomon
 * @see TextBox
 * @see Ruler
 * @see SliceOrientationMarkers
 */
//@formatter:on
public class SliceScaleBar extends SoAnnotation
{
  /**
   * Position in normalized screen coordinates (-1 to 1). Default is (0,0,0).
   */
  public SoSFVec2f position;

  /**
   * Label (default is empty string).
   */
  public SoSFString label;

  /**
   * Length in normalized screen coordinates (-1 to 1) if not tracking, else
   * length in 3D world coordinates. Default is 1. <p> For example, if the
   * application is viewing DICOM data measured in millimeters (mm), then
   * setting length to 100 displays a 10 cm scale bar on screen.
   */
  public SoSFFloat length;

  /**
   * Number of tick intervals (default is 0). If numTickIntervals is 0, no tick
   * marks are drawn. For example, if the length is set to 100 mm, then set
   * numTickIntervals to 10 to get a tick mark every 10 mm (1 cm).
   */
  public SoSFInt32 numTickIntervals;

  /**
   * Tracked camera (default is null). This should be the camera that is viewing
   * the tracked scene. It will be used to determine the length of the axis in
   * NDC based on the specified length in 3D world coordinates.
   */
  public SoSFNode trackedCamera;

  /**
   * Scale bar orientation (default is HORIZONTAL).
   */
  public SoSFEnum<Orientation> orientation;

  /**
   * Scale bar alignment (default is CENTER).
   */
  public SoSFEnum<Alignment> alignment;

  /**
   * Scale bar orientation
   */
  public enum Orientation implements IntegerValuedEnum
  {
    /** Horizontal */
    HORIZONTAL(0),
    /** Vertical */
    VERTICAL(1);

    private Orientation(int orient)
    {
      _orient = orient;
    }

    @Override
    public int getValue()
    {
      return _orient;
    }

    private final int _orient;
  };

  /**
   * Scale bar alignment
   */
  public enum Alignment implements IntegerValuedEnum
  {
    /** Left (for horizontal orientation). */
    LEFT(0),
    /** Bottom (for vertical orientation). */
    BOTTOM(0),
    /** Center (for either orientation). */
    CENTER(1),
    /** Right (for horizontal orientation). */
    RIGHT(2),
    /** Top (for vertical orientation). */
    TOP(2);

    @Override
    public int getValue()
    {
      return _align;
    }

    private Alignment(int align)
    {
      _align = align;
    }

    private final int _align;
  };

  /**
   * Constructor
   */
  public SliceScaleBar()
  {
    position = new SoSFVec2f(this, "position", SoField.FieldTypes.EXPOSED_FIELD, new SbVec2f(0, 0));
    label = new SoSFString(this, "label", SoField.FieldTypes.EXPOSED_FIELD, "");
    length = new SoSFFloat(this, "length", SoField.FieldTypes.EXPOSED_FIELD, 1.0f);
    numTickIntervals = new SoSFInt32(this, "numTickIntervals", SoField.FieldTypes.EXPOSED_FIELD, 10);
    trackedCamera = new SoSFNode(this, "camera", SoField.FieldTypes.EXPOSED_FIELD, null);
    orientation = new SoSFEnum<Orientation>(this, "orientation", SoField.FieldTypes.EXPOSED_FIELD, Orientation.class,
        Orientation.HORIZONTAL);
    alignment =
        new SoSFEnum<Alignment>(this, "alignment", SoField.FieldTypes.EXPOSED_FIELD, Alignment.class, Alignment.CENTER);

    // Note: _ndcLength must be set before calling BuildSceneGraph().
    _cameraChanged = false;
    _ndcLength = length.getValue();

    // Window size in pixels (approximate until first traversal)
    _winSizePix = new SbVec2i32(500, 500);
    // Conversion from 1 ndc unit to pixels
    _pixelPerNdc = new SbVec2f(0.5f * _winSizePix.getX(), 0.5f * _winSizePix.getY());

    // Tick line length in pixels
    // Tick line length in NDC (depends on viewport so updated later)
    _tickLenPix = 5;
    _tickLenNdc = _tickLenPix / _pixelPerNdc.getX();

    // Create the internal scene graph
    buildSceneGraph();

    // Monitor changes to the tracked camera (if any)
    _cameraFieldSensor = new SoFieldSensor();
    _cameraFieldSensor.setTask(new FieldSensorCB(_cameraFieldSensor));
    _cameraFieldSensor.attach(trackedCamera);
  }

  private class FieldSensorCB implements Runnable
  {
    public FieldSensorCB(SoFieldSensor sensor)
    {
      _sensor = sensor;
    }

    @Override
    public void run()
    {
      SoField attached = _sensor.getAttachedField();
      if ( attached == trackedCamera )
        _cameraChanged = true;
    }

    private SoFieldSensor _sensor;
  }

  /**
   * Builds the scene graph that represents a slice scalar bar object.
   */
  protected void buildSceneGraph()
  {
    setName("SliceScaleBar");

    SoCallback callbackNode = new SoCallback();
    callbackNode.setCallback(new SoCallback.CB()
    {
      @Override
      public void invoke(SoAction action)
      {
        if ( action instanceof SoGLRenderAction || action instanceof SoGetBoundingBoxAction )
          renderCB(action);
      }
    });
    addChild(callbackNode);

    SoBBox bbox = new SoBBox();
    bbox.mode.setValue(SoBBox.Modes.NO_BOUNDING_BOX);
    addChild(bbox);

    SoLightModel lmodel = new SoLightModel();
    lmodel.model.setValue(SoLightModel.Models.BASE_COLOR);
    addChild(lmodel);

    SoPickStyle pstyle = new SoPickStyle();
    pstyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
    addChild(pstyle);

    SoOrthographicCamera camera = new SoOrthographicCamera();
    camera.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);
    addChild(camera);

    // Lines -------------------------------------------------------------
    _lineSep = new SoSeparator();
    addChild(_lineSep);

    // Initial geometry
    _vertProp = new SoVertexProperty();
    computeEndPoints();
    _vertProp.vertex.set1Value(0, _p0);
    _vertProp.vertex.set1Value(1, _p1);

    _axisLineSet = new SoLineSet();
    _axisLineSet.vertexProperty.setValue(_vertProp);
    _lineSep.addChild(_axisLineSet);

    // Text -------------------------------------------------------------
    _textSep = new SoSeparator();
    addChild(_textSep);

    _labelPos = new SoTranslation();
    _textSep.addChild(_labelPos);

    _labelFont = new SoFont();
    _labelFont.size.setValue(15);
    _labelFont.name.setValue("Arial:Bold");
    _labelFont.renderStyle.setValue(SoFont.RenderStyles.TEXTURE);
    _textSep.addChild(_labelFont);

    SoTextProperty textProp = new SoTextProperty();
    if ( orientation.getValue() == Orientation.HORIZONTAL.getValue() )
    {
      textProp.alignmentH.setValue(SoTextProperty.AlignmentHs.LEFT);
      textProp.alignmentV.setValue(SoTextProperty.AlignmentVs.BASE);
    }
    else
    {
      textProp.alignmentH.setValue(SoTextProperty.AlignmentHs.LEFT);
      textProp.alignmentV.setValue(SoTextProperty.AlignmentVs.TOP);
    }
    _textSep.addChild(textProp);

    _labelText = new SoText2();
    _labelText.justification.setValue(SoText2.Justifications.INHERITED);
    _labelText.string.connectFrom(label); // Text will auto update
    _textSep.addChild(_labelText);
  }

  /**
   * Compute new end-points of scale bar based on orientation and alignment.
   * Must use the member variable _ndcLength, not the length field, in case
   * actual length is tracking a 3D distance.
   */
  protected void computeEndPoints()
  {
    float xoffset = 0;
    float yoffset = 0;
    float xlength = 0;
    float ylength = 0;
    if ( orientation.getValue() == Orientation.HORIZONTAL.getValue() )
    {
      xlength = _ndcLength;
      if ( alignment.getValue() == Alignment.CENTER.getValue() )
        xoffset -= _ndcLength / 2;
      else if ( alignment.getValue() == Alignment.RIGHT.getValue() )
        xoffset -= _ndcLength;
    }
    else
    { // VERTICAL
      ylength = _ndcLength;
      if ( alignment.getValue() == Alignment.CENTER.getValue() )
        yoffset -= _ndcLength / 2;
      else if ( alignment.getValue() == Alignment.TOP.getValue() )
        yoffset -= _ndcLength;
    }
    float x = position.getValue().getX();
    float y = position.getValue().getY();
    x += xoffset;
    y += yoffset;
    _p0 = new SbVec3f(x, y, 0);
    _p1 = new SbVec3f(x + xlength, y + ylength, 0);
  }

  protected SoSeparator _lineSep;
  protected SoLineSet _axisLineSet;
  protected SoVertexProperty _vertProp;

  protected SoSeparator _textSep;
  protected SoTranslation _labelPos;
  protected SoFont _labelFont;
  protected SoText2 _labelText;

  protected boolean _cameraChanged; // Set when we need to recompute
  protected float _ndcLength; // Same as length field if not tracked.
  protected SbVec3f _p0; // Current end-points in NDC
  protected SbVec3f _p1; // (they depend on orientation and alignment)

  protected int _tickLenPix; // Target tick line length in pixels
  protected float _tickLenNdc; // Tick line length in NDC (depends on viewport)
  protected SbVec2f _pixelPerNdc; // Converts 1 ndc unit to pixels (but not a
                                  // coordinate)
  protected SbVec2i32 _winSizePix; // Window size in pixels

  protected SoFieldSensor _cameraFieldSensor; // Monitors tracked camera node

  /**
   * Call from StaticCB on render action traversal
   */
  protected void renderCB(SoAction action)
  {
    boolean mustUpdateAxis = false;

    // Get values from state we need
    // (creates a dependency on these elements in the state)
    SoState state = action.getState();
    SbViewportRegion vpRegion = SoViewportRegionElement.get(state);
    SbViewVolume viewVol = SoViewVolumeElement.get(state);

    SbVec2i32 winSize = vpRegion.getViewportSizePixelsi32();

    // Unsafe to continue if current viewport has not been set yet.
    if ( winSize.getX() <= 0 || winSize.getY() <= 0 )
      return;

    if ( (winSize.getX() != _winSizePix.getX()) || (winSize.getY() != _winSizePix.getY()) )
    {
      mustUpdateAxis = true;
      _winSizePix = winSize;
      _pixelPerNdc.setValue(0.5f * _winSizePix.getX(), 0.5f * _winSizePix.getY());

      // Tick line length in NDC.
      // This must be computed orthogonal to the axis orientation.
      int horizOrVert = 1 - orientation.getValue();
      _tickLenNdc = _tickLenPix / _pixelPerNdc.getValueAt(horizOrVert);
    }

    // If tracked camera changed, we need to recompute length in NDC.
    if ( _cameraChanged )
    {
      _cameraChanged = false;
      mustUpdateAxis = true;

      // First project the NDC mid-point of the scale bar back to 3D space.
      SbVec3f tmp = _p1.minus(_p0);
      tmp.divide(2);
      SbVec3f ndcPt = _p0.plus(tmp);
      SbVec3f xyzPt = viewVol.projectFromScreen(ndcPt);

      // What direction is camera "horizontal" (or "vertical") in the current
      // view?
      // With a little luck we don't need to know what the camera orientation
      // "means".
      // We just need the xyz points +/- half the length from the projected
      // point.
      SoCamera camera = (SoCamera) trackedCamera.getValue();
      float len = length.getValue(); // Tracked length
      SbVec3f plus, minus;

      SbRotation camOrient = camera.orientation.getValue();
      if ( orientation.getValue() == Orientation.HORIZONTAL.getValue() )
      {
        plus = camOrient.multVec(new SbVec3f(len / 2, 0, 0));
        minus = camOrient.multVec(new SbVec3f(-len / 2, 0, 0));
      }
      else // VERTICAL
      {
        plus = camOrient.multVec(new SbVec3f(0, len / 2, 0));
        minus = camOrient.multVec(new SbVec3f(0, -len / 2, 0));
      }

      SbVec3f p0xyz = xyzPt.plus(minus);
      SbVec3f p1xyz = xyzPt.plus(plus);

      // Project these points back to NDC space
      // But note that projectToScreen outputs 0..1 and we need -1..1 space.
      SbVec3f p0ndc, p1ndc;
      p0ndc = viewVol.projectToScreen(p0xyz);
      p1ndc = viewVol.projectToScreen(p1xyz);
      p0ndc.setX(2 * p0ndc.getX() - 1);
      p0ndc.setY(2 * p0ndc.getY() - 1);
      p1ndc.setX(2 * p1ndc.getX() - 1);
      p1ndc.setY(2 * p1ndc.getY() - 1);

      // Compute the new NDC length and corresponding end-points
      _ndcLength = (p1ndc.minus(p0ndc)).length();
      computeEndPoints();
    }

    if ( mustUpdateAxis )
      updateAxis();
  }

  /**
   * Clear lines and start editing
   */
  protected void resetLines()
  {
    _axisLineSet.numVertices.deleteValues(0); // SetNum() is obsolete so using
                                              // DeleteValues instead!
    _vertProp.vertex.deleteValues(0);
  }

  /**
   * Add a new line (e.g. tick) Not the most efficient implementation we can
   * imagine, but safe :)
   */
  protected void addLine(SbVec3f p0, SbVec3f p1)
  {
    int numVerts = _vertProp.vertex.getNum();
    SbVec3f[] vertexArray = new SbVec3f[] { p0, p1 };
    _vertProp.vertex.setValues(numVerts, vertexArray);

    int numLines = _axisLineSet.numVertices.getNum();
    int[] lineSetArray = new int[] { 2 };
    _axisLineSet.numVertices.setValues(numLines, lineSetArray);
  }

  /**
   * Recreate/reposition axis geometry
   */
  protected void updateAxis()
  {
    _lineSep.enableNotify(false);
    resetLines();

    // Main line
    addLine(_p0, _p1);

    // End lines ------------------------------------------------------
    // Slightly exaggerated length looks nicer?
    SbVec3f offset = new SbVec3f(0, _tickLenNdc, 0);
    if ( orientation.getValue() != Orientation.HORIZONTAL.getValue() )
      offset.setValue(_tickLenNdc, 0, 0);

    SbVec3f p2, p3;
    p2 = _p0.plus(offset.times(1.25f));
    addLine(_p0, p2);
    p2 = _p1.plus(offset.times(1.25f));
    addLine(_p1, p2);

    // Tick marks -----------------------------------------------------
    int numTickInt = numTickIntervals.getValue();
    if ( numTickInt > 0 )
    {
      float dist = _ndcLength / numTickInt;
      SbVec3f tickInterval = new SbVec3f(dist, 0, 0);
      if ( orientation.getValue() != Orientation.HORIZONTAL.getValue() )
        tickInterval.setValue(0, dist, 0);

      for ( int i = 1; i < numTickInt; ++i )
      {
        p2 = _p0.plus(tickInterval.times(i));
        p3 = p2.plus(offset);
        addLine(p2, p3);
      }
    }
    _lineSep.enableNotify(true);

    // Label ----------------------------------------------------------
    SbVec3f labelPos;
    if ( orientation.getValue() == Orientation.HORIZONTAL.getValue() )
    {
      labelPos = _p1;
      labelPos.setX(labelPos.getX() + _tickLenNdc);
    }
    else
    { // VERTICAL
      float textHeight = _labelFont.size.getValue() / _pixelPerNdc.getY();
      // Cvt pixel size to ndc
      labelPos = _p0;
      labelPos.setY(labelPos.getY() - (textHeight + _tickLenNdc));
    }
    _textSep.enableNotify(false);
    _labelPos.translation.setValue(labelPos);
    _textSep.enableNotify(true);
  }
}
