#include <iostream>
#include <chrono>
#include <thread>
#include <atomic>
#include <csignal>
#include "RemoteViz/Rendering/Service.h"
#include "RemoteViz/Rendering/ServiceSettings.h"
#include "RemoteViz/Rendering/Monitoring.h"

#include "MonitoredRenderingServiceListener.h"
#include "CSVMetricsListener.h"


using namespace RemoteViz::Rendering;

std::atomic<bool> running; 

void sighandler(int /*sig*/)
{
  running = false;
}

int main(int argc, char** argv)
{
  // Default value for service IP address
  std::string serviceIPaddress = "127.0.0.1";
  // Default value for service port
  unsigned short servicePort = 8080;

  // Instantiate a service settings class
  auto settings = std::make_shared<ServiceSettings>();

  // Use passed values for the service IP address and port
  if (argc == 3)
  {
   serviceIPaddress = argv[1];
   servicePort = atoi(argv[2]);
  }

  // Set the IP address used by the service
  settings->setIP(serviceIPaddress);
  // Set the port used by the service
  settings->setPort(servicePort);
  // Instantiate a metricsListener class to monitor the service
  auto metricsListener = std::make_shared<CSVMetricsListener>();
  // Enable all metrics
  settings->getMonitoring()->enableMetrics(Monitoring::ALL);
  settings->getMonitoring()->addListener(metricsListener);

  // Instantiate a serviceListener class to manage the service events.
  auto serviceListener = std::make_shared<MonitoredRenderingServiceListener>();
  // Add the serviceListener class as listener
  Service::instance()->addListener(serviceListener);

  // To use an Open Inventor extension with RemoteViz, uncomment the following line and set the extension name.
  //settings.setUsedExtensions(ServiceSettings::VOLUMEVIZLDM);

  // Open the service by using the settings
  if (Service::instance()->open(settings))
  {
    std::cout << "IP : " << settings->getIP() << std::endl;

    std::cout << "Hostname : " << settings->getHostname() << std::endl;

    std::cout << "Port : " << settings->getPort() << std::endl;

    std::cout << "The CSV Monitored Rendering Service is running. Press Ctrl + C to save monitoring data in monitoring.csv and to stop." << std::endl;

    running = true;

    // Register the signal handler to stop the main loop
    std::signal(SIGABRT, &sighandler);
    std::signal(SIGTERM, &sighandler);
    std::signal(SIGINT, &sighandler);
    #ifdef SIGBREAK
      std::signal(SIGBREAK, &sighandler);
    #endif

    // Main loop
    while (running)
    {
      Service::instance()->dispatch();
      std::this_thread::sleep_for(std::chrono::milliseconds(1));
    }

    // Save metrics
    metricsListener->saveMetrics();
    // Close the service
    Service::instance()->close();

  }
  else
  {
    std::cout << "Error occurred during service initialization." << std::endl;
  }

  return 0;
}

