#pragma once

#include <fstream>
#include <sstream>
#include <memory>

#include <InfluxdbMetricsDispatcher.h>

#include <RemoteViz/Rendering/MetricsListener.h>

class InfluxdbMetricsListener : public RemoteViz::Rendering::MetricsListener
{
public:

  /**
  * Constructor
  */
  InfluxdbMetricsListener();

  /**
  *  Triggered when a new measure of network latency is available.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NETWORK_LATENCY.
  *
  *  \param time : network latency in milliseconds
  *
  *  \param client : The network latency is related to this client.
  */
  virtual void onMeasuredNetworkLatency(unsigned int time, std::shared_ptr<RemoteViz::Rendering::Client> client) override;

  /**
  *  Triggered when a new measure of frame decoding time is available.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#DECODING_TIME.
  *
  *  \param time : frame decoding time in milliseconds
  *
  *  \param connection : The frame decoding time is related to this connection.
  */
  virtual void onMeasuredDecodingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Connection> connection ) override;

  /**
  *  Triggered when a new measure of frame rendering time is available.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#RENDERING_TIME.
  *
  *  \param time : frame rendering time in milliseconds
  *
  *  \param renderArea : The frame rendering time is related to this render area.
  */
  virtual void onMeasuredRenderingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea ) override;

  /**
  *  Triggered when a new measure of frame encoding time is available.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#ENCODING_TIME.
  *
  *  \param time : frame encoding time in milliseconds
  *
  *  \param connection : The frame encoding time is related to this connection.
  */
  virtual void onMeasuredEncodingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Connection> connection ) override;

  /**
  *  Triggered when the number of clients changes.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NUM_CLIENTS.
  *
  *  \param number : number of clients
  */
  virtual void onChangedNumClients( unsigned int number ) override;

  /**
  *  Triggered when the number of connections changes.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NUM_CONNECTIONS.
  *
  *  \param number : number of connections
  */
  virtual void onChangedNumConnections( unsigned int number ) override;

  /**
  *  Triggered when the number of renderArea changes.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NUM_RENDERAREAS.
  *
  *  \param number : number of render areas
  */
  virtual void onChangedNumRenderAreas( unsigned int number ) override;

  /**
  *  Destroy the metrics dispatcher.
  */
  void clear();

private:

  /**
  *  Send a measurement to InfluxDB using UDP.
  *  The tag system is used to store the client, connection or render area ID.
  */
  static void postMetric( uint64_t time, std::string name, long value, std::string tagName = "", std::string tagValue = "");

  /**
  *  Create a nanoseconds timestamp.
  */
  uint64_t createTimestamp() const;

  bool m_connected;
  std::unique_ptr<InfluxdbMetricsDispatcher> m_dispatcher;
};
