#include <iostream>
#include <iterator>
#include <chrono>
#include <algorithm>
#include <functional>

#include <InfluxdbMetricsListener.h>
#include <influxdb.hpp>

#include <RemoteViz/Rendering/Client.h>
#include <RemoteViz/Rendering/Connection.h>
#include <RemoteViz/Rendering/RenderArea.h>

using time_stamp = std::chrono::time_point<std::chrono::system_clock, std::chrono::nanoseconds>;

// InfluxDb Settings
// Docker machine ip
static std::string influxdb_ip = "127.0.0.1";
// Do not edit the dbname without editing config.db and dashboard.json
static std::string influxdb_dbname = "rv_monitoring";
// Do not edit the ports without editing docker-compose.yml and influxdb.conf
static int influxdb_http_port = 8086;
static int influxdb_udp_port = 4444;

//--------------------------------------------------------------------------------
/*
 * Check if the rv_monitoring database exists, creates it if not
 */
InfluxdbMetricsListener::InfluxdbMetricsListener()
{
  // Check if the database exists
  bool dbExists = false;
  std::string resp;
  influxdb_cpp::server_info si( influxdb_ip, influxdb_http_port, "" );
  int ret = influxdb_cpp::query( resp, "show databases", si );
  m_connected = ( ret == 0 );
  if (!m_connected)
  {
    std::cout << "Error (" << ret << ") getting databases list\nResp : " << resp << std::endl;
    std::cout << "Is the InfluxDb server running?" << std::endl;
    exit( EXIT_FAILURE );
  }

  // Check if the database name is found in the query result
  dbExists = ( resp.find( influxdb_dbname ) != std::string::npos );

  // Create the database if necessary
  if (!dbExists && m_connected)
  {
    std::string resp;
    ret = influxdb_cpp::create_db( resp, influxdb_dbname, si );
    m_connected = ( ret == 0 );
    if (!m_connected)
    {
      std::cout << "Error (" << ret << ") creating database : " << influxdb_dbname << "\nResp : " << resp << std::endl;
      exit( EXIT_FAILURE );
    }
  }

  if (m_connected)
    std::cout << "Connected to InfluxDB : " << influxdb_ip << ":" << influxdb_http_port << "/" << influxdb_udp_port << std::endl;

  m_dispatcher = std::unique_ptr<InfluxdbMetricsDispatcher>(new InfluxdbMetricsDispatcher());
}

//--------------------------------------------------------------------------------
/*
 * Reset the dispatcher unique ptr to safely close its thread
 */
void InfluxdbMetricsListener::clear()
{
  m_dispatcher.reset();
}

//--------------------------------------------------------------------------------
/*
 * Create and send a measurement to InfluxDB using the UDP protocol
 */
void InfluxdbMetricsListener::postMetric( uint64_t time, std::string name, long value, std::string tagName, std::string tagValue )
{
  int ret = -1;

  // Build the measurement with or without a tag
  if (!tagName.empty())
    ret = influxdb_cpp::builder().meas( name ).tag( tagName, tagValue ).field( "value", value ).timestamp( time ).send_udp( influxdb_ip, influxdb_udp_port );
  else
    ret = influxdb_cpp::builder().meas( name ).field( "value", value ).timestamp( time ).send_udp( influxdb_ip, influxdb_udp_port );

  if (ret != 0)
    std::cout << "Error (" << ret << ") posting metric : " << name << std::endl;
}

//--------------------------------------------------------------------------------
/*
 * Create a timestamp in nanoseconds, as requested by InfluxDB
 */
uint64_t InfluxdbMetricsListener::createTimestamp() const
{
  time_stamp ts = std::chrono::time_point_cast<std::chrono::nanoseconds>(std::chrono::system_clock::now());
  return ts.time_since_epoch().count();
}

//--------------------------------------------------------------------------------
void InfluxdbMetricsListener::onMeasuredNetworkLatency( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Client> client )
{
  if (m_connected)
  {
    // Dispatch the function execution to another thread
    m_dispatcher->dispatch( std::bind( postMetric, createTimestamp(), "rv_network_latency", (long)time, "client_id", client->getId() ) );
  }
}

//--------------------------------------------------------------------------------
void InfluxdbMetricsListener::onMeasuredDecodingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Connection> connection )
{
  if (m_connected)
  {
    // Dispatch the function execution to another thread
    m_dispatcher->dispatch( std::bind( postMetric, createTimestamp(), "rv_decoding_time", (long)time, "connection_id", connection->getId() ) );
  }
}

//--------------------------------------------------------------------------------
void InfluxdbMetricsListener::onMeasuredRenderingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea )
{
  if (m_connected)
  {
    // Dispatch the function execution to another thread
    m_dispatcher->dispatch( std::bind( postMetric, createTimestamp(), "rv_rendering_time", (long)time, "renderarea_id", renderArea->getId() ) );
  }
}

//--------------------------------------------------------------------------------
void InfluxdbMetricsListener::onMeasuredEncodingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Connection> connection )
{
  if (m_connected)
  {
    // Dispatch the function execution to another thread
    m_dispatcher->dispatch( std::bind( postMetric, createTimestamp(), "rv_encoding_time", (long)time, "connection_id", connection->getId() ) );
  }
}

//--------------------------------------------------------------------------------
void InfluxdbMetricsListener::onChangedNumClients( unsigned int number )
{
  if (m_connected)
  {
    // Dispatch the function execution to another thread
    m_dispatcher->dispatch( std::bind( postMetric, createTimestamp(), "rv_num_clients", (long)number, "", "" ) );
  }
}

//--------------------------------------------------------------------------------
void InfluxdbMetricsListener::onChangedNumConnections( unsigned int number )
{
  if (m_connected)
  {
    // Dispatch the function execution to another thread
    m_dispatcher->dispatch( std::bind( postMetric, createTimestamp(), "rv_num_connections", (long)number, "", "" ) );
  }
}

//--------------------------------------------------------------------------------
void InfluxdbMetricsListener::onChangedNumRenderAreas( unsigned int number )
{
  if (m_connected)
  {
    // Dispatch the function execution to another thread
    m_dispatcher->dispatch( std::bind( postMetric, createTimestamp(), "rv_num_renderareas", (long)number, "", "" ) );
  }
}
