#include <SharedViewerServiceListener.h>

#include <RemoteViz/Rendering/RenderArea.h>
#include <RemoteViz/Rendering/Connection.h>

#include <Inventor/SoSceneManager.h>
#include <Inventor/touch/SoTouchManager.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/manips/SoHandleBoxManip.h>
#include <Inventor/SoPath.h>
#include <Inventor/SoPickedPoint.h>

//--------------------------------------------------------------------------------
SharedViewerServiceListener::SharedViewerServiceListener()
{
  // Instantiate the class overriding the renderAreaListener class behavior to manage the renderArea events.
  m_renderAreaListener_session1 = std::make_shared<SharedViewerRenderAreaListener>();
  m_renderAreaListener_session2 = std::make_shared<SharedViewerRenderAreaListener>();
  m_renderAreaListener_session3 = std::make_shared<SharedViewerRenderAreaListener>();
}

//--------------------------------------------------------------------------------
// Returns path to xform left of the input path tail.
// Inserts the xform if none found. In this example,
// assume that the xform is always the node preceeding
// the selected shape.
SoPath *findXform(SoPath *p)
{
  SoPath *returnPath;

  // Copy the input path up to tail's parent.
  returnPath = p->copy(0, p->getLength() - 1);

  // Get the parent of the selected shape
  SoGroup *g = (SoGroup *) p->getNodeFromTail(1);
  int tailNodeIndex = p->getIndexFromTail(0);

  // Check if there is already a transform node
  if (tailNodeIndex > 0) {
    SoNode *n = g->getChild(tailNodeIndex - 1);
    if (n->isOfType(SoTransform::getClassTypeId())) {
      // Append to returnPath and return it.
      returnPath->append(n);
      return returnPath;
    }
  }

  // Otherwise, add a transform node.
  SoTransform *xf = new SoTransform;
  g->insertChild(xf, tailNodeIndex); // right before the tail
  // Append to returnPath and return it.
  returnPath->append(xf);
  return returnPath;
}

//--------------------------------------------------------------------------------
// Returns the manip affecting this path. In this example,
// the manip is always preceeding the selected shape.
SoPath* findManip(SoPath *p)
{
  SoPath *returnPath;

  // Copy the input path up to tail's parent.
  returnPath = p->copy(0, p->getLength() - 1);

  // Get the index of the last node in the path.
  int tailNodeIndex = p->getIndexFromTail(0);

  // Append the left sibling of the tail to the returnPath
  returnPath->append(tailNodeIndex - 1);
  return returnPath;
}

//--------------------------------------------------------------------------------
// Add a manipulator to the transform affecting this path
// The first parameter, userData, is not used.
void selCB(void *, SoPath *path)
{
  if (path->getLength() < 2) return;

  // Find the transform affecting this object
  SoPath *xfPath = findXform(path);
  xfPath->ref();

  // Replace the transform with a manipulator
  SoHandleBoxManip *manip = new SoHandleBoxManip;
  manip->ref();
  manip->replaceNode(xfPath);

  // Unref the xfPath
  xfPath->unref();
}

//--------------------------------------------------------------------------------
// Remove the manipulator affecting this path.
// The first parameter, userData, is not used.
void deselCB(void *, SoPath *path)
{
  if (path->getLength() < 2) return;

  // Find the manipulator affecting this object
  SoPath *manipPath = findManip(path);
  manipPath->ref();

  // Replace the manipulator with a transform 
  SoTransformManip *manip = 
    (SoTransformManip *) manipPath->getTail();
  manip->replaceManip(manipPath, new SoTransform);
  manip->unref();

  // Unref the manipPath
  manipPath->unref();
}

//--------------------------------------------------------------------------------
void SharedViewerServiceListener::onInstantiatedRenderArea(std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea)
{

  // Add the renderAreaListener instance as renderArea listener. Each session has its own renderAreaListener.
  if (renderArea->getId() == "session1")
  {
    renderArea->addListener(m_renderAreaListener_session1);
  } 
  else  if (renderArea->getId() == "session2")
  {
    renderArea->addListener(m_renderAreaListener_session2);
  } 
  else  if (renderArea->getId() == "session3")
  {
    renderArea->addListener(m_renderAreaListener_session3);
  }

  // Instantiate a sceneExaminer to interact with the camera
  SceneExaminer* examiner = new SceneExaminer();

  // Add recognizers for gesture events
  renderArea->getTouchManager()->addDefaultRecognizers();

  /* Scene graph*/

  // Create and set up the selection node
  SoSelection* sel = new SoSelection;
  sel->policy.setValue(SoSelection::SINGLE);
  sel->addSelectionCallback(selCB);
  sel->addDeselectionCallback(deselCB);
  sel->addChild(examiner);

  // Add the color switch
  SoSwitch* colorswitch = new SoSwitch();
  SoMaterial* material = new SoMaterial();
  material->diffuseColor.setValue(1.0,0,0);
  colorswitch->addChild(material);
  examiner->addChild(colorswitch);

  // Add the background
  examiner->addChild(new SoGradientBackground());

  // Apply the sceneExaminer node as renderArea scene graph
  renderArea->getSceneManager()->setSceneGraph(sel);

  // viewall
  examiner->viewAll(renderArea->getSceneManager()->getViewportRegion());
}
