/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T.MEHAMLI (Jun 2007)
**=======================================================================*/

#ifndef SO_GUI_GL_WIDGET_H
#define SO_GUI_GL_WIDGET_H

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoGuiGLWidget
//
// SoXXGLWidgets implementation class
// 
//
//////////////////////////////////////////////////////////////////////////////

#include <Inventor/antialiasing/SiAntialiasingEventListener.h>
#include <Inventor/Gui/SoGuiComponent.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/devices/SoGLFormat.h>
#include <Inventor/nodes/SoFullSceneAntialiasing.h>
#include <Inventor/image/SbRasterImage.h>

class SoGLGraphicConfigTemplate;

#ifndef HIDDEN_FROM_DOC
typedef SbBool (*soGUIDependentCB)(void* data);
typedef void* (*soGUIWindowDependentCB)(void* data);
#endif

/** [OIV-WRAPPER-NO-WRAP] */
class INVENTORGUI_API SoGuiGLWidget : public SoGuiComponent 
{
#ifndef HIDDEN_FROM_DOC
  SO_FIELDCONTAINER_HEADER( SoGuiGLWidget );
#endif

SoINTERNAL public:

  SoSFFieldContainer gcTemplate;
  SoSFBool focus;

  enum FloatColorBufferSize 
  {
    /**
      * 16-bit rendering per component.
      */
    FLOAT_16_COLOR_BUFFER = 0,
    /**
      * 32-bit rendering per component.
      */
    FLOAT_32_COLOR_BUFFER = 1
  };
  
  /**
  * Constructor.
  */
  SoGuiGLWidget();

  /** 
    * @copydoc SoSceneManager::setAntialiasing(float,AntialiasingMode)
    */
  void setAntialiasing(const float quality, const SoSceneManager::AntialiasingMode mode = SoSceneManager::AUTO);

  /** 
    * @copydoc SoSceneManager::setAntialiasing(SoAntialiasingParameters*)
    */
  void setAntialiasing(SoAntialiasingParameters* advancedParameters);

  /** 
    * @copydoc SoSceneManager::getAntialiasingQuality()
    */
  float getAntialiasingQuality() const;

  /** 
    * @copydoc SoSceneManager::getAntialiasingMode()
    */
  SoSceneManager::AntialiasingMode getAntialiasingMode() const;

  /** 
    * @copydoc SoSceneManager::getAntialiasingParameters
    */
  SoAntialiasingParameters* getAntialiasingParameters() const;

  /**
   * Sets the size of the color map
   */
  void setColorMapSize( int size );

  /**
   * Returns the size of the color map
   */
  int getColorMapSize();

  /**
   * Returns whether double buffering is on or off.
   */
  SbBool isDoubleBuffer();

  /**
   * Shows/hides a border of thickness 3 around the glX widgets. This
   * if OFF by default, but turned on by the render area.
   */
  void setBorder( int newValue );

  /**
   * Included for portability only.
   */
  int getBorderSize();

  /**
   * Returns whether the border is on or off.
   */
  SbBool isBorder() const;
 
  /**
   * Sets drawing to the front buffer. Controls drawing to the front buffer when
   * an obscured portion of the window is exposed. Default: TRUE.
   */
  void setDrawToFrontBufferEnable( SbBool enableFlag );

  /**
   * Queries drawing to the front buffer.
   */
  SbBool isDrawToFrontBufferEnable() const;

  /**
   * Enables/disables floating point rendering using 16- or 32-bit components.
   * If TRUE, Open Inventor will automatically render to a floating point color
   * buffer.
   *
   * To determine if floating point rendering was successfully enabled,
   * use #getFloatingColorBuffer.
   *
   * Using floating point rendering can improve image quality, particularly when
   * many objects are being blended.
   */
  void setFloatingColorBuffer( SbBool enable, FloatColorBufferSize size = FLOAT_16_COLOR_BUFFER );

  /**
   * Returns TRUE if floating point rendering is used and its precision.
   */
  void getFloatingColorBuffer( SbBool& enable, FloatColorBufferSize& size );

  /**
   * Returns TRUE if floating point rendering is available.
   */
  static SbBool isFloatingColorBufferSupported();

  /**
   * Sets the glMode member
   * This function updates the format returned by getGLFormat()
   * @param mode the new mode.
   */
  void setGLModes( int mode );

  /**
   * Returns the glMode member.
   */
  int getGLModes() const;

  /**
   * This function allows custom formats to be used by the render area.
   * It is possible to get the current format using getGLFormat() and enable some features
   * or different colors formats.
   * 
   * Returns TRUE on success.
   */
  SbBool setGLFormat( const SoGLFormat& format );

  /**
   * Returns the current OpenGL format used by the render area.
   */
  const SoGLFormat& getGLFormat() const;

  /** 
   * Format of the callback function to provide to the setOnFormatChangeCalback().
   *
   * Returns FALSE if the format is not usable.
   */
  typedef SbBool (*FormatChangeCallback)( SoGLFormat&, void* userData );

  /** 
   * This function sets a callback function which checks and applies new formats.
   * It must destroy and create the OpenGL surfaces according to the new format.
   */
  void setOnFormatChangeCalback( FormatChangeCallback callback, void* userData = NULL );

  /** 
   * Enable double buffering in the current OpenGL configuration.
   *
   * When the function is called the OpenGL surface is recreated.
   *
   * Returns TRUE on success.
   */
  virtual SbBool setDoubleBuffer( SbBool enable );

  /**
   * Enable remote rendering (X11 systems only).
   */
  void setRemoteRendering( bool enable );

  /** 
   * Returns true if remote rendering is enabled (X11 systems only).
   */
  bool isRemoteRendering() const;

  /** 
   * Enable OpenGL "quad buffer" stereo in the current render area.
   * Returns TRUE on success.
   */
  SbBool setStereoEnabled( SbBool enable );

  /** 
   * Returns true if OpenGL stereo is enabled.
   */
  SbBool isStereoEnabled() const;

  /** @copydoc SoWinGLWidget::saveSnapshot
   */
  // Note: It would make more sense to copydoc *from* this class, but
  //       Doxygen won't do that because this class is not currently documented... :-(
  bool saveSnapshot( const SbString& filename, bool override = true );

  // Accessor and mutator for the member m_drawToFrontBuffer
  void setDrawToFrontBuffer( SbBool enable );
  SbBool getDrawToFrontBuffer() const;

  // Accessor and mutator for the member m_colorMap
  void setColorMap( unsigned long colorMap );
  unsigned long getColorMap() const;

  // ----------------------- Extension usage -----------------------------------
 
  // sets/gets the size of the glx widget(s) - Note this size could be
  // different from the SoXtComponent::getSize() method which return
  // the size of the component, not necessary the same as the glx widget
  // window (because of extra stuff like the decoration in the viewers).
  void setGlxSize( SbVec2s newSize );
  const SbVec2s& getGlxSize() const;
 
  SbBool isStereoBuffer();
 
  // returns TRUE if main window is in rgb mode (FALSE if color index)
  SbBool isRGBMode();

  // return the graphic config template from field gcTemplate.
  void setGraphicConfigTemplate( SoGLGraphicConfigTemplate* gc );
  SoGLGraphicConfigTemplate* getGraphicConfigTemplate();

  // setup system dependent GLWidget handles.
  void setGLWidgetHandle( const void* glWidgetHandle,
                          soGUIDependentCB swapBufferCB,
                          soGUIDependentCB bindCurrentCB,
                          soGUIDependentCB unbindCurrentCB,
                          soGUIWindowDependentCB getNormalCB,
                          soGUIWindowDependentCB getDisplayCB = NULL);

  // These call are internal public to let IvTune to access them.
  // handle on GLWidget handle/function comming from system independent GUI
  // initialized through setGLWidgetHandle;
  static soGUIDependentCB swapNormalBuffersCB;
  static soGUIDependentCB bindNormalCurrentCB;
  static soGUIDependentCB unbindNormalCurrentCB;
  static soGUIWindowDependentCB getNormalWindowCB;
  static soGUIWindowDependentCB getCurrentDisplayCB;
  void * m_glWidgetHandle;

  virtual void* getNormalWindow() { return getNormalWindowCB(m_glWidgetHandle); }
  virtual void* getDisplay() { return getCurrentDisplayCB(m_glWidgetHandle); }

  virtual SbBool bindNormalContext();
  virtual SbBool unbindNormalContext();

  bool onAntialiasingSwitch(const SoSceneManager::AntialiasingMode& mode, const float& quality, const SoAntialiasingParameters* parameters);



protected:

  /**
  * Destructor.
  */
  virtual ~SoGuiGLWidget();

  SiAntialiasingEventListener* getAntialiasingListener() const;

// ----------------------- Private usage -------------------------------------

private:

  // Used by deprecated Antialiasing API
  SoAntialiasingParameters* m_fsaaParameters;

  // local vars
  SbVec2s m_glxSize; // size of glx widgets which is always up to date
  int m_glModes;
  int m_borderSize;

  SoGLFormat m_format;
  FormatChangeCallback m_formatChangeCallback;
  void* m_formatChangeCallbackUserData;
  
  unsigned long m_colorMap; // set when color index is used
  int m_colorMapSize;

  SbBool m_drawToFrontBuffer; // used by RA to draw new scenes to front buffer
  SbBool m_enableDrawToFrontBuffer;
 
  unsigned long m_transparentPixel;
  
  bool m_isRemoteRendering;

  int m_snapshotFBOId;

  SoGLGraphicConfigTemplate* m_gcTemplate;

  SiAntialiasingEventListener* m_antialiasingListener;
};


#endif // SO_GUI_GL_WIDGET_H


