/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : C. HUFSCHMITT (Jul 2001)
**=======================================================================*/
/*=======================================================================
** Modified by : Tristan Mehamli (Nov 2009)
**=======================================================================*/

#ifndef _SO_MPEGRENDERER_
#define _SO_MPEGRENDERER_

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/helpers/SbGlContextHelper.h>
#include <Inventor/SbBasic.h>
#include <Inventor/SbColor.h>

#include <Inventor/STL/list>

class TGSMPEG;
class SbThreadSignal;
class SbThreadMutex;

/*----------------------------------------------------------------------------*/
/**
* @VSGEXT Base class for generating MPEG output from a scene graph.
* 
* @ingroup MPEG
* 
* @DESCRIPTION
*
*   This class is the base class for creating MPEG video output from a 
*   scene graph. Two specialized classes SoMPEGFrameRenderer and SoMPEGNavRenderer 
*   are provided to give applications more control over video generation.
*
*   Warning : Writing in multiple streams at the same time is not possible.
*
*   This class is used by the viewer classes to record MPEG video (see for example
*   the SoWinRenderArea method \if_dotnet SetMPEGRecorder \else setMPEGRecorder \endif).
*   The default right-mouse popup menu in the viewer classes provides an option to 
*   display a video recording dialog box.  This dialog allows the user to conveniently
*   start and stop recording the scene in the viewer window.
*
*   The \if_dotnet #SetShareContext \else #setShareContext \endif method
*   allows you to share an existing OpenGL context with the MPEG renderer. 
*   This avoids the necessity to re-generate textures and display lists 
*   if they are already available in another OpenGL context (the viewer 
*   context, for instance). This can dramatically reduce offscreen 
*   rendering time, depending on your scene graph. 
*
*   Here's how you might use these methods to share OpenGL contexts:
*   \if_cpp
*     \par
*     \code
*     SoMPEGRenderer* renderer = new SoMPEGRenderer();
*     renderer->setShareContext( viewer->getShareContext() );
*     \endcode
*   \endif
*   \if_dotnet
*     \par
*     \code
*     SoMPEGRenderer renderer = new SoMPEGRenderer();
*     renderer.SetShareContext( viewer.GetShareContext() );
*     \endcode
*   \endif
*   \if_java
*     \par
*     \code
*     SoMPEGRenderer renderer = new SoMPEGRenderer();
*     renderer.setShareContext( viewer.getArea().getGLSharedContext() );
*     \endcode
*   \endif
*
*   The MPEGRender uses two separate tasks. One to record rendered frames and
*   one to encode the frames in MPEG format (this is a much slower task). If 
*   the recording task is stopped, the encoding one can continue. If 
*   \if_dotnet #OpenFile \else #openFile \endif is called before the end of 
*   this task, two scenarios can occur. If the filename is the same file as the
*   one currently in use by the encoding task, this task is stopped.
*   The behavior is the same if \if_dotnet #Record \else #record \endif is called 
*   after \if_dotnet #Stop \else #stop \endif\.
*   If the filename is different, the file is not opened and encoding continues.
*
*   It is important to note that the MPEGRenderer can only generate raw MPEG-1
*   video streams. In order for the generated video to be played correctly by any
*   video player, it needs to be embedded into a container. This operation can
*   be easily performed by the multimedia tool
*   <a href="https://www.ffmpeg.org/">FFmpeg</a>. For example, the following
*   command allows you to embed the generated raw video stream into an MP4
*   container format:
*
*   \code
*   ffmpeg -i input_raw_stream.mpg -codec copy output_video.mp4
*   \endcode
*
*   The <i>-codec copy</i> argument is important because it allows ffmpeg
*   to only copy the input stream inside the container instead of doing a real
*   re-encoding.
*
*   Different container formats can also be used, like AVI or MKV.
*
* @SEE_ALSO
*    SoMPEGNavRenderer, SoMPEGFrameRenderer
* 
* 
*/
class INVENTOR_API SoMPEGRenderer 
{
public:

  /**
   * Constructor.
   */
  SoMPEGRenderer();

  /**
   * Destructor.
   */
  virtual ~SoMPEGRenderer();

  /** Components */
  enum Components 
  {
    /** Luminance */
    LUMINANCE = 1,
    /** Luminance transparency */
    LUMINANCE_TRANSPARENCY = 2,
    /**
     *  (Default) 
     */
    RGB = 3,
    /** RGB and Alpha channel */
    RGB_TRANSPARENCY = 4
  };

  /**
   * Specifies the MPEG output file name.
   * @I numFrames @i gives an estimate of the number of frames that
   * the MPEG output will contain.
   *
   * If filename is the same as the one currently in use by the encoding task
   * (if there is one), the encoding task is stopped. If the filename is 
   * different, the file is not opened and the current encoding task continues.
   *
   * Returns FALSE if the file cannot be opened, TRUE otherwise.
   */
  SbBool openFile( const char* filename, unsigned int numFrames = 10000 );

  /**
   * Closes the MPEG file previously opened with the openFile() method.
   */
  void closeFile();

  /**
   * Sets the file pointer for the MPEG output.
   */
  void setFilePointer( FILE* fp );

  /**
   * Gets the MPEG file pointer.
   */
  FILE* getFilePointer() const;

  /**
   * Sets the frame pixel size. The width and height of the frame must be a 
   * multiple of 16. If they are not, they are reduced to the next lower 
   * multiple of 16.
   *  
   * By default, width and height values are 96.
   *
   * NOTE: This method must be called before #openFile or 
   * #setFilePointer in order to have an effect. Calling it after
   * calling #openFile or #setFilePointer may cause undesirable
   * results.
   */
  void setSize( const SbVec2s& size );

  /**
   * Gets the frame pixel size. These values may be different from those given
   * by the setSize() method.
   */
  SbVec2s getSize() const;

  /**
   * Set the number of color components of the frames recorded.
   */
  void setComponents( const Components components );

  /**
   * Returns the number of color components of the frames recorded.
   */
  SoMPEGRenderer::Components getComponents() const;

  /** 
   * Sets the scene graph used for generating frames in the MPEG output.
   */ 
  virtual void setSceneGraph( SoNode* node );

  /**
   * Gets the scene graph used for generating frames in the MPEG output.
   */
  SoNode* getSceneGraph() const;

  /**
   * Sets the background color for rendering each frame.
   * The default background color is 0 0 0 (black).
   *
   * The default value can be set using the environment variable
   * OIV_BACKGROUND_COLOR. Specify three floats (R, G, B) in the range 0. to 1.,  
   * separated by spaces.
   */
  void setBackgroundColor( const SbColor& c );

  /**
   * Gets the background color for rendering each frame.
   */
  SbColor getBackgroundColor() const;

  /**
   *  Sets the GL Render action used to generate each frame.
   *  If no render action is specified, an internal instance of an 
   *  SoGLRenderAction will be maintained with a viewport
   *  region set to the size given with the method setSize().
   */
  void setGLRenderAction( SoGLRenderAction* ra );

  /**
   *  Gets the GL Render action used to generate each frame.
   */
  SoGLRenderAction* getGLRenderAction() const;

  /**
   * Compression rate of the MPEG output.
   * Values are between 0 and 1. 
   * 0=no compression, 1=maximum compression.
   *
   * The default value is 0.3. 
   *
   * NOTE: This method must be called before #openFile or 
   * #setFilePointer in order to have an effect. Calling it after
   * calling #openFile or #setFilePointer may cause undesirable
   * results.
   */
  void setCompressionRate( float value ); 

  /**
   * Returns the compression rate
   */
  float getCompressionRate() const;

  /**
   * This method allows the user to specify a fixed bit rate.
   * This is useful when the MPEG output is to be put on a video CD, for example.
   * If you do not know what this rate means, you can use the default value (-1)
   * which indicates that the bit rate is variable.
   * The standards bit rates for VCD are 1205862 bps and 2726300 bps.
   * (It is the speed the MPEG stream is read by VCD players)
   *
   * The default value is   1205862.
   * The maximum value is 104857600.
   * NOTE: This method must be called before #openFile or 
   * #setFilePointer in order to have an effect. Calling it after
   * calling #openFile or #setFilePointer may cause undesirable
   * results.
   */
  void setBitPerSec( float mbps );

  /** 
   * Sets the number of frames per second encoded in the MPEG output.
   * It has nothing to do with the speed of playback.
   * The default value is 30.
   *
   * NOTE: This method must be called before #openFile or 
   * #setFilePointer in order to have an effect. Calling it after
   * calling #openFile or #setFilePointer may cause undesirable
   * results.
   */
  void setNumFramesPerSecond( float num );

  /**
   * Gets the number of frames per second to be encoded in the MPEG output.
   */
  int getNumFramesPerSecond();

  /**
   * Low level method to add a new frame to the MPEG output from a buffer of 
   * unsigned characters.
   * Each pixel is stored sequentially by scanline, starting
   * with the lower left corner. The data stored for each pixel is determined by the
   * components set before rendering (3 by default). 
   * Pixels are stored in RGBA order and are packed without any padding between pixels 
   * or scanlines.
   */
  void addFrame( const unsigned char* frame );

  /**
   * Sets the OpenGL context to be shared by the SoMPEGRenderer.
   * Its use is strongly recommended, because it can improve performance when 
   * switching between on-screen and off-screen rendering. This is because it allows 
   * the display lists and texture objects to be shared between the on-screen and 
   * off-screen render context.
   * The shared context info is normally obtained by calling the viewer's
   * @B getShareContext@b method.
   *
   * NOTE: It has no effect when the hardware (or driver) does not allow sharing 
   * with the off-screen context, for example if Pbuffers are not supported.
   */
  void setShareContext( const SbGLShareContext shareCxt );

  /**
   * Gets the OpenGL context shared by this object.
   */
  const SbGLShareContext getShareContext() const;

  /** 
   * Starts the recording.
   * If called directly after #stop method, the current encoding task (if one) is stopped.
   *
   * Note: A file must have been opened or a file pointer set.
   */
  virtual void record();

  /** 
   * Pauses the recording.
   */
  virtual void pause();

  /** 
   * Stops the recording and closes the opened file.
   */
  virtual void stop();

  /** 
   * Returns true if this renderer is recording.
   */
  SbBool isRecording() const;

  /**
   * Tells the recorder that the viewer is idle.
   */
  void setViewerIsIdle( bool isIdle );
  bool isViewerIdle() const { return m_viewerIsIdle; };

  /*----------------------------------------------------------------------------*/
  SoINTERNAL public:
  /**
   * It computes the ratio between recorder frame rate and the viewer's FPS.
   * It is used for an adaptative encoding of the frames.
   */
  void computeRate( double curVwrFPS );

  /**
   * Adds the last frame of the stack to the stack of frames to encode.
   */
  void addIdleFrame();

  /**
   * Returns the number of frame left to encode 
   */
  int getNumFrameToEncode();

protected:
  SoMPEGRenderer& operator =( const SoMPEGRenderer& a );
  // Share context
  SbGLShareContext m_glShareContext;

private:
  // Encoding Task that "loops" until there is no more frame to encode.
  void encodeTask(void);
  // main method for the recording thread.
  static void* encodeCB( void* data );
  // Streamed buffers stack.
  static std::list<unsigned char*>* m_frameStack;
  // Synchronization between app thread and encoding thread.
  static SbThreadMutex* m_stackMutex; // Used when accessing the stack.
  SbThread* m_stackThread;
  // Frame size in pixel.
  SbVec2s m_Size;
  // Component number for one frame, frame rate, frame number in m_frameStack.
  int m_frameSize, m_recFrameRate, m_frameCount;
  // Real frame rate (according to viewer FPS).
  double m_rate;
  // Backgroung color.
  SbColor m_BgColor;
  // Render action set by setGLRenderAction.
  SoGLRenderAction* m_ra, *m_defaultRenderAction;
  // MPEG algorithm.
  TGSMPEG* m_mpeg;
  // Current scene graph to render. 
  SoNode* m_SceneGraph;
  // Number of component for one frame.
  Components m_numComponents;
  // True if the scene do not change.
  bool m_viewerIsIdle;
  // True if a file is opened, true if recording.
  SbBool m_hasFileOpened, m_isRecording;

  // Name of the current opened file.
  SbString m_openedFile;
  // True if openFile displayed a warning to indicate that a thread
  // is encoding on this file.
  SbBool m_openedFileWarningDisplayed;
};/*----------------------------------------------------------------------------*/

#endif /* _SO_MPEGRENDERER_ */


