/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef _SB_QT_HELPER
#define _SB_QT_HELPER

#include <QCoreApplication>
#include <QString>
#include <QFile>
#include <QWidget>

#include <Inventor/SoPreferences.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/sys/SoDynamicLibManager.h>

/**
* @VSGEXT Utilities class for Qt management.
*
* @ingroup Qt
*
* [OIV-WRAPPER-NO-WRAP]
*/
class SbQtHelper
{
public:
  /**
   * When a Qt application is run, Qt will first treat the application's executable
   * directory as the base directory for searching for platform plugins. The purpose
   * of this function is to load Qt platform plugins by guessing the Qt platforms path
   * from the Open Inventor binaries path.
   *
   * The Qt environment variable QT_QPA_PLATFORM_PLUGIN_PATH also enables to specify
   * the platform plugins path. Nevertheless, the use of both this environment variable
   * and the Qt method QCoreApplication::addLibraryPath causes significant load delay.
   *
   * Qt platform plugins are stored in $OIVHOME/$OIVHARCH/bin/platforms
   *
   * \param path : force the directory for searching for platforms plugins.
   */
  static void addPlatformPluginsPath(SbString path = "")
  {
    // if the QApplication already exists, we can safely assume everything is in order
    // if there is a qt.conf file, we do not want to apply our library path modifications
    // as we should trust the qt.conf to do everything that is required.
    const bool qtconfExists = QFile::exists( "qt.conf");
    if ( qApp || qtconfExists )
      return;

    if ( SoPreferences::getString("QT_QPA_PLATFORM_PLUGIN_PATH", "").isEmpty() )
    {
      if ( path.isEmpty() )
      {
        // Try to guess QT platforms path from Open Inventor binaries path
        const SbString pluginPath = qtPluginsPath();
        if ( SbFileHelper::isAccessible(pluginPath) )
        {
          QCoreApplication::addLibraryPath( QString(pluginPath.getString()) );
        }
      }
      else
      {
        QCoreApplication::addLibraryPath( QString(path.getString()) );
      }
    }
  }

  static SbString qtPluginsPath()
  {
    const SbString binariesPath = SoDynamicLibManager::getLibraryFromSymbol((void*)SoInventorBase::init);
    const SbString qtPlatformsPath = SbFileHelper::toUnixPath(SbFileHelper::getDirName(binariesPath) + "qtplugins/");
    return qtPlatformsPath;
  }

  /** Return pixel ratio of the widget. HDPI devices, like Retina screen, may contains several pixel per point.
   * Qt coordinates, like widget size, move event position, etc... are generally returned in points.
   * Coordinates must be multiplied by this value to be converted in pixels.
   * warning: Function return relevant value only once a SoGlContext has been created
   * Note that Retina screen can be simulated on windows by setting QT_DEVICE_PIXEL_RATIO=2
   * See http://doc.qt.io/qt-5/highdpi.html for details.
   *
   * \param widget : widget from which pixel ratio is return.
   */
  static float getDevicePixelRatio(QWidget* widget)
  {
    if (widget == nullptr || widget->windowHandle() == nullptr)
      return 1.f;

    return static_cast<float>( widget->devicePixelRatio() );
  }

};

#endif // SB_QT_HELPER


