/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef  SB_DATATYPE_H
#define  SB_DATATYPE_H

#include <Inventor/STL/map>

#include <Inventor/SbBase.h>
#include <Inventor/SbString.h>
#include <Inventor/sys/port.h>
#include <Inventor/STL/limits>
#include <Inventor/errors/SoDebugError.h>

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

/**
 * Class encoding a data type
 *
 * @ingroup Basics
 *
 * @DESCRIPTION
 *   This is a basic Open Inventor type that is used for representing a data type,
 *   for example unsigned int32, and some information about that type, for example
 *   is it signed or not. It is used by classes handling user buffers
 *   such as SoMemoryObject, SoSFArray2D, SoSFArray3D, SoSFImage and also SoDataSet.
 *
 * @SEE_ALSO
 *   SoMemoryObject,
 *   SoSFArray2D,
 *   SoSFArray3D
 *
 *
 */
class INVENTORBASE_API SbDataType
{
public:
  /**
   * Supported Data type
   */
  enum DataType
  {
    /** unsigned byte */
    UNSIGNED_BYTE = 0,
    /** unsigned short */
    UNSIGNED_SHORT = 1,
    /** unsigned int (32bits) */
    UNSIGNED_INT32 = 2,
    /** signed byte */
    SIGNED_BYTE = 4,
    /** signed short */
    SIGNED_SHORT = 5,
    /** signed int (32bits) */
    SIGNED_INT32 = 6,
    /** float */
    FLOAT = 10,
    /** Double */
    DOUBLE = 11,
    /** unknown data type */
    UNKNOWN = 0xFFFF
  };

  /**
   * Copy constructor
   */
  explicit SbDataType(DataType type) : m_type(type) {}

  /**
   * Constructor from a string.
   * Valid strings are the enum names.
   */
  SbDataType(const SbString& type);

  /**
   * Default constructor.  The initial value is UNSIGNED_BYTE.
   */
  SbDataType() : m_type(UNSIGNED_BYTE) {}

  /* 
   * Returns the type as an unsigned int (rather than an enum).
   * See also getType().
   */
  inline operator unsigned int() const;

  /**
   * Returns the type as an enum.
   * See also getString().
   */
  inline DataType getType() const;

  /**
   * Returns size in bytes of the type.
   */
  inline unsigned int getSize() const;

  /**
   * Returns true if the type is signed.
   */
  inline SbBool isSigned() const;

  /**
   * Returns true if the type is an integer type.
   */
  inline SbBool isInteger() const;

  /**
   * Returns the numer of bits in the type.
   */
  inline unsigned int getNumBits() const;

  /**
   * Returns the type as a string, e.g. "FLOAT".
   * See also getType().
   */
  SbString getString() const;

  /**
   * Returns the minimum value of the type. @BR
   * For floating point type, returns the minimum positive normalized value.
   */
  double getMin() const;

  /**
   * Returns the maximum value of the type.
   */
  double getMax() const;

SoINTERNAL public:

  /**
   * Returns the lowest signed value of the type. @BR
   * For floating point types, returns the lowest normalized value.
   */
  double getLowest() const;

  /**
   * Init static members
   */
  static void initClass();

  /**
   * Normalize given value into a double
   * precision floating point.
   * Floating point value are not normalized and returned as is.
   * @param val value to normalize
   * @return a double between [-1,1]
   */
  inline double normalize(double val) const;

  /** ostream operator for SbDataType */
  INVENTORBASE_API friend inline std::ostream& operator << (std::ostream& os, const SbDataType type);

  /**
   * Used to disable warning constant expression warning
   * in macros using the "if (true)" trick. See SbDataTypeMacros.h
   */
  static const bool m_true;

  /**
   * Constructor from a int.
   */
  SbDataType(int type) { m_type = static_cast<DataType>(type); };

  template<typename T> static SbDataType getTemplateType (const T&);

  /** union used to cast the generic value into specific type */
  typedef union
  {
    unsigned char  l_uCHAR;
    unsigned short l_uSHORT;
    unsigned int   l_uINT;
    signed char    l_sCHAR;
    signed short   l_sSHORT;
    signed int     l_sINT;
    float          l_FLOAT;
    double         l_DOUBLE;
  } DataValue;

  /** 
   * Cast value to type defined by this dataType.
   * returned value is an union where value is well represented in memory and can 
   * be used directly in a memcpy:
   * [PRE]
   * SbDataType dataType(FLOAT);
   * SbDataType::DataValue dataValue = dataType.cast(42);
   * memcpy( buff, &dataValue, dataType.getSize() );
   * [/PRE]
   */
  template <typename T>
  DataValue cast( T value ) const;

  template <typename T>
  T cast( void* value) const;

private:
  typedef std::map<DataType, SbString> TypeToStrMap;

  /** Return min value of given type */
  template<typename T> static double getMinInternal();

  /** Return max value of given type */
  template<typename T> static double getMaxInternal();

  /** Return lowest value of given type */
  template<typename T> static double getLowestInternal();

  /** The data type */
  DataType m_type;

  /** Used to map between type and string representation */
  static TypeToStrMap s_typeToStrMap;
  static std::map<SbString, DataType> s_strToTypeMap;
};

/*******************************************************************************/
SbDataType::DataType
SbDataType::getType() const
{
  return m_type;
}

/*******************************************************************************/
SbDataType::operator unsigned int() const
{
  return static_cast<unsigned int >(m_type);
}

/*******************************************************************************/
SbBool
SbDataType::isSigned() const
{
  // signed type start from enum value 4
  return ( (m_type >> 2)?TRUE:FALSE );
}

/*******************************************************************************/
unsigned int
SbDataType::getSize() const
{
  return (1 << (m_type % 4));
}

/*******************************************************************************/
unsigned int
SbDataType::getNumBits() const
{
  return (1 << (m_type % 4)) * 8;
}

/*******************************************************************************/
SbBool
SbDataType::isInteger() const
{
  return ( (m_type < 10)?TRUE:FALSE );
}

/*******************************************************************************/
double
SbDataType::normalize(double val) const
{
  if ( m_type != SbDataType::FLOAT )
  {
    double minType = getMin();
    double maxType = getMax();
    val = (val-minType)/(maxType-minType);

    if ( isSigned() )
      val = (val-0.5)*2.;
  }

  return val;
}

/*******************************************************************************/
template <typename T>
SbDataType::DataValue
SbDataType::cast( T value ) const
{ 
  DataValue dataValue; 
  switch ( m_type )
  {
  case UNSIGNED_BYTE : dataValue.l_uCHAR  = (unsigned char)value;  break;
  case UNSIGNED_SHORT: dataValue.l_uSHORT = (unsigned short)value; break;
  case UNSIGNED_INT32: dataValue.l_uINT   = (unsigned int)value;   break;
  case SIGNED_BYTE   : dataValue.l_sCHAR  = (signed char)value;    break;
  case SIGNED_SHORT  : dataValue.l_sSHORT = (signed short)value;   break;
  case SIGNED_INT32  : dataValue.l_sINT   = (signed int)value;     break;
  case FLOAT         : dataValue.l_FLOAT  = (float)value;          break;
  case DOUBLE        : dataValue.l_DOUBLE = (double)value;         break;
  default            : SoDebugError::postWarning("SbDataType::cast()", "Unknown data type %d", m_type); memset(&dataValue, 0, sizeof(DataValue)); break;
  }
  return dataValue;
}

/*******************************************************************************/
template <typename T>
T
SbDataType::cast( void* value ) const
{
  switch ( m_type )
  {
  case UNSIGNED_BYTE :  return (T)(*(unsigned char*)value);  break;
  case UNSIGNED_SHORT:  return (T)(*(unsigned short*)value); break;
  case UNSIGNED_INT32:  return (T)(*(unsigned int*)value);   break;
  case SIGNED_BYTE   :  return (T)(*(signed char*)value);    break;
  case SIGNED_SHORT  :  return (T)(*(signed short*)value);   break;
  case SIGNED_INT32  :  return (T)(*(signed int*)value);     break;
  case FLOAT         :  return (T)(*(float*)value);          break;
  case DOUBLE        :  return (T)(*(double*)value);         break;
  default            : SoDebugError::postWarning("SbDataType::cast()", "Unknown data type %d", m_type); break;
  }
  return (T)0;
}

template<>
inline 
SbDataType
SbDataType::getTemplateType (const unsigned char&)
{
  return SbDataType(SbDataType::UNSIGNED_BYTE);
}

template<>
inline 
SbDataType
SbDataType::getTemplateType (const unsigned short&)
{
  return SbDataType(SbDataType::UNSIGNED_SHORT);
}

template<>
inline 
SbDataType
SbDataType::getTemplateType (const uint32_t&)
{
  return SbDataType(SbDataType::UNSIGNED_INT32);
}

template<>
inline 
SbDataType
SbDataType::getTemplateType (const signed char&)
{
  return SbDataType(SbDataType::SIGNED_BYTE );
}

template<>
inline 
SbDataType
SbDataType::getTemplateType (const signed short&)
{
  return SbDataType(SbDataType::SIGNED_SHORT);
}

template<>
inline 
SbDataType
SbDataType::getTemplateType (const int&)
{
  return SbDataType(SbDataType::SIGNED_INT32);
}

template<>
inline 
SbDataType
SbDataType::getTemplateType (const float&)
{
  return SbDataType(SbDataType::FLOAT);
}

template<>
inline 
SbDataType
SbDataType::getTemplateType (const double&)
{
  return SbDataType(SbDataType::DOUBLE);
}

template<typename T> 
inline 
SbDataType 
SbDataType::getTemplateType (const T&)
{
  return SbDataType(SbDataType::UNKNOWN);
}

/**
 * Writes the type to the specified output stream.
 */
inline std::ostream& operator << (std::ostream& os, const SbDataType type)
{
  switch ( type.m_type )
  {
  case SbDataType::UNSIGNED_BYTE:
    return os << "uint8";
  case SbDataType:: UNSIGNED_SHORT:
    return os << "uint16";
  case SbDataType:: UNSIGNED_INT32:
    return os << "uint32";
  case SbDataType:: SIGNED_BYTE:
    return os << "int8";
  case SbDataType:: SIGNED_SHORT:
    return os << "int16";
  case SbDataType:: SIGNED_INT32:
    return os << "int32";
  case SbDataType:: FLOAT:
    return os << "float";
  case SbDataType:: DOUBLE:
    return os << "double";
  default:
    return os << "unknown data type!";
  }
}


#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif // SB_DATATYPE_H


