/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Mar 2015)
**=======================================================================*/

#ifndef  _SB_EVENTHANDLER_H_
#define  _SB_EVENTHANDLER_H_

#include <Inventor/SbDelegate.h>

/** 
 * Class representing an event.
 *
 * @ingroup Basics
 *
 * @DESCRIPTION
 * An SbEventHandler encapsulates a list of "callbacks" or "delegates" that will be called when an event occurs.
 * We generally think of events as input events, like mouseMove or keyboard, but they can be anything, like starting
 * a computation or notifying that an object's state has changed.
 *
 * The callback signature must be:
 * \par
 * void callback(EventArg arg) @BR
 * where @B EventArg@b is the template type of SbEventHandler.
 *
 * All SbEventHandler callbacks must return @B void @b and have a single argument of type @B EventArg @b.
 * This argument will generally be a structure containing information about the event and possibly methods
 * to query additional information.
 * In OpenInventor, all EventArg will inherit from the SbEventArg class, but it is not mandatory
 * if you want to define your own events. In any case, EventArg must not be @B void @b
 *
 * Callbacks are registered using one of the add() methods. You can register a free function,
 * a static class method or a member class method.
 *
 * @EXAMPLE
 * \code
 * // a free function
 * void freeFunction(int arg) { }
 *
 * // Some class
 * class MyClass {
 * public:
 *   // a member function
 *   void memberFunction(int arg) { }
 *   // a static function
 *   static void staticFunction(int arg) { }
 * };
 *
 * // event raised when something occurs. Prototype of callback should be void callback(int arg);
 * SbEventHandler<int> onSomth;
 *
 * // To register a free or a static function, simply add it to EventHandler:
 * onSomth.add(freeFunction);
 * onSomth.add(MyClass::staticFunction);
 *
 * // Registering a member function is a little bit more complicated.
 * // You need to specified the instance of class and a pointer to member function.
 * // WARNING: The EventHandler will keep a reference on given instance. Be careful concerning lifetime
 * // or this instance. If you want to delete it, don't forget to remove this callback front EventHandler
 * // before deleting your instance.
 * MyClass myInstance;
 * onSomth.add(myInstance, &MyClass::memberFunction);
 *
 * // Raise the event. This will call all the registered callback with given argument. Here, "12".
 * onSomth(12);
 *
 * // Unregister callback
 * onSomth.remove(freeFunction);
 * onSomth.remove(MyClass::staticFunction);
 * onSomth.remove(myInstance, &MyClass::memberFunction);
 *
 * \endcode
 *
 * @B Warning@b: All registered callbacks will be called exactly once, but the order is undefined.
 *
 * @SEE_ALSO
 *   SbEventArg
 *
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
template <typename ArgType>
class SbEventHandler
{
public:

  /**
   * Call all registered callbacks.
   * @param arg Argument that will be given to callback.
   * @warning Calling order of callbacks is undefined.
   */
  void operator()(ArgType arg)
  {
    //Copy the delegate vector because the called function may change the vector
    DelegateVector delegateCopy = m_delegates;
    typename DelegateVector::const_iterator it = delegateCopy.begin();
    typename DelegateVector::const_iterator end = delegateCopy.end();
    while ( it != end )
    {
      (*it)(arg);
      ++it;
    }
  }

  /** Add a static function callback for this event */
  void add(void (*f)(ArgType))
  { 
    inventor::helper::SbDelegate<void, ArgType> delegate = inventor::helper::SbDelegate<void, ArgType>(f);
    m_delegates.push_back(delegate); 
  }

  /** Add a member function callback for this event */
  template <typename ClassType>
  void add(ClassType& a, void (ClassType::*f)(ArgType))
  { 
    inventor::helper::SbDelegate<void, ArgType> delegate = inventor::helper::SbDelegate<void, ArgType>(a, f);
    m_delegates.push_back(delegate); 
  }

  /**
   * Remove a static function.
   * If you add the same callback several times, you must remove it several times.
   * @return True if callback has been removed.
   */
  bool remove(void (*f)(ArgType))
  {
    for ( typename DelegateVector::iterator itDelegate = m_delegates.begin(); itDelegate != m_delegates.end(); itDelegate++ )
    {
      if ( itDelegate->equal(f) )
      {
        m_delegates.erase(itDelegate);
        return true;
      }
    }
    return false;
  }

  /**
   * Remove a member function.
   * If you add the same callback several times, you must remove it several times.
   * @return True if callback has been removed.
   */
  template <typename ClassType>
  bool remove(ClassType& a, void (ClassType::*f)(ArgType))
  {
    for ( typename DelegateVector::iterator itDelegate = m_delegates.begin(); itDelegate != m_delegates.end(); itDelegate++ )
    {
      if ( itDelegate->equal(a, f) )
      {
        m_delegates.erase(itDelegate);
        return true;
      }
    }
    return false;
  }

private:

  typedef std::vector< inventor::helper::SbDelegate<void, ArgType> > DelegateVector;
  /** list of delegate connected to this event */
  DelegateVector m_delegates;
};

#endif // _SB_EVENTHANDLER_H_
