#ifndef SB_MATH_HELPER_H
#define SB_MATH_HELPER_H

/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/


#include <Inventor/SbBase.h>
#include <cfloat>
#include <cmath>
#include <Inventor/STL/limits>
#include <Inventor/STL/complex>

namespace SbMathHelper
{
  /**
  * Maximum value returned by SbMathHelper::rand()
  */
  static const int OIV_RAND_MAX = 32767;
  /**
  * Default epsilon value for coincidence and interval checking 
  */
  static const float OIV_DEF_MATH_HELPER_EPS = 1.e-6f;

  /**
   * Return true if A and B are equal. @B numFloat@b is the number of allowed
   * representable IEEE float between A and B. For example, there is no
   * number between 10000 and 10000.000977. So if numFloat=1 it will return true
   * for these two numbers.
   */
  INVENTORBASE_API SbBool floatIsEqual(float A, float B, unsigned int numFloat);
  template <typename T> inline T Max(T a, T b) { return (a>b)?a:b; }
  template <typename T> inline T Min(T a, T b) { return (a<b)?a:b; }

  /** shift the value of offset representable values considering value type.
   * If T is an integer type (char, short, etc...), it just return v + offset.
   * If T is a float type (float, double, ...) it return the offset'th representable
   * value starting from v. 
   * ex: shiftValue(10000.0f, 1) returns 10000.000977...
   */
  template <typename T> T shiftValue(T v, int offset) { return v + offset; }
  template <> INVENTORBASE_API float shiftValue(float v, int offset);
  template<> INVENTORBASE_API double shiftValue(double v, int offset);

  /** 
  * Clamps value to given range [minV, maxV]
  */
  template <typename T> inline T Clamp(T a, T minV, T maxV) { return Min(Max(minV, a), maxV); }

  /**
   * Return an integer multiple of 2^shift greater or equals to n
   */
  template <typename T> inline T alignToNextPowerOf2(T n, size_t shift)
  {
    if ( n &  ((T(1)<<shift)-1) )
      n = (n + (T(1)<<shift)) & ~((T(1)<<shift)-1);
    return n;
  }

  /**
   * Return the next power of 2 greater or equal to a
   */
  template <typename T> inline T getNextPow2(T a)
  {
    T pow = 1;
    while ( pow < a )
      pow *= 2;
    return pow;
  }

  /** Return the nearest upper multiple of some value.
   * for example nearestUpperMultipleOf<4>(7) return 8. 
   * Template because it let compiler to do some optimizations. */
  template <int N, typename T> 
  inline T nearestUpperMultipleOf(T v)
  {
    if ( v%N == 0 )
      return v;
    else
      return v + (N - (v%N));
  }

  /** Return the nearest lower multiple of some value.
   * for example nearestLowerMultipleOf<4>(7) return 4. 
   * Template because it let compiler to do some optimizations. */
  template <int N, typename T> 
  inline T nearestLowerMultipleOf(T v)
  {
    return v - (v%N);
  }

  /**
   * Return the next log of 2 greater or equal to a
   */
  INVENTORBASE_API int getNextLog2(int num);

  /**
   * Convert degrees to radians
   */
  inline float deg2Rad(float a)
  {
    return (a*float(M_PI))/180.f;
  }

  /**
   * Convert radians to degrees
   */
  inline float rad2Deg(float a)
  {
    return (a*180.f)/float(M_PI);
  }

  /** Round the double value to the nearest integer value.
  *
  * Examples:
  * roundToNearestInt(4.8) returns 5
  * roundToNearestInt(-7.4) returns -7
  *
  * This method always round by the lower value when there is 2 nearest integer values.
  * Example:
  * roundToNearestInt(0.5) returns 0
  * roundToNearestInt(-7.5) returns -8
  */
  inline double roundToNearestInt(double x)
  {
    return floor(x + 0.5);
  }

  /**
   * Returns true if the specified value is NaN.
   */
  INVENTORBASE_API inline bool isNaN(double a)
  {
#ifdef _WIN32
    return _isnan(a) != 0;
#else
	return std::isnan(a);
#endif
  }

  /**
   * Returns true if the value is a finite value (i.e. is is normal, subnormal or zero but
   * not infinite or NaN).
   * @{
   */
  INVENTORBASE_API inline bool isFinite( double value )
  {
    // Alternative to support all compilers
#if defined(_MSC_VER) && (_MSC_VER < 1800)
    return value == value &&
           value != std::numeric_limits<double>::infinity() &&
           value != -std::numeric_limits<double>::infinity();
#else
    return std::isfinite( value );
#endif // #if defined(_MSC_VER) && (_MSC_VER < 1800)
  }

  INVENTORBASE_API inline bool isFinite( float value )
  {
#if defined(_MSC_VER) && (_MSC_VER < 1800)
    return value == value &&
           value != std::numeric_limits<float>::infinity() &&
           value != -std::numeric_limits<float>::infinity();
#else
    return std::isfinite( value );
#endif // #if defined(_MSC_VER) && (_MSC_VER < 1800)
  }
  /**
   * @}
   */

  /**
   * Returns a pseudo-random integer between 0 and OIV_RAND_MAX.
   * The returned sequence will be the same on all machines.
   */
  INVENTORBASE_API int rand();

  /**
   * Set seed for a new sequence of pseudo-random integers to be returned by rand()
   */
  INVENTORBASE_API void srand(unsigned seed);

  /** 
  * Coincidence test using given tolerance
  */
  template< typename T >
  inline bool isCoinc( const T& x, const T& y, T tol = (T)OIV_DEF_MATH_HELPER_EPS )
  {
    return ( fabs( x - y ) <= tol );
  }

  /** Return absolute value of v */
  template<typename T> inline T abs( const T& v ) { return ::abs(v); }
  template<> inline long int abs( const long int& v ) { return ::labs(v); }
  template<> inline float abs( const float& v ) { return std::abs(v); }
  template<> inline double abs( const double& v ) { return std::abs(v); }

  /** 
  * Less than test (x < y) using given tolerance
  */
  template<typename T>
  inline bool isLessThan( const T& x, const T& y, T tol = (T)OIV_DEF_MATH_HELPER_EPS )
  {
    return ( x < y ) && !isCoinc( x, y, tol );
  }

  /** 
  * Greater than test (x > y) using given tolerance
  */
  template<typename T>
  inline bool isGreaterThan( const T& x, const T& y, T tol = (T)OIV_DEF_MATH_HELPER_EPS )
  {
    return ( x > y ) && !isCoinc( x, y, tol );
  }

  /**  
  * Less or equal than test (x <= y) using given tolerance
  */
  template<typename T>
  inline bool isLessOrEqualThan( const T& x, const T& y, T tol = (T)OIV_DEF_MATH_HELPER_EPS )
  {
    return ( x < y ) || isCoinc( x, y, tol );
  }

  /**  
  * Greater or equal than test (x >= y) using given tolerance
  */
  template<typename T>
  inline bool isGreaterOrEqualThan( const T& x, const T& y, T tol = (T)OIV_DEF_MATH_HELPER_EPS )
  {
    return ( x > y ) || isCoinc( x, y, tol );
  }

  /** 
  * Inside closed interval (including endpoints) using given tolerance test
  */
  template<typename T>
  inline bool checkRangeI( const T& x, const T& min, T max, T tol = (T)OIV_DEF_MATH_HELPER_EPS )
  {
    return ( isLessOrEqualThan( x, max, tol ) &&
             isGreaterOrEqualThan( x, min, tol ) );
  }

  /** 
  * Inside open interval (excluding endpoints) using given tolerance test
  */
  template<typename T>
  inline bool checkRangeE( const T& x, const T& min, T max, T tol = (T)OIV_DEF_MATH_HELPER_EPS )
  {
    return ( isLessThan( x, max, tol ) &&
             isGreaterThan( x, min, tol ) ) ;
  }

  /** 
  * Relative or absolute error without sign comparison
  */
  template <typename T>
  bool isCoincRelativeOrAbsolute( const T& A, const T& B, T maxRelativeError, T maxAbsoluteError )
  {
    long double  ABfabs(std::fabs((long double)(A - B)));
    if ( ABfabs < maxAbsoluteError )
      return true;
    long double Afabs( std::fabs((long double) A) );
    long double Bfabs( std::fabs((long double) B) );
    T relativeError = T( ABfabs / ( ( Bfabs > Afabs ) ? Bfabs : Afabs ) );
    return ( relativeError <= maxRelativeError ) ;
  }

  /**
  * Implements signum
  * return value in { -1, 0, +1 }
  */
  template <typename T>
  int sgn( const T& val )
  {
    return (T(0) < val) - (val < T(0));
  }

  /** Returns the maximum finite value representable by the numeric type T. */
  template <typename T> inline T rangeMax() { return std::numeric_limits<T>::max(); }

  /** Returns the lowest finite value representable by the numeric type T, 
   * that is, a finite value x such that there is no other finite value y where y < x. */
  template <typename T> inline T rangeMin() { return std::numeric_limits<T>::min(); }
  template <> inline float rangeMin<float>() { return -rangeMax<float>(); }
  template <> inline double rangeMin<double>() { return -rangeMax<double>(); }

  /** Return fractional part of given value.
   * ex: fract(23.15) return 0.15 */
  template <typename T> inline T fract(const T& /*value*/)
  {
    // default implementation is for integer type. Floating type are managed by specific template implementation.
    return 0;
  }
  template <> inline float fract(const float& value)
  {
    float intpart;
    return modff(value, &intpart);
  }
  template <> inline double fract(const double& value)
  {
    double intpart;
    return modf(value, &intpart);
  }
  template <> inline long double fract(const long double& value)
  {
    long double intpart;
    return modfl(value, &intpart);
  }

}

#endif


