/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (oct 2011)
**=======================================================================*/

#ifndef _SB_NAME_
#define _SB_NAME_

#ifndef HIDDEN_FROM_DOC

#include <Inventor/SbString.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SbNameEntry (internal to SB)
//
//  This is used to make lists of SbName instances.
//
//////////////////////////////////////////////////////////////////////////////


// The size of a chunk is about a page (4K) minus room for other stuff
// in the structure
SoINTERNAL class INVENTORBASE_API SbNameChunk
{
public:

  // Inserts string in table
  static const char* insert(const char *s);

  // Removes string from table
  static SbBool remove(const char *s);

  // Initializes/Unload SbNameEntry class
  static void initClass();
  static void exitClass();

  struct SbNameEntry
  {
    SbNameEntry* next;            // Pointer to next entry
    uint32_t hashValue;           // Its hash value
    
    struct SbNameEntryUsage
    {
      uint32_t _isStrDup : 1;     // String allocated in chunk (0) or elsewhere (1)
      uint32_t _refCount : 31;    // Place for ref counting names
    } usage;

    union SbNameEntryString       // Data depends on _isStrDup
    {
      char _const;          // _isStrDup == 0 : this is the string.
      char* _ptr;                 // _isStrDup == 1 : this is a pointer to the string.
    } nameString;

    const char* getString() const // Access to the string.
    {
      if (usage._isStrDup)
        return nameString._ptr;
      return (const char*)&(nameString._const);
    }
  };

private:
  SbNameChunk();
  ~SbNameChunk();

  char* mem;         // Chunk o'memory
  char* curByte;     // Current byte of chunk
  size_t bytesLeft;  // Bytes left in chunk
  SbNameChunk* next; // Pointer to next chunk
  
  static int nameTableSize;               // Number of buckets in name table
  static SbNameEntry** nameTable; // Array of name entries
  static SbNameChunk* chunk;              // Chunk of memory for string storage
  static int chunkSize;

  static SbNameEntry*& findEntry(const char *s); // Finds string entry in table
};

#endif // HIDDEN_FROM_DOC

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SbName
//
//  An SbName is a character string that is stored in a special table.
//  When a string is stored in this table, a pointer to the storage is
//  returned. Two identical strings will return the same pointer. This
//  means that comparison of two SbNames for equality can be
//  accomplished by comparing their pointers!
//
//////////////////////////////////////////////////////////////////////////////


/**
 * Character string stored in a hash table.
 *
 * @ingroup Basics
 *
 * @DESCRIPTION
 *   This class of strings stores the string in a hash table. It is used by the Open
 *   Inventor toolkit for keywords and other unique names. It is not recommended for
 *   general use (only in the context of Open Inventor objects). When a string is
 *   stored in this table, a pointer to the storage is returned. Two identical
 *   strings will return the same pointer. This means that comparison of two SbNames
 *   for equality can be accomplished by comparing their identifiers. SbNames are
 *   used for strings which are expected to show up frequently, such as node names.
 *
 *   @B Note: @b SbName does not support Unicode strings.
 *
 * @SEE_ALSO
 *    SbString,
 *    SoBase::setName()
 *
 * [OIV-WRAPPER-NO-WRAP]
 */
class INVENTORBASE_API SbName
{
 public:
  /**
   * Constructor.
   */
  SbName() { string = SbNameChunk::insert(""); }

  /**
   * Constructor.
   */
  SbName(const char *s) { string = SbNameChunk::insert(s); }

  /**
   * Constructor.
   */
  SbName(const SbString &s) { string = SbNameChunk::insert(s.toLatin1()); }

  /**
   * Constructor.
   */
  SbName(const SbName &n) { string = n.string; }

  /**
   * Destructor.
   */
  ~SbName() {};

  /**
   * Returns pointer to the character string.
   */
  const char* getString() const
  {
    return string;
  }

  /**
   * Returns length of string.
   */
  int getLength() const
  {
    return static_cast<int>(strlen(string));
  }

  /**
   * Returns TRUE if given character is a legal starting character for an identifier.
   */
  static SbBool isIdentStartChar(char c);

  /**
   * Returns TRUE if given character is a legal nonstarting character for an
   * identifier.
   */
  static SbBool isIdentChar(char c);

  /**
   * Returns TRUE if given character is a legal starting character
   * for an SoBase's name.
   */
  static SbBool isBaseNameStartChar(char c);

  /**
   * Returns TRUE if given character is a legal nonstarting
   * character for an SoBase's name.
   */
  static SbBool isBaseNameChar(char c);

  /**
   * Unary "not" operator; returns TRUE if string is empty ("").
   */
  int operator !() const
  {
    return ( string[0] == '\0');
  }

  /**
   * Equality operator for SbName/char* and SbName/SbName comparison.
   */
  INVENTORBASE_API friend int operator ==(const SbName &n, const char *s)
  {
    return (n.string[0] == s[0] && ! strcmp(n.string, s));
  }

  /**
   * Equality operator for SbName/char* and SbName/SbName comparison.
   */
  INVENTORBASE_API friend int operator ==(const char *s, const SbName &n)
  {
    return (n.string[0] == s[0] && ! strcmp(n.string, s));
  }

  /**
   * Equality operator for SbName/char* and SbName/SbName comparison.
   */
  INVENTORBASE_API friend int operator ==(const SbName &n1, const SbName &n2)
  {
    return (n1.string[0] == n2.string[0] && ! strcmp(n1.string, n2.string));
  }

  /**
   * Inequality operator for SbName/char* and SbName/SbName comparison.
   */
  INVENTORBASE_API friend int operator !=(const SbName &n, const char *s)
  {
    return ! (n.string[0] == s[0] && ! strcmp(n.string, s));
  }

  /**
   * Inequality operator for SbName/char* and SbName/SbName comparison.
   */
  INVENTORBASE_API friend int operator !=(const char *s, const SbName &n)
  { return ! (n.string[0] == s[0] && ! strcmp(n.string, s)); }

  /**
   * Inequality operator for SbName/char* and SbName/SbName comparison.
   */
  INVENTORBASE_API friend int operator !=(const SbName &n1, const SbName &n2)
  { return ! (n1.string[0] == n2.string[0] && ! strcmp(n1.string, n2.string)); }

  /**
   * "less than" relational operator for SbName / SbName that returns
   * TRUE if the first operand is less than the second,
   * FALSE otherwise.
   */
  INVENTORBASE_API friend bool operator < (const SbName &n1, const SbName &n2)
  {
    return (strcmp(n1.string, n2.string) < 0);
  }

  /**
   * Writes the SbName to the specified output stream.
   */
  INVENTORBASE_API friend std::ostream& operator << (std::ostream& os, const SbName& n)
  {
    return os << n.getString();
  }

SoINTERNAL public:
  static void enableUseNameCompat( const SbBool enable )
  { s_useNameCompat = enable; }

  static SbBool useNameCompat()
  { return s_useNameCompat; }

 private:
  // Name string storage
  const char* string;

  // Flag to control use of valid names
  static SbBool s_useNameCompat;
};

#endif // _SB_NAME_


