/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SB_PLIST_
#define _SB_PLIST_

#include <Inventor/SbBase.h>
#include <Inventor/STL/cassert>

class SbVec3f;

/** 
 * List of generic (void *) pointers.
 * 
 * @ingroup Basics
 * 
 * @DESCRIPTION 
 *   This class manages a dynamic list of generic void * pointers. This class allows
 *   random access, insertion, and removal of pointers.
 *   Changes in SbPList can be detected by attaching an SoListSensor to this list.
 *
 * @SEE_ALSO
 *   SoListSensor
 *
 * 
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
class INVENTORBASE_API SbPList
{
 public:

  /**
   * Default constructor.  The list is initially empty.
   */
  SbPList();

  /**
   * Constructor. @B initSize @b specifies an initial size for the
   * list, which is useful as an optimization if you can estimate the length of the
   * list before you construct it.
   */
  SbPList(int initSize);

  /**
   * Constructor. It returns a copy of the specified SbPList.
   */
  SbPList(const SbPList &pl);

  /**
   * Destructor.
   */
  virtual ~SbPList();

  /**
   * Adds given pointer to end of list.
   */
  void append(void* ptr);

  /**
   * Returns index of given pointer in list, or -1 if not found.
   */
  int find(const void* ptr) const;

  /**
   * Inserts given pointer in list before pointer with given index.
   */
  void insert(void* ptr, int addBefore);

  /**
   * Removes pointer with given index.
   */
  virtual void remove(int which);

  /**
   * Returns number of pointers in list.
   */
  inline int getLength() const
  { return nPtrs; }

  /** 
   * Removes all pointers after one with given index, inclusive
   */
  virtual void truncate(int start);

  /**
   * Copies a list.
   */
  void copy(const SbPList &pl);

  /**
   * Assignment operator: copies list into this list.
   */
  SbPList &operator =(const SbPList &pl);

  /**
   * Returns pointer with given index.
   * If requested index does not exists, list is grown to return a valid pointer.
   */
  void *&operator [](const int index) const;

  /**
   * Equality operator.
   */
  int operator ==(const SbPList &pl) const;

  /**
   * Inequality operator.
   */
  int operator !=(const SbPList &pl) const;

  /**
  * Swaps element having index1 with the one having index2.
  */
  void swap( int index1, int index2 );

 SoINTERNAL public:

  /** Internal versions of operator[] that do not check for bounds */
  void *get(int i) const;
  void set(int i, void *j);
  
  void **getArray()
 { return ptrs; }

  // internal public call to expand function (see below) in order to reserve more memory in the list.
  // same name than stl reserve function
  void reserve(int size)
  { expand(size); }

protected:
  // Number of pointers used
  int nPtrs;

  // There are three(!) methods to expand the list.  grow() is used
  // when space is dynamically created, and needs to be initialized
  // to NULL:
  void grow(int max) const;

  // setSize is used by grow and in other places internally where we
  // know that nothing needs to be initialized to NULL.
  void setSize(int size);

 private:

  // NOTE: this must be called only if the number of elements in the two
  // lists is the same, otherwise badness could result
  int compare(const SbPList &pl) const;

  // The collection of pointers
  void **ptrs;

  // Number of pointers allocated
  int ptrsSize;

  // expand is the lowest level routine.  It just reallocates the
  // array and copies over the old values.
  void expand(int size);

};

inline void* SbPList::get(int i) const
{
  assert(i>=0);
  return ((i>=0)&&(i<nPtrs))?ptrs[i]:NULL;
}

inline void SbPList::truncate(int start)
{
  assert(start>=0);
  nPtrs = (start>=0)?start:0;
}

inline void SbPList::append(void * ptr)
{
  if (nPtrs + 1 > ptrsSize)
    expand(nPtrs + 1);
  ptrs[nPtrs++] = ptr;
}

inline void*& SbPList::operator[](const int index) const
{
  assert(index>=0);
  if (index>= nPtrs)
    grow(index);
  return ptrs[index];
}


// Keep include for compatibility
#include <Inventor/lists/SbIntList.h>
#include <Inventor/lists/SbVec3fList.h>
#include <Inventor/lists/SbStringList.h>

#endif
