/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author : VSG (MMM 2015)
**=======================================================================*/

#ifndef SB_RENDER_ENGINE_MODE_H_
#define SB_RENDER_ENGINE_MODE_H_

#include <Inventor/SbBase.h>

/**
 * @VSGEXT Render engine mode.
 *
 * @ingroup Basics
 *
 * @DESCRIPTION
 * The render engine mode class indicates how a node performs its rendering. For example, the node can be using OpenGL
 * calls or Open Inventor calls in order to achieve this task.
 *
 * By default, a custom node is assumed to use the OpenGL compatibility-profile which means that it is allowed
 * to use all OpenGL features even the ones that were deprecated in recent versions. If a node is only using
 * core-profile OpenGL calls, it should give this information to the renderer using this enum.
 *
 * However, it is strongly encouraged to avoid calling directly OpenGL methods to perform the rendering of a node and
 * rely on Open Inventor instead. Using only Open Inventor calls is the optimal way as it allows harnessing a wide range
 * of optimizations that are not available when using direct OpenGL calls.
 *
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
class INVENTOR_API SbRenderEngineMode
{
public:
  /** Default Constructor. */
  SbRenderEngineMode()
    :m_renderMode(OIV_UNKNOWN_RENDERING_MODE), m_isChildrenInherited(false)
  {}

  /** 
   * Defines the list of supported render mode for a custom node.
   */
  enum RenderMode
  {
    /** Internally used to detect that a node did not set explicitly its RenderEngineMode by calling setRenderEngineMode method (This is the default).
     * If parent class has called setChildrenInherit(true), then this node type will inherit it.
     * Else a warning will be issued and OIV_OPENINVENTOR_RENDERING will be used.
     */
    OIV_UNKNOWN_RENDERING_MODE,

    /** The node uses only Open Inventor calls to perform its rendering (optimal way) */
    OIV_OPENINVENTOR_RENDERING,

    /** The node relies on OpenGL fixed-function pipeline & deprecated features (OpenGL compatibility profile) */
    OIV_OPENGL_COMPATIBILITY_RENDERING,

    /** The node uses OpenGL but only shader and modern features (OpenGL 3.3+ core profile) */
    OIV_OPENGL_CORE_RENDERING
  };

  /** Set the RenderMode. Default is OIV_UNKNOWN_RENDERING_MODE.
   * \if_cpp
   * If you specify your node type in the initClass() method, this method should be
   * called before calling the SO_NODE_INIT_CLASS macro
   * \endif
   */
  void setRenderMode(RenderMode renderMode)
  {
    m_renderMode = renderMode;
  }

  /** Returns the current RenderMode. */
  RenderMode getRenderMode() const
  {
    if ( isNotDefined() )
      return OIV_OPENINVENTOR_RENDERING;
    return m_renderMode;
  }

  /** Indicates if any derived node should inherit this RenderMode. Default is false */
  void setChildrenInherit(bool isChildrenInherited)
  {
    m_isChildrenInherited = isChildrenInherited;
  }

  /** Returns true is derived node should inherit this RenderMode. */
  bool isChildrenInherit() const
  {
    return m_isChildrenInherited;
  }

  /** Return true if the RenderMode has not be defined explicilty. */
  bool isNotDefined() const
  {
    return (m_renderMode == OIV_UNKNOWN_RENDERING_MODE);
  }

  /**
   * Enable or disable RenderEngineMode warning messages. Default is true.
   * Default value can be changed by using OIV_OPENGL_COMPATIBILITY_WARNING environment variable.
   */
  static void enableWarnings(bool flag);

  /**
   * Post a warning related to Open Inventor 10 OpenGL support.
   * SbRenderEngineMode has not been set before SO__NODE_INIT_CLASS macro by a custom node.
   */
  static void postInitWarning(const char *methodName);

private:
  RenderMode m_renderMode;
  bool m_isChildrenInherited;
  static int s_isWarningsEnabled;
};


#endif // SB_RENDER_ENGINE_MODE_H_
