/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SB_VIEWPORT_REGION_
#define  _SB_VIEWPORT_REGION_

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SbViewportRegion
//
//  Class that represents the active viewport region in a display
//  window. It contains the screen-space size of the window as well as
//  the origin and size of the viewport within the window. By default,
//  the viewport is the same as the full window. Methods allow the
//  viewport to be set either in terms of screen space pixels or as
//  normalized coordinates, where (0,0) is the lower-left corner of
//  the window and (1,1) is the upper-right corner.
//
//////////////////////////////////////////////////////////////////////////////

#include <Inventor/SbLinear.h>

/**
 * Class for representing a viewport.
 * 
 * @ingroup Basics
 * 
 * @DESCRIPTION
 *   This class represents the active viewport region in a display window. It
 *   contains the screen-space (pixels) size of the window as well as the origin and size of
 *   the viewport within the window. The viewport origin and size is stored both as pixel
 *   values and as normalized values (fraction of the window).
 *
 *   Initially the viewport is the same as the full
 *   window. Methods allow the viewport to be set either in terms of screen-space
 *   pixels or as normalized coordinates, where (0,0) is the lower-left corner of the
 *   window and (1,1) is the upper-right corner.
 * 
 *   Remember that getWindowSize() returns the size of the window, @I not@i the viewport.
 *   Always call getViewportOriginPixels() and getViewportSizePixels() if you need the
 *   actual viewport in pixels.
 *
 *   Also remember:
 *   - To normalize a point: use the viewport origin and size.
 *   - To normalize a distance: use only the viewport size.
 *
 *  The #normalize() method is convenient for normalizing a point, but note that it
 *  normalizes to the range 0..1 and some Open Inventor classes expect values normalized
 *  to the the range -1..1.  Currently you must do this normalization yourself.
 * 
 * @SEE_ALSO
 *    SbVec2f,
 *    SbVec2s
 * 
 * 
 */
class INVENTORBASE_API SbViewportRegion {

 public:
  /**
   * Default constructor. Sets window size and viewport to default 100 x 100 pixels.
   */
  SbViewportRegion();
  
  /**
   * Constructor that takes width and a height in pixels.
   * Sets the window and the viewport.
   */
  SbViewportRegion(short width, short height, float newScale = 1.0);

  /**
   * Constructor that takes width and a height in pixels.
   * Sets the window and the viewport.
   */
  SbViewportRegion(SbVec2s   winSize, float newScale = 1.0);

  /**
   * Constructor that takes width and a height in pixels.
   * Sets the window and the viewport.
   */
  SbViewportRegion(SbVec2i32 winSize, float newScale = 1.0);
  
  /**
   * Constructor that takes a viewport region.
   */
  SbViewportRegion(const SbViewportRegion &vpReg);
  
  /**
   * Changes window size to given width and height in pixels.
   *
   * If viewport has not been explicitly set by the application, the viewport
   * is also changed to the specified width and height.  If the viewport has
   * been explicitly set, then the stored normalized values are applied to the
   * new window to compute a new viewport origin and size in pixels.
   */
  void setWindowSize(short width, short height)
    { setWindowSize(SbVec2i32(width, height)); }

  /**
   * Changes window size to given width and height in pixels, given as SbVec2s.
   *
   * If viewport has not been explicitly set by the application, the viewport
   * is also changed to the specified width and height.  If the viewport has
   * been explicitly set, then the stored normalized values are applied to the
   * new window to compute a new viewport origin and size in pixels.
   */
  void setWindowSize(SbVec2s winSize);

  /**
   * Changes window size to given width and height in pixels, given as SbVec2i32.
   *
   * If viewport has not been explicitly set by the application, the viewport
   * is also changed to the specified width and height.  If the viewport has
   * been explicitly set, then the stored normalized values are applied to the
   * new window to compute a new viewport origin and size in pixels.
   */
  void setWindowSize(SbVec2i32 winSize);
  
  /**
   * Sets viewport to given region, specified as normalized coordinates relative
   * to the current window size, where (0,0) is the lower-left corner and (1,1) is the upper-right.
   *
   * The normalized values are applied to the current window size to compute a
   * new viewport origin and size in pixels.
   */
  void setViewport(float left, float bottom, float width, float height)
    { setViewport(SbVec2f(left, bottom), SbVec2f(width, height)); }
  
  /**
   * Sets viewport to given region, specified as normalized coordinates relative
   * to the current window size, where (0,0) is the lower-left corner and (1,1) is the upper-right.
   *
   * The viewport origin and size in pixels are recomputed.
   */
  void setViewport(SbVec2f origin, SbVec2f size);
  
  /**
   * Sets viewport to given region, specified as pixel coordinates in window: (0,0)
   * is the lower-left corner.
   *
   * The normalized viewport origin and size are recomputed.
   * The specified values are not clamped to the window size.
   */
  void setViewportPixels(short left, short bottom, short width, short height)
    { setViewportPixels(SbVec2i32(left, bottom), SbVec2i32(width, height)); }

  /**
   * Sets viewport to region with given origin (lower-left corner) and size, given as
   * pixel coordinates.
   *
   * The normalized viewport origin and size are recomputed.
   * The specified values are not clamped to the window size.
   */
  void setViewportPixels(SbVec2s origin, SbVec2s size);

  /**
   * Sets viewport to region with given origin (lower-left corner) and size, given as
   * pixel coordinates.
   *
   * The normalized viewport origin and size are recomputed.
   * The specified values are not clamped to the window size.
   */
  void setViewportPixels(SbVec2i32 origin, SbVec2i32 size);
  
  /**
   * Returns window size in pixels.
   *
   * NOTE: By default this is also the viewport size in pixels, but in general
   * this is @I not@i the viewport size. Use getViewportSizePixels().
   */
  const SbVec2s &getWindowSize() const;
 
  /**
   * Returns window size in pixels.
   *
   * NOTE: By default this is also the viewport size in pixels, but in general
   * this is @I not@i the viewport size. Use getViewportSizePixels().
   */
  const SbVec2i32 &getWindowSize_i32() const;

  /**
   * Returns viewport origin in 0..1 normalized coordinates relative to the
   * current window size.
   */
  const SbVec2f &getViewportOrigin() const
    { return vpOrigin; }

  /**
   * Returns viewport origin in pixels.
   */
  const SbVec2s &getViewportOriginPixels() const;

  /**
   * Returns viewport origin in pixels.
   */
  const SbVec2i32 &getViewportOriginPixels_i32() const;

  /**
   * Returns viewport size in 0..1 normalized coordinates relative to the
   * current window size.
   */
  const SbVec2f &getViewportSize() const
    { return vpSize; }
  
  /**
   * Returns viewport size in pixels.
   */
  const SbVec2s &getViewportSizePixels() const;

  /**
   * Returns viewport size in pixels.
   */
  const SbVec2i32 &getViewportSizePixels_i32() const;

  /**
   * Returns aspect ratio (width/height) of viewport.
   */
  float getViewportAspectRatio() const;

  /**
   * Scales viewport within window to be the given ratio of its current width, 
   * leaving the resulting viewport centered about the same point as the
   * current one.
   */
  void scaleWidth(float ratio);

  /**
   * Scales viewport within window to be the given ratio of its current
   * height, leaving the resulting viewport centered about the same point as the
   * current one.
   */
  void scaleHeight(float ratio);
  
  /**
   * Sets the pixel-per-inch ratio for the display device the viewport is
   * part of. The default value is 72 (1 pixel per printer's point).
   */
  void setPixelsPerInch(float ppi)
    { pixelsPerInch = ppi; }

  /**
   * Returns the pixel-per-inch ratio for the display device the viewport is
   * part of.
   */
  float getPixelsPerInch() const
    { return pixelsPerInch; }
  
  /**
   * Convenience function that returns number of pixels per printer's point.
   */
  float getPixelsPerPoint() const
    { return (pixelsPerInch / s_historicalPixelPerInch); }

  /**
  * Compute the 0..1 normalized coordinates of the specified pixel position,
  * relative to the current viewport (not window).
  *
  * @param position in pixel coordinates
  * @return position in normalized coordinates (0 <= x,y <= 1)
  */
  SbVec2f normalize( const SbVec2s& position ) const;

  /**
   * Equality comparison operator.
   */
  INVENTORBASE_API friend int operator ==(const SbViewportRegion &reg1,
                         const SbViewportRegion &reg2);

  /**
   * Inequality comparison operator.
   */
  INVENTORBASE_API friend int operator !=( const SbViewportRegion &reg1, const SbViewportRegion &reg2 )
  {
    return !operator==( reg1, reg2 );
  }

 SoINTERNAL public:

  /**
   * Constructor that takes a viewport size and window size in pixels.
   * Each other contructor consider that these two vectors have the same value.
   * This is an issue, for example, for tile rendering.
   */
  SbViewportRegion(SbVec2s viewSize, SbVec2s winSize, float newScale = 1.0);

  void setViewportScale(float vpScale)
    { viewportScale = vpScale; }

  float getViewportScale() const
    {return viewportScale; }
 
  static const float s_historicalPixelPerInch;
  static float s_defaultPixelsPerInch;
  static void setDefaultPixelsPerInch(float ppi)
    { s_defaultPixelsPerInch = ppi; }
  
 private:
  SbVec2i32 windowSize_i32;  // Window size in pixels
  SbVec2s   windowSize;      // Window size in pixels
  SbVec2f   vpOrigin;        // Viewport origin, normalized
  SbVec2s   vpOriginPix;     // Viewport origin in pixels
  SbVec2i32 vpOriginPix_i32; // Viewport origin in pixels
  SbVec2f   vpSize;          // Viewport size, normalized
  SbVec2s   vpSizePix;       // Viewport size in pixels
  SbVec2i32 vpSizePix_i32;   // Viewport size in pixels
  SbBool    vpSet;           // TRUE if viewport was set explicitly
  float     pixelsPerInch;   // Number of pixels per inch
  float     viewportScale;   // Viewport scale (used for remote rendering)
  
  // Sets viewport to be full window
  void setFullViewport();
  
  // Adjusts viewport pixel size based on new window size or new viewport
  void adjustViewport();
};
 
/**
 * Class to access a MeshViz view.
 * 
 * @ingroup Basics
 * 
 * @DESCRIPTION
 *   This class gives access from the Inventor kernel to the external MeshViz nodes 
 *   managing the views (i.e PoBaseView::viewportOrigin or 
 *   PoBaseView::viewportSize and its derived class fields).
 * 
 * @SEE_ALSO
 *    PoBaseView.
 * 
 * [OIV-WRAPPER-CLASS CUSTOM_CODE]
 */
class INVENTORBASE_API SoGetView {
 public:
  /**
   * Default constructor.
   */
  SoGetView();
  virtual ~SoGetView() {};

  /**
   * Returns TRUE if SoGetView functionalities are enabled, FALSE otherwise.
   * By default, returns FALSE.
   */
  virtual SbBool getViewportIsEnabled() const;

  /**
   * Returns SbViewportRegion origin. Default value = (0,0).
   */
  virtual const SbVec2f getViewportOrigin() const;

  /**
   * Returns SbViewportRegion size. Default value = (1,1).
   */
  virtual const SbVec2f getViewportSize() const;
};

#endif /* _SB_VIEWPORT_REGION_ */

