/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SO_DB_
#define _SO_DB_

#include <Inventor/SoType.h>
#include <Inventor/SbTime.h>
#include <Inventor/threads/SbThread.h>
#include <Inventor/threads/SbThreadLocalStorage.h>

#ifndef LIBRARYBUILD
// uneeded header be kept for application built compatibility
#include <Inventor/sensors/SoSensorManager.h>
#include <Inventor/SbPList.h>
#include <Inventor/SoPreferences.h>
#endif

class SbPList;
class SoBase;
class SoSFRealTime;
class SoGroup;
class SoSeparator;
class SoInput;
class SoPath;
class SoNode;
class SoField;
class SbColor;
class SoGlobalField;
class SoSensorManager;
class SoTimerSensor;
class SoSensor;
class SoSystemTimer;

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <Inventor/SoInventorLibName.h>

#include <Inventor/SoModule.h>
SO_MODULE_HEADER(SoInventor, __INVENTORDLL)

#if defined(_WIN32)
// Detect compilation that uses any form of /clr
#if ((_MANAGED == 1) || (_M_CEE == 1))
#if defined(SO_MODULE_CHECK)
static SoModuleCheck myModule((bool)_SECURE_SCL);
#endif
#elif !defined HIDDEN_FROM_DOC && !defined(LIBRARYBUILD)
static SoModuleCheck myModule((bool)_SECURE_SCL);
#endif /* (_MANAGED == 1) || (_M_CEE == 1) */
#endif /* _WIN32 */

#if defined SO_VERSION_DEPRECATED_ERROR && !defined(LIBRARYBUILD)
static SoModuleCheck soVersionDeprecatedErrorCheck(SO_VERSION_DEPRECATED_ERROR);
#endif

/**
 * @memberof SoDB
*/
typedef void SoDBHeaderCB(void *userData, SoInput *in);

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoDB
//
//  Inventor database class.  This class maintains all the global
//  information necessary for reading and writing, naming classes, and
//  so on. All public methods on this class are static - users should
//  never explicitly create instances of this class. Initialization of
//  the database class causes a global instance to be created.
//
//////////////////////////////////////////////////////////////////////////////

/**
* Scene graph database class.
*
* @ingroup General
*
* @DESCRIPTION
*   The SoDB class holds all scene graphs, each representing a 3D scene used by an
*   application. A scene graph is a collection of SoNode objects which come in
*   several varieties (see SoNode). 
* \if_cpp
*   Application programs must initialize the
*   database by calling SoDB::init() before calling any other database routines and
*   before constructing any nodes, paths, functions, or actions. Note that
*   SoDB::init() is called by SoInteraction::init(),
*   SoNodeKit::init(), and SoWin::init(), so if you are calling any of
*   these methods, you do not need to call SoDB::init() directly. All methods on
*   this class are static.
*
*   Application programs should close and cleanup the database by calling
*   SoDB::finish() after deleting all explicitly created Open Inventor objects.
*   Note that in when using C++ API, "reference counted" objects like nodes and
*   paths cannot be explicitly deleted.
* \endif
*
*  All methods in this class are static.
*
*   @B Data Files: @b 
*
*   SoDB also provides convenient methods for reading data files, for example:
* \if_cpp
* \par 
*   \code
*      SbString filename = "$OIVHOME/data/models/bird.iv";
*      SoSeparator* rootSep = NULL;
*      SoInput input;
*      if (input.openFile( filename )) {
*        rootSep = SoDB::readAll( &in );
*        input.closeFile();
*      }
*   \endcode
* \endif
* \if_dotnet
* \par 
*   \code
*     String filename = "$OIVHOME/data/models/bird.iv";
*     SoSeparator rootSep = null;
*     SoInput input = new SoInput();
*     if (input.OpenFile(filename))
*     {
*         rootSep = SoDB.ReadAll( input );
*         input.CloseFile();
*     }
*   \endcode
* \endif
* \if_java
* \par 
*   \code
*     String filename = "$OIVHOME/data/models/bird.iv";
*     SoSeparator rootSep = null;
*     SoInput input = new SoInput();
*     if (input.openFile(filename))
*     {
*         rootSep = SoDB.readAll(input);
*         input.closeFile();
*     }
*   \endcode
* \endif
*
*   Open Inventor has built-in support for reading the following file formats:
*   - Open Inventor
*   - VRML/X3D
*   - DXF
*   - STL (see SoSTLFileFormat and SoSTLInputReader)
*
*   Open Inventor also supports many standard CAD file formats.
*   See SoCADInputReader for the current list.  Also see SoCADFileFormat.
* 
* \if_cpp
*   @B Multithread Programming: @b 
*
*   Since Open Inventor 10 in a multithreaded program, only the main thread has to call SoDB::init.
*   Note that it is still valid to call init/finish in each threads.
* \endif

* 
* @SEE_ALSO
*    SoBase,
*    SoNode,
*    SoEngine,
*    SoField,
*    SoInput,
*    SoFile,
*    SoPath,
*    SoOneShotSensor,
*    SoDataSensor,
*    SoWin
*/
class INVENTOR_API SoDB {

 public:


   /** Render cache modes */
  enum SoDEPRECATED_ENUM(10300,"RenderCacheMode is no longer used.")
  RenderCacheMode {
    /**
     *  Display lists are created with COMPILE mode.
     */
    COMPILE,
    /**
     *  Display lists are created with COMPILE_AND_EXECUTE mode.
     */
    COMPILE_AND_EXECUTE,
    /**
     *  Automatically choose mode depending on the hardware. (Default).
     */
    AUTO_RENDER_CACHE
  };

  /**
   * Initializes the database. This must be called before calling any other database
   * routines, including the construction of any nodes, paths, engines, or actions.
   * [OIV-WRAPPER-NOT-WRAP]
   */
  static void init();

  /**
   * Frees Open Inventor's internal static memory
   * allocations. This avoids false positives from memory
   * leak checking software. We recommend calling this method
   * and it should be the last Open Inventor method called by
   * the application.
   *
   * @B Note@b : Open Inventor objects must be destroyed before you call this method.
   * See SoXt::finish() method for additional info.
   * [OIV-WRAPPER-NOT-WRAP]
   */
  static void finish();

#if SoDEPRECATED_BEGIN(10000)
  /**
   * Initializes the database.
   * This must be called before calling any other database routines, including the construction of
   * any nodes, paths, engines, or actions. This method calls SoDB::init.
   *
   * [OIV-WRAPPER-CUSTOM-CODE]
   */
  SoDEPRECATED_METHOD(10000, "Use SoDB::init() instead.")
  static void threadInit();
#endif /** @DEPRECATED_END */

  /**
   * The system dependent timer allows Inventor to manage its sensor queue (for timers, animation)
   * when creating custom viewers.
   * Typically, this method could be called in the constructor of the first interactive render area created.
   * Calling this method more than once, does nothing.
   * \if_cpp The SoSystemTimer instance is reference counted and
   * will be deleted when its reference count reaches 0. \endif
   * \if_java This method is mandatory to create viewers based on viewer components like SoRenderAreaCore. \endif
   * [OIVJAVA-WRAPPER HELPER_END{onSetSystemTimer(timer)}]
   */
  static void setSystemTimer(SoSystemTimer* timer);

  /**
  * \if_java
  * Reads a graph from the file specified by the given SoInput, returning the resulting
  * root node.
  * \else
  * Reads a graph from the file specified by the given SoInput, returning the resulting
  * root node in rootNode.
  * \endif
  * There is an alternate read routine that returns the resulting path.
  * The programmer is responsible for determining which routine to use, based
  * on the contents of the input.
  * \if_java
  * These routines return null if any error occurred during reading.
  * \else
  * These routines return FALSE if any error occurred during reading.
  * \endif
  *
  * If the passed SoInput was used to open a file and the name of the file contains a
  * directory, SoDB automatically adds the directory to the end of the current
  * directory search path in the SoInput. This means that nested files named in
  * SoFile nodes may be found relative to that directory. The directory is removed
  * from the search path when reading is complete.
  */
  static SbBool read(SoInput *in, SoNode *&rootNode);

  /**
  * \if_java
  * Reads a graph from the file specified by the given SoInput, returning
  * the resulting path.
  * \else
  * Reads a graph from the file specified by the given SoInput, returning
  * the resulting path in path.
  * \endif
  * There is an alternate read routine
  * that returns the resulting root node.
  * The programmer is responsible for determining which routine to use, based
  * on the contents of the input.
  * \if_java
  * These routines return null if any error occurred during reading.
  * \else
  * These routines return FALSE if any error occurred during reading.
  * \endif
  *
  * If the passed SoInput was used to open a file and the name of the file contains a
  * directory, SoDB automatically adds the directory to the end of the current
  * directory search path in the SoInput. This means that nested files named in
  * SoFile nodes may be found relative to that directory. The directory is removed
  * from the search path when reading is complete.
  * [OIVJAVA-WRAPPER NAME{readPath}]
  */
  static SbBool read(SoInput *in, SoPath *&path);

  /**
  * Reads all graphs and paths from the file specified by the given SoInput. If
  * there is only one graph in the file and its root is an SoSeparator,
  * the root is returned. In all other cases, this creates an SoSeparator, adds the
  * root nodes of all graphs read as children of it, and returns it.
  * This returns NULL on error. This processes directory paths in the same way as
  * the other reading routines.
  *
  * Reading a DXF file into Open Inventor is just
  * like reading in an Open Inventor format file. Open Inventor
  * will open the file and automatically detect that it is a
  * DXF file. Optionally, you can tell Open Inventor
  * explicitly that the file type is DXF. See SoInput for details about
  * support of the DXF format.
  *
  */
  static SoSeparator *readAll(SoInput *in);

  /**
   * Registers the given string as a valid header for input files. The string must be
   * 80 characters or less, and start with the comment character '#'. If the passed
   * @B isBinary @b flag is true, any file with this header will be read as a binary
   * file. Usually, a user-defined header represents a file format that is a superset
   * of the Open Inventor file format. The @B ivVersion @b number indicates which
   * Open Inventor file version this header corresponds to. The user-defined callback
   * functions @B preCB @b and @B postCB @b are called before and after a file with
   * this header is read. The @B userData @b is passed to both callback functions.
   * The method returns TRUE if the header is successfully registered. Note, nothing
   * prevents you from registering the same string multiple times.
   */
  static SbBool registerHeader(const SbString &headerString,
                               SbBool isBinary,
                               float ivVersion,
                               SoDBHeaderCB *preCB,
                               SoDBHeaderCB *postCB,
                               void *userData = NULL);
  /**
  * Passes back the data registered with the given header string, including the flag
  * specifying whether the string is for a binary file, pointers to the callback
  * functions invoked before and after reading the file, and a pointer to the user
  * data passed to the callback functions. If the given header string does not match
  * any of the registered headers, and the @B substringOK @b flag is TRUE, then the
  * method will search for a registered header that is a substring of the given
  * string. The method returns TRUE if a matching registered header, or subheader,
  * was found.
  * [OIVJAVA-WRAPPER PACK{HeaderData}]
  */
  static SbBool getHeaderData(const SbString &string,
                              SbBool &isBinary,
                              float &ivVersion,
                              SoDBHeaderCB *&preCB,
                              SoDBHeaderCB *&postCB,
                              void *&userData,
                              SbBool substringOK = FALSE);
  /**
   * Returns the number of valid headers, including standard Open Inventor headers,
   * and user-registered headers. See #registerHeader().
   */
  static int getNumHeaders();

  /**
   * Returns the i'th header.
   * See #registerHeader().
   */
  static SbString getHeaderString(int i);

  /**
   * This returns TRUE if the given character string is one of the valid Open
   * Inventor file headers, (e.g., "#Inventor V2.0 binary"), or if the string has been
   * registered as a valid header through the registerHeader() method.
   */
  static SbBool isValidHeader(const char *testString);

  /**
  * The database maintains a namespace for global fields, making sure that there is
  * at most one instance of a global field with any given name in the database. This
  * routine is used to create new global fields. If there is no global field with
  * the given name, it will create a new global field with the given name and type.
  * If there is already a global field with the given name and type, it will return
  * it. If there is already a global field with the given name but a different type,
  * this returns NULL.
  *
  * All global fields must be derived from SoField; typically the result of this
  * routine is cast into the appropriate type. For example:
  * \if_cpp
  *  \code
  *  SoSFInt32 *intField = (SoSFInt32 *)SoDB::createGlobalField("Frame", SoSFInt32::getClassTypeId());
  *  \endcode
  * \endif
  * \if_dotnet
  *  \code
  *  SoSFInt32 intField = (SoSFInt32)SoDB.CreateGlobalField("Frame", typeof(SoSFInt32));
  *  \endcode
  * \endif
  * \if_java
  *  \code
  *  SoSFInt32 intField = (SoSFInt32)SoDB.createGlobalField("Frame", SoSFInt32.class);
  *  \endcode
  * \endif
  */
  static SoField *createGlobalField(const SbName &name, SoType type);

  /**
   * Returns the global field with the given name, or NULL if there is none. 
   * See #createGlobalField(). The type of the field may be checked using the SoField::isOfType(),
   * SoField::getClassTypeId(), and SoField::getTypeId() methods.
   */
  static SoField *getGlobalField(const SbName &name);

  /**
   * Renames the global field named oldName. 
   * Renaming a global field to an empty name
   * ("") deletes it. If there is already a global field with the new name, that
   * field will be deleted (the getGlobalField() method can be used to guard
   * against this). See #createGlobalField().
   */
  static void renameGlobalField(const SbName &oldName, const SbName &newName);

  /**
   * The database automatically creates one global field when SoDB::init() is called.
   * The @B realTime @b global field, which is of type SoSFTime, can be connected to
   * engines and nodes for real-time animation. The database will automatically
   * update the @B realTime @b global field 12 times per second, using a timer
   * sensor. Typically, there will be a node sensor on the root of the scene graph
   * which schedules a redraw whenever the scene graph changes; by updating the
   * @B realTime @b global field periodically, scene graphs that are connected to
   * @B realTime @b (and are therefore animating) will be redrawn. The rate at which
   * the database updates @B realTime @b can be controlled with this routine.
   * Passing in a zero time will disable automatic update of @B realTime @b. If
   * there are no enabled connections from the @B realTime @b field to any other
   * field, the sensor is automatically disabled. Note that the SoSceneManager class
   * automatically updates realTime immediately after redrawing, which will result in
   * as high a frame rate as possible if the scene is continuously animating. This
   * method ensures that engines that do not
   * continuously animate (such as SoTimeCounter) will eventually be scheduled.
   * See also #getRealTimeInterval().
   * The realTime global field can be accessed like this:
   * \if_cpp
   *  \code
   *   SoSFTime* realTimeField = (SoSFTime*)SoDB::getGlobalField( "realTime" );
   *  \endcode
   * \endif
   * \if_dotnet
   *  \code
   *   SoSFTime realTimeField = (SoSFTime)SoDB.GetGlobalField( "realTime" );
   *  \endcode
   * \endif
   * \if_java
   *  \code
   *   SoSFTime realTimeField = (SoSFTime)SoDB.getGlobalField( "realTime" );
   *  \endcode
   * \endif
   */
  static void setRealTimeInterval(const SbTime &deltaT);

  /**
   * Returns how often the database is updating the @B realTime @b global field.
   * See also #setRealTimeInterval().
   * The realTime global field can be accessed like this:
   * @code
   * SoSFTime* realTimeField = (SoSFTime*)SoDB::getGlobalField( "realTime" );
   * @endcode
   */
  static const SbTime &getRealTimeInterval();

  /**
   * This sets the timeout value for sensors that are delay queue sensors (one-shot
   * sensors, data sensors). Delay queue sensors are triggered whenever there is idle
   * time. If a long period of time elapses without any idle time (as when there are
   * continuous events to process), these sensors may not be triggered. Setting this
   * timeout value ensures that if the specified length of time elapses without any
   * idle time, the delay queue sensors will be processed anyway.
   *
   * The default timeout for delay queue sensors is 0.083 seconds.
   * So if delay queue sensors are only triggered by the timeout,
   * then by default the viewer animation can only run at about
   * 12 frames per second. To trigger the delay queue sensors at,
   * for example, 30 frames per second, make this call:
   *
   * SoDB::setDelaySensorTimeout( 0.03f );
   */
  static void setDelaySensorTimeout(const SbTime &t);

  /**
   * Returns the current delay queue timeout value.
   * See #setDelaySensorTimeout().
   */
  static const SbTime &getDelaySensorTimeout();

  /** 
   * Process Open Inventor sensor queues (TimerQueue and DelayQueue).
   * Normally this is handled automatically by the viewer.
   */
  static void processEvents();

  /**
   * @B UNIX/Linux Only @b @BR
   * In order to keep timer and idle sensors running as expected, it is necessary
   * that an Open Inventor application not block while waiting for input. If the Open
   * Inventor application uses the SoXt component library, this can be handled
   * automatically. However, if the application is using its own event loop, this
   * function is provided as a wrapper around select() that will handle Open
   * Inventor tasks if necessary instead of blocking.
   * [OIV-WRAPPER-NOT-WRAP]
   */
  static int doSelect(int nfds, fd_set *readfds, fd_set *writefds,
                      fd_set *exceptfds, struct timeval *userTimeOut);

#if SoDEPRECATED_BEGIN(10000)
  /**
   * Returns whether multi-thread support is enabled in Open Inventor.
   */
  SoDEPRECATED_METHOD(10000, "Not used anymore. You can safely assume the value is TRUE.")
  static SbBool isMultiThread();
#endif /** @DEPRECATED_END */

  /**
   * Acquire the global scene graph write-lock.
   *
   * The calling thread will block until the write-lock is available
   * (no threads hold a global read-lock).  The global write-lock is exclusive.
   * Only a single thread can hold this lock and while it is held, no thread
   * can acquire a global read-lock.  While holding the global write-lock it is
   * safe to modify the scene graph including adding/removing nodes, modifying
   * field values, etc.  Do not hold the global write-lock longer than necessary.
   * See #writeunlock().
   *
   * @B Note: @b Be sure the global write-lock is always released or there is a
   * good chance the application will "dead lock" because all standard actions
   * automatically attempt to acquire a global read-lock.
   */
  static void writelock();

  /**
   * Release the global scene graph write-lock.
   * See #writelock().
   */
  static void writeunlock();

  /**
   * Acquire a non-exclusive global scene graph read-lock.
   *
   * The calling thread will block until a read-lock is available
   * (no thread is holding the global write-lock).  Multiple threads may
   * hold global read-locks at the same time.  While holding a read-lock
   * it is safe to traverse the scene graph and to query values of fields,
   * but not to modify the scene graph (see method SoDB::writelock).
   * See #readunlock().
   */
  static void readlock();

  /**
   * Release a non-exclusive global scene graph read-lock.
   * See #readunlock().
   */
  static void readunlock();

#if SoDEPRECATED_BEGIN(10300)
  /**
   * Since Open Inventor 10.3, this method has @I no effect @i because
   * render caches are automatically managed.
   */
  SoDEPRECATED_METHOD(10300, "NumRenderCaches has no meaning anymore.")
  static void setNumRenderCaches( int /*num*/ )
  {}
#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(10300)
  /**
   * See method setNumRenderCaches().
   */
  SoDEPRECATED_METHOD(10300, "NumRenderCaches has no meaning anymore, this method will always return 1.")
  static int  getNumRenderCaches()
  { return s_numRenderCaches; }
#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(10300)
  /**
   * Since Open Inventor 10.3, this method has @I no effect @i because
   * render caches are automatically managed.
   */
  SoDEPRECATED_METHOD( 10300, "Display lists are no longer used, this method has no effect." )
  static void setRenderCacheMode( RenderCacheMode mode );
#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(10300)
  /**
   * See #setRenderCacheMode().
   */
  SoDEPRECATED_METHOD( 10300, "Render caches are no longer used, this method always return AUTO_RENDER_CACHE." )
  static RenderCacheMode getRenderCacheMode();
#endif /** @DEPRECATED_END */

  /**
   * Returns the current time.
   */
  static SbTime getCurrentTime();

  /** 
   * Loads a plugin library.
   * A plugin library may implement one or more classes extending Open Inventor classes.
   * For example a custom node or a custom file reader derived from SoInputReader.
   *
   * A plugin library must implement the initPlugin() and exitPlugin() C-formatted entry-points.
   * The initPlugin() function is responsible for registering the new class (or classes)
   * in the Open Inventor database through the standard SoType type creation mechanism.
   * The exitPlugin() function is used to disable the use of the custom class(s)
   * by removing the corresponding type(s) from the Open Inventor database.
   *
   * By default, Open Inventor attempts to load all files that may be plugins (files with the
   * extension .dll or .so depending on the platform) from the following locations:
   * - Application specified directories, @BR
   *   defined using the OIV_PLUGINS_DIRECTORY environment variable, or
   *   the SoDB::addPluginsDirectory() method.
   * - The current application working directory.
   * - The directory containing the Open Inventor libraries.
   *
   * Specific plugin libraries can be loaded and unloaded by direct calls to the SoDB::addPlugin()
   * and SoDB::removePlugin() methods.
   *
   * Different plugins can be designed depending on their usage. Please refer to the Open Inventor 
   * plugin section in the documentation
   */
  static bool addPlugin( const SbString& fileName );

  /** 
   * Unloads the specified plugin library.
   * See #addPlugin(). Returns true if plugin was successfully unloaded.
   */
  static bool removePlugin( const SbString& fileName );

  /** 
   * Returns the list of currently loaded plugins (file names).
   * See #addPlugin().
   */
  static const std::vector<SbString>& getLoadedPlugins();

  /** 
   * Returns the list of plugins that failed to load (file names).
   * See #addPlugin().
   */
  static const std::vector<SbString>& getUnloadedPlugins();

  /** 
   * Adds a directory to search for libraries defining new Open Inventor plugins.
   * See #addPlugin(). Returns true if directory has been successfully added.
   */
  static bool addPluginsDirectory( const SbString& dirName );

  /** 
   * Remove directory from the list of search directories for plugin loading.
   * See #addPlugin(). Returns true if directory has been successfully removed.
   */
  static bool removePluginsDirectory( const SbString& dirName );

  /** 
   * Returns the list of search directories for plugin loading.
   * See #addPlugin().
   */
  static const std::vector<SbString>& getPluginsDirectories();

SoEXTENDER_Documented public:
  /**
   * Registers a field conversion engine that can be used to
   * convert from one type of field to another. The type id's of the
   * two fields are passed in, as is the type id of the field
   * converter engine (derived from SoFieldConverter).
   */
  static void addConverter(SoType fromField, SoType toField,
                           SoType converterEngine);

  /**
   * Returns the field conversion engine registered for the two
   * given field types. If no such engine exists,
   * SoType::badType() is returned. See #addConverter().
   */
  static SoType getConverter(SoType fromField, SoType toField);

  /**
   * Remove a previously added converter.
   * See #addConverter().
   */
  static void removeConverter(SoType fromField, SoType toField);

  /** 
   * Specifies if launching IvTune interactively using the keyboard shortcut is allowed.
   * See SoIvTune to set the keyboard shortcut.
   */
  static void setIvTuneAllowed(SbBool ivTuneAllowed);

  /** 
   * Returns TRUE if IvTune can be launched interactively using the keyboard shortcut.
   * See #setIvTuneAllowed().
   */
  static SbBool getIvTuneAllowed();

SoEXTENDER public:

#ifdef _WIN32
  /**
   * Registry Utility Function
   * You must have Administrator privileges to use these functions
   * All methods work off of the key
   * \\HKEY_CURRENT_USER\\SOFTWARE\\VSG
   * If the key does not exist it is created.
   * Call this if you are going to make multiple calls to getRegistryKeyValue
   */
  static long openRegistryKey(const SbBool readOnlyFlag=FALSE);

  /**
   * This calls RegQueryValueEx.
   * If openRegistryKey has not been called the key is opened and closed,
   * otherwise the currently open key is used.
   */
  static long getRegistryKeyValue(LPTSTR keyName,
                             LPTSTR lpValueName, // address of name of value to query
                             LPDWORD lpType,     // address of buffer for value type
                             LPBYTE lpData,      // address of data buffer
                             LPDWORD lpcbData,   // address of data buffer size
                             const SbBool readOnlyFlag=FALSE // readOnly access if TRUE
                             );

  /**
   * This closes the key opened by openRegistryKey
   */
  static long closeRegistryKey();

#endif //_WIN32

 SoINTERNAL public:

#if SoDEPRECATED_BEGIN(10000)
  /**
   * Called from SoDB::finish()
   */
  SoDEPRECATED_METHOD( 10000, "Use SoDB::init() instead." )
  static void threadFinish();
#endif /** @DEPRECATED_END */

  // The following two functions are internal-only for now because we do
  // not have a reliable cross-platform implementation of trylock for a
  // read-write lock.

  // Provides a global try-write-lock mutex (see threads/SbThreadRWMutex.h)
  static SbBool trywritelock();

  // Provides a global try-read-lock mutex (see threads/SbThreadRWMutex.h)
  static SbBool tryreadlock();

  // Get ptr to global RWMutex for use with autolock
  // just an alias to SoInventorBase::getGlobalMutex()
  static SbThreadRWMutex *getGlobalMutex()
  { return SoInventorBase::getGlobalMutex(); }

  // Accesses sensor manager
  static SoSensorManager *getSensorManager();

  // Accesses UNLOCK sensor manager
  static SoSensorManager *getUnlockSensorManager();

  // Returns TRUE if database is initialized for the current thread.
  static bool isInitialized();

  // Returns FALSE if pick culling has been disabled
  static SbBool getPickCullingEnabled()
    { return m_pickCullingEnabled; }

  static const char* profileName;

  // This is called when some instance begins or ends a notification
  // process. It increments or decrements a counter of notifications
  // in progress. When the counter reaches 0, all priority 0
  // (immediate) delay queue sensors are triggered. By doing this,
  // all notification has a chance to finish before any evaluation
  // (due to data sensors, primarily) takes place.
  static void startNotify();
  static SbBool isNotifying();
  static void endNotify();

  // Enables/disables realTime sensor processing
  static void enableRealTimeSensor(SbBool enable);

  // Reads a base of any type, returning a pointer to it. Returns
  // FALSE on error.
  static SbBool read(SoInput *in, SoBase *&base);

  static SbBool m_readingAll;

  // Returns a copy of the path to the SoInput file (dirname).
  // If in does not contain a filename or if in is NULL, returns an empty string.
  // Returns empty string on error.
  static SbString getDirNameFromFullFileName(SoInput *in);

  // Flag to control use of valid names
  static SbBool useNameCompat;

  // Returns the threadId of the first thread that called SoDB::init()
  // it is usefull for nodes that do not support Multi-thread
  static SbThreadId_t getMainThreadId();

  /** True if in main thread */
  static bool isInMainThread();

  // OpenInventor Java internal usage only
  static void resetMainThreadId();

  // remove a header String previously added by calling registerHeader()
  static SbBool unregisterHeader(const SbString &headerString);

  // Two-Way synchronization between a field and a GlobalField.
  template <class T> static void
  syncFieldGlobaly( T& field, const SbName& name )
  {
    T* globalField = static_cast<T*>( getGlobalField( name ) );
    if ( !globalField )
    {
      globalField = static_cast<T*>( createGlobalField( name, T::getClassTypeId() ) );
      globalField->setValue( field.getValue() );
    }
    field.connectFrom( globalField );
    field.setOnValueChangedCallback( [name]( SoField* _field ) {
      T* field = static_cast<T*>( _field );
      T* globalField = static_cast<T*>( getGlobalField( name ) );
      if ( globalField && ( *field != *globalField ) )
        *globalField = *field;
    } );
  }

  // declare TLS class
  SB_THREAD_TLS_HEADER();

 private:
#ifdef _WIN32
  // Key used by openRegistryKey, getRegistryKeyValue, and closeRegistryKey
  static HKEY s_vsgRegistryKeys;
#endif

	SoDB();
	virtual ~SoDB();

  // Protects notification
  static SbThreadMutex *notifyMutex;

  // Used to read all graphs and paths from the Open Inventor file
  // specified by the given SoInput.
  static SoGroup *internalReadAll(SoInput *, SoGroup *);

  // Reads the graph from the external file specified by the given SoInput.
  // This returns NULL if nothing is read.
  static SoSeparator *readAllFromInputReaders(SoInput *in);

  // Flag to control what time is used
  static SbBool useGlobalFieldTime;

  // Store if OpenGL/RenderEngine must be init/finish
  // OIV_NOTUSEGPU SoPreference
  static bool s_notUseGPU;

  // Struct that contains all the information about the Global Database
  class SoGlobalDBInfos
  {
  public:
    // Constructor (allocate the sensorManager)
    SoGlobalDBInfos();

    // Init members that depends on node/field
    void init();
    // Cleanup members that depends on node/field
    void cleanup();

    // Destructor
    ~SoGlobalDBInfos();

    SoSensorManager* m_sensorManager;
    SoGlobalField* m_globalField;
    SoTimerSensor* m_realTimeSensor;
    SoSFRealTime *m_realTime;
    bool m_isInit;
  };

  // Thread local storage struct declaration
  struct MTstruct
  {
    SoGlobalDBInfos* globalDBInfos;
  };

  // get access to global database info
  // depending on the OIV mode used it is local to Thread or
  // global to all thread
  static SoGlobalDBInfos* getGlobalDBInfos();

  // WARNING: this static vars should never be access directly
  // always use getGlobalDBInfos() accessor
  static SoGlobalDBInfos* s_globalInfos;

  // How many current notifications
  static int notifyCount;

  static void realTimeSensorCallback(void *data, SoSensor *sensor);

  // List of valid header strings, and their corresponding callbacks
  static SbPList *headerList;

  // This dictionary stores field conversion engine types. The key
  // is created by mangling the types of the two fields to convert
  // from. The dictionary maps the key to the type of the conversion
  // engine.
  static SbDict *conversionDict;

  // Returns a conversionDict key from the two given field type id's.
  static uint32_t getConversionKey(SoType fromField, SoType toField);

  static void iv2_1PreCallback(void *userData, SoInput *in);
  static void iv2_0PreCallback(void *userData, SoInput *in);
  static void iv1_0PreCallback(void *userData, SoInput *in);
  static void iv2_0PostCallback(void *userData, SoInput *in);
  static void iv1_0PostCallback(void *userData, SoInput *in);

  // Global maximum number of render caches per node
  // (used by nodes that create OpenGL display lists or texture objects)
  static int s_numRenderCaches;
  static SbBool m_ivTuneAllowed;

  // Render cache creation mode
  // (used by nodes that create OpenGL display lists)
  static RenderCacheMode s_renderCacheMode;

  static SbBool m_pickCullingEnabled;

  static int s_initRefCount;

  static SbThreadId_t m_mainThreadId;

  static bool internalInit();

  friend class SoGlobalField;
  friend class SoVRMLDB;
  friend class SoVRMLInline;
};

#if defined(_WIN32)
#pragma warning( pop )
#endif

#endif /* _SO_DB_ */


