/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (MMM yyyy)
**=======================================================================*/

#ifndef _SO_INVENTOR_BASE_
#define _SO_INVENTOR_BASE_

#include <Inventor/SbBase.h>
#include <Inventor/STL/set>
#include <Inventor/STL/map>

#include <Inventor/threads/SbThreadMutex.h>

class SbThreadRWMutex;

/**
* General basic object database.
*
* @ingroup General
*
* @DESCRIPTION
*  Allows a minimal set of core Open Inventor functionality, specifically
*  NOT including the scene graph, to be initialized.  This is only useful
*  in special cases like using the MeshVizXLM Extraction module separate
*  from the Open Inventor scene graph. In the normal case applications
*  should use the SoDB methods (which call these methods).
*
* @SEE_ALSO
*    SoDB
*
*/

SoINTERNAL class INVENTORBASE_API SoInventorBase
{
public:
  /**
   * Initializes the basic inventor nodes.
   * This must be called before calling any other database routines.
   * [OIV-WRAPPER-NO-WRAP]
   */
  static void init();

  /**
   * Frees Open Inventor's internal static memory
   * allocations. This avoids false positives from memory
   * leak checking software. 
   * [OIV-WRAPPER-NO-WRAP]
   */
  static void finish();

  /**
   * Returns TRUE if database is initialized (for error checking)
   * [OIV-WRAPPER-NO-WRAP]
   */
  static bool isInitialized();

SoINTERNAL public:
  /** Possible power state */
  enum PowerState
  { 
    /** The machine is running */
    RUNNING,
    /** The machine is coming back from sleep*/
    RESUME,
    /** The machine is going to sleep */
    SUSPEND
  };

  /** Set the current power state to the specified value */
  static void setPowerState(PowerState state);

  /** Return the current power state */
  static PowerState getPowerState();

  // We are able to manage 2 multi-threads cases on MS Windows platforms:
  // 1- standard case where all threads share common global objects
  // and can interact each other (notification/rendering/multiGPU...)
  //
  // => this mode is enabled by default
  // 
  // 2- Experimental case where each threads can be considered as a separate Open Inventor application
  // each thread do not share anything, and then are independent from each other.
  // The following objects are thread specific:
  // - SoSensorManager::m_pThreadSensorMgr : SoSensorManagerThread handling notification queues
  // - SoDB::SoGlobalDBInfos               : main globalField, realTimeSensor sensor, realTime field
  // - SoGlobalField::nameDict             : globalField's dictionnary
  // - 
  // IMPORTANT: When using this mode, no notification MUST be done from a a thread to anoher.
  // (ie, it is not allowed to modified an Open Inventor field/node/scenegraph... from another 
  // thread than the one who created the given object.
  //
  // => this mode is enabled through envvar OIV_MULTITHREAD_APPLICATION
  //
  static SbBool isMultiThreadApplication()
  {
    if (s_isMultiThreadApplicationSet)
      return s_isMultiThreadApplication;
    return checkMultiThreadApplicationMode();
  }

#if SoDEPRECATED_BEGIN(10000)
  SoDEPRECATED_METHOD(10000, "Use SoInventorBase::init() instead.")
  static void threadInit();

  SoDEPRECATED_METHOD(10000, "Use SoInventorBase::finish() instead.")
  static void threadFinish();
#endif /** @DEPRECATED_END */

  static bool checkMultithread()
  { return s_checkMultithread; }

  // globalMutex provides a way for any application to use a global mutex.
  // It is also used inside OIV in some instances
  static SbThreadRWMutex* getGlobalMutex()
  { return globalMutex; }

  // returns true if OIV_MULTITHREAD_APPLICATION_DISABLE_LOCK is set, it allows to disable
  // any mutex lock and do internal performance test with OIV_MULTITHREAD_APPLICATION mode set
  static SbBool forceShouldCreateMutex();

  // global lock for init/finish operation
  static SbThreadMutex s_initFinishLock;

  // returns TRUE if we should use double precision math internally
  // see also SoPreference (OIV_DOUBLE_PRECISION)
  static inline bool useDoublePrecision()
  { return s_doublePrecision; }

  static inline void setDoublePrecision( bool val )
  { s_doublePrecision = val; }

  // returns TRUE if we should print debug msg when getting Zero vector length
  // see also SoPreference (OIV_REPORT_ZERO_VECTORS)
  static inline bool useReportZeroVector()
  { return s_reportZeroVectors; }

  // returns the current intersection epsilon value to use (OIV_INTERSECT_EPSILON)
  static inline double getIntersectionEpsilon()
  { return s_intersectionEpsilon; }

  // returns the current intersection epsilon value to use (OIV_INTERSECT_EPSILON)
  static inline double getIntersectionEpsilonSquared()
  { return s_intersectionEpsilonSquared; }

  static inline void setIntersectionEpsilon(double epsilon)
  { s_intersectionEpsilon = epsilon; s_intersectionEpsilonSquared = epsilon*epsilon;}

  //return true if finish is on the run
  static bool isFinishing();

private:
  // library usage counter
  static int s_initRefCount;

  // The isMultithreadedApplication flag can be set to true or false only one time, that's
  // why we need a threadApplicationModeIsSet flag.
  static bool checkMultiThreadApplicationMode();
  static bool s_isMultiThreadApplication;
  static bool s_isMultiThreadApplicationSet;

  static const char *s_versionString;
  // Global mutex
  static SbThreadRWMutex *globalMutex;

  static bool s_checkMultithread;

  // Use double precision math internally? (OIV_DOUBLE_PRECISION)
  static bool s_doublePrecision;

  // see SoPreference::OIV_REPORT_ZERO_VECTORS envvar
  static bool s_reportZeroVectors;

  // see SoPreferences::OIV_INTERSECT_EPSILON
  static double s_intersectionEpsilon;
  static double s_intersectionEpsilonSquared;

  static bool s_isFinishing;

  /** The current power state */
  static PowerState s_powerState;
};

#endif /* _SO_INVENTOR_BASE_ */


