/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul Isaacs (MMM yyyy)
** Modified by : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_NODE_KIT_PATH_
#define  _SO_NODE_KIT_PATH_

#include <Inventor/SbBasic.h>
#include <Inventor/SoPath.h>

class SoBaseKit;
class SoSearchAction;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoNodeKitPath
//
//  A SoNodeKitPath represents a scene graph or subgraph. It contains
//  pointers to a chain of nodeKitss, each of which is a child of the
//  previous. 
//  
//  Intermediary nodes between nodeKits are not included in the nodeKit path.
//
//  
//////////////////////////////////////////////////////////////////////////////

/**
 * Path that points to a list of hierarchical nodekits.
 * 
 * @ingroup nodekits
 * 
 * @DESCRIPTION
 *   SoNodeKitPath is a subclass of SoPath that lets you look at nodekits below the
 *   top nodekit in the path. Since nodekits have hidden children, when you call
 *   getTail() on a regular path, it returns the top-most nodekit on the path.
 *   This occurs even though the path might contain extra internal information
 *   leading to a node far deeper in the scene graph. For example, when picking an
 *   object inside an SoSceneKit, the @B regular @b path would end at the scenekit.
 *   But a @B nodekit @b path would continue further down listing the other nodekits
 *   below it in the path.
 *   
 *   Intermediary (private) nodes between nodekits are not included in the nodekit
 *   path.
 *   
 *   Note that there is no constructor for an SoNodeKitPath, so you can not create
 *   one. Rather, you cast an (SoPath *) into an (SoNodeKitPath *),
 *   which returns nodekit-style values from all the same questions as SoPath.
 *   
 *   Also, some methods of SoPath may not be called on an SoNodeKitPath. Any methods
 *   which take a regular SoNode as an argument (except for setHead()) are not
 *   accessible, and replaced by methods that take an SoBaseKit as an argument
 *   instead. Methods which allow the programmer to refer to the child index of a
 *   node beneath its parent are also inaccessible; since an SoNodeKitPath only shows
 *   nodekits and hides any private parts, successive nodekits in the path may not
 *   actually be parent and child.
 * 
 * @SEE_ALSO
 *    SoBaseKit,
 *    SoPath,
 *    SoRayPickAction,
 *    SoSearchAction
 * 
 * [OIVJAVA-WRAPPER-CLASS CUSTOM_CODE]
 */
class INVENTOR_API SoNodeKitPath : public SoPath {

 public:

  /**
   * Returns length of path chain (number of nodekits).
   */
  int                 getLength() const;

  /**
   * Return the last nodekit in a path chain. Note that getHead() is not
   * redefined from SoPath, since an SoNodeKitPath need not begin with a nodekit; the
   * restriction is placed only on successive nodes on the path.
   */
  SoNode *            getTail() const;

  /**
  * Returns the i'th node in the nodekit path.
  */
  SoNode *            getNode(int i) const;

  /**
  * Returns the i'th nodekit in the chain, counting backward from the
  * tail nodekit. Passing 0 for i returns the tail nodekit.
  */
  SoNode *            getNodeFromTail(int i) const;

  /**
   * Truncates the path chain, removing all nodes from index start on. Calling
   * @B truncate(0) @b empties the path entirely.
   */
  void                truncate(int start);

  /**
   * Pops the last nodekit off the end of the path.
   */
  void                pop();

  /**
   * Adds childKit to end of chain; uses first occurrence of childKit as a part
   * within current last nodekit. If the path is empty, this is equivalent to
   * @B setHead(childKit) @b.
   */
  void                append(SoBaseKit *childKit);

  /**
   * Adds all nodekits in fromPath's chain to end of chain; the head node of fromPath
   * must be the same as or a child of the current tail node.
   */
  void                append(const SoNodeKitPath *fromPath);

  /**
   * Returns TRUE if the passed nodekit is found anywhere in the path chain.
   */
  SbBool              containsNode(SoBaseKit *node) const;
    
  /**
   * If the two paths have different head nodes, this returns -1. Otherwise, it
   * returns the path chain index of the last nodekit (starting at the head) that is
   * the same for both paths.
   */
  int                 findFork(const SoNodeKitPath *path) const;

  /**
  * Returns TRUE if all nodes in the two nodekit path chains are equal.
  */
  INVENTOR_API friend int operator ==(const SoNodeKitPath &p1, const SoNodeKitPath &p2);

 protected:
  // Hide the constructor
  SoNodeKitPath(int approxLength) : SoPath(approxLength)      {}
  // Destructor
  virtual ~SoNodeKitPath();

 private:
  // This action is used to search for nodes.
  static SoSearchAction *searchAction;

};

#endif /* _SO_NODE_KIT_PATH_ */

