/*
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
***                                                                                ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
***                                                                                ***
***                        RESTRICTED RIGHTS LEGEND                                ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
***                                                                                ***
***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
***                        BORDEAUX, FRANCE                                        ***
***                      ALL RIGHTS RESERVED                                       ***
*/

#pragma once

#include <Inventor/misc/SoRefCounter.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/SbColorRGBA.h>
#include <Inventor/SbVec.h>
#include <Inventor/SbString.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/SbEventHandler.h>
#include <Inventor/SbEventArg.h>
#include <Inventor/SbPImpl.h>
#include <Inventor/image/SbRasterImage.h>
#include <Inventor/ViewerComponents/SiRenderArea.h>
#include <Inventor/ViewerComponents/SiRenderAreaAntialiasing.h>
#include <Inventor/ViewerComponents/SiRenderAreaTransparency.h>

class SoGLContext;
class SoBufferObject;
class SoRenderAreaCore;
class SoNode;
class SoSceneManager;


SO_PIMPL_BASE_PUBLIC_DECLARATION(SoOffscreenRenderArea);

/**
 * Render to an off-screen buffer for printing or generating textures.
 *
 * @ingroup General
 *
 * @DESCRIPTION
 *   This class is used to render a scene graph and to write the
 *   the result into a file or a buffer.
 *
 *   This is not a screen capture class. The specified scene graph is rendered
 *   to generate an image. This allows generation of images larger (higher
 *   resolution) than the user's screen and/or using different (e.g. higher
 *   quality) rendering parameters. The offscreen rendering may take more or
 *   less time than the same rendering on screen.
 *
 *   The renderer can render very large images because it is able to render
 *   the scene as multiple "tiles" and assemble the tiles into a complete
 *   image. Of course this requires multiple render traversals of the specified
 *   scene graph (one for each tile).
 *
 *   Rendering a "snapshot":
 *
 *   Your application may want to render the same image that the user sees on
 *   the screen. There are several things to consider. First, the scene graph
 *   given to the renderer must include the camera node that the viewer is using.
 *   If the application allows the viewer to create a camera automatically, the
 *   scene graph returned by the viewer's getSceneGraph() method does not include
 *   the camera. It's always safer to get the SoSceneManager object then get the
 *   scene graph. Second, some rendering options are set on the viewer object, not
 *   in the scene graph. These options, which include background color and
 *   transparency mode, must be explicitly queried from the viewer and set on the renderer object.
 *
 * Notes & Limitations:
 * - The default tile size is set to the maximum viewport size supported by the
 *   hardware (GL_MAX_VIEWPORT_DIMS), which means that tiling will be enabled
 *   only when large viewport sizes are requested.
 * - When tiling is enabled, the elements SoViewportRegionElement,
 *   SoLogicalViewportElement and SoModifyViewVolumeElement are set with
 *   different values for each tile, which means that scenegraphs that manage
 *   sub-viewports will need to take these elements into account in order for
 *   the final rendering to be correct.
 * - Maximum output image size is 25000 x 25000 pixels.
 *
 * @EXAMPLE
 *\if_cpp
 *  \code
 *   SoOffscreenRenderArea myOffscreen;
 *   // Set the size of the output
 *   myOffscreen.setSize( SbVec2i32( 15360, 8640 ) );
 *   // Set the sceneGraph to render
 *   myOffscreen.setSceneGraph(mySceneGraph);
 *   // If you want to render by tile
 *   if ( tileRendering )
 *     myOffscreen.setTile(SbVec2i32(3840, 2160), 2);
 *
 *   // Specify the color of the clear performed before rendering, here clear with white color
 *   myOffscreen.setClearColor( SbColorRGBA(1.f, 1.f, 1.f, 1.f) );
 *
 *   // Save the render as a PNG file
 *   // Specify that we also want to write the alpha component
 *   myOffscreen.renderToFile("result.png", SoOffscreenRenderArea::RGBA);
 *
 *   // Or... Save the render in a buffer
 *   myOffscreen.renderToBuffer(myBuffer);
 *  \endcode
 *\endif
 *
 *\if_dotnet
 *  \code
 *   SoOffscreenRenderArea myOffscreen = new SoOffscreenRenderArea();
 *   // Set the size of the output
 *   myOffscreen.setSize( new SbVec2i32( 15360, 8640 ) );
 *   // Set the sceneGraph to render
 *   myOffscreen.SetSceneGraph(mySceneGraph);
 *   // If you want to render by tile
 *   if ( tileRendering )
 *     myOffscreen.SetTile(new SbVec2i32(3840, 2160), 2);
 *
 *   // Specify the color of the clear performed before rendering, here clear with white color
 *   myOffscreen.SetClearColor( new SbColorRGBA(1.f, 1.f, 1.f, 1.f) );
 *
 *   // Save the render as a PNG file
 *   // Specify that we also want to write the alpha component
 *   myOffscreen.RenderToFile("result.png", SoOffscreenRenderArea.OutputFormats.RGBA);
 *
 *   // Or... Save the render in a buffer
 *   myOffscreen.RenderToBuffer(myBuffer);
 *  \endcode
 *\endif
 *
 *\if_java
 *  \code
 *   SoOffscreenRenderArea myOffscreen = new SoOffscreenRenderArea();
 *   // Set the size of the output
 *   myOffscreen.setSize( new SbVec2i32( 15360, 8640 ) );
 *   // Set the sceneGraph to render
 *   myOffscreen.setSceneGraph(mySceneGraph);
 *   // If you want to render by tile
 *   if ( tileRendering )
 *     myOffscreen.setTile(new SbVec2i32(3840, 2160), 2);
 *
 *   // Specify the color of the clear performed before rendering, here clear with white color
 *   myOffscreen.setClearColor( new SbColorRGBA(1.f, 1.f, 1.f, 1.f) );
 *
 *   // Save the render as a PNG file
 *   // Specify that we also want to write the alpha component
 *   myOffscreen.renderToFile("result.png", SoOffscreenRenderArea.OutputFormats.RGBA);
 *
 *   // Save the render in a buffer
 *   myOffscreen.renderToBuffer(myBuffer);
 *  \endcode
 *\endif
 * @M_SINCE 10.0
 */
class INVENTOR_API SoOffscreenRenderArea : public SoRefCounter, public SiRenderArea, public SiRenderAreaAntialiasing, public SiRenderAreaTransparency
{
  SO_PIMPL_BASE_PUBLIC_HEADER(SoOffscreenRenderArea);

 public:
  /**
  * Default constructor. 
  * NOTE: An internally created SoGLContext will be shared with the current context or an existing one if possible.
  * In order to have control over the SoGLContext sharing, use the alternate constructor.
  */
  SoOffscreenRenderArea();

  /**
  * Builds a new offscreen render area by sharing the given SoGlContext.
  */
  SoOffscreenRenderArea(SoGLContext* glContex);

  /**
  * Builds a new offscreen render area by using the given renderAreaCore.
  * Any following call to setSceneGraph will change the scene graph used by this instance.
  */
  SoOffscreenRenderArea(SoRenderAreaCore* renderAreaCore);

  /**
  * Destructor.
  */
  ~SoOffscreenRenderArea();

  /**
  * Defines the scene graph which will be traversed for rendering.
  */
  virtual void setSceneGraph(SoNode *newScene);

  /**
  * @see setSceneGraph().
  */
  virtual SoNode* getSceneGraph() const;

  /**
  * Defines viewport region (within the image) to use for rendering.
  */
  void setViewportRegion(const SbViewportRegion &newRegion);

  /**
  * @see setViewportRegion().
  */
  SbViewportRegion getViewportRegion() const;

  /**
  * Defines the algorithm for rendering transparent objects.
  * Default is NO_SORT. See SoGLRenderAction for possible transparency types.
  * See also SoGLRenderAction::TransparencyType.
  *
  * Note: When using OPAQUE_FIRST, SORTED_OBJECT or SORTED_PIXEL transparency, the depth buffer is
  * not updated (depth buffer writes are disabled) while rendering transparent objects.
  * As a result complex 3D shapes may not be rendered correctly.
  */
  virtual void setTransparencyType(SoGLRenderAction::TransparencyType type);

  /**
  * @see setTransparencyType().
  */
  virtual SoGLRenderAction::TransparencyType getTransparencyType() const;

  /**
  * Returns the encapsulated scene manager.
  */
  SoSceneManager* getSceneManager() const;

  /** Describes the format of the rendering output */
  enum OutputFormat
  {
    /** 3 components: red, green and blue */
    RGB,

    /** 4 components: red, green, blue and alpha */
    RGBA
  };

  /**
   * Render the given scene graph and save the result in the given file.
   * The format of the generated image is chosen automatically according to the
   * file extension.  Returns true if successful.
   *
   * If your application needs more control over creation of the file, please
   * see the #renderToBuffer() method.
   *
   * @B Supported file types are: @b
   *
   *   @TABLE_1B
   *   @TR @B extension file (case insensitive) @b      @TD @B raster image type @b            @TD @B Alpha Component Support @b
   *   @TR   .bmp                                       @TD Windows bitmap                     @TD YES
   *   @TR   .dds                                       @TD DirectDraw surface                 @TD YES
   *   @TR   .gif                                       @TD Graphics Interchange Format        @TD YES
   *   @TR   .hdr                                       @TD hdr                                @TD NO
   *   @TR   .jp2, .j2c, .j2k, .jpc, .jpx, .ecw, .ecwp  @TD jpeg 2000                          @TD YES
   *   @TR   .jpg, .jpeg, .jif, .jfif                   @TD Joint Photographic Experts Group   @TD NO
   *   @TR   .png                                       @TD Portable Network Graphics          @TD YES
   *   @TR   .ps                                        @TD PostScript                         @TD NO
   *   @TR   .rgb .sgi                                  @TD sgi                                @TD YES
   *   @TR   .tif .tiff                                 @TD Tagged Image File Format           @TD YES
   *   @TABLE_END
   *
   * @param filename path and name of the image file to be generated.
   *        If the file extension is not one of the above type, no file
   *        is generated.
   * @param outputFormat defines which components to write to the output.
   *        If set to #RGBA, the alpha component of the rendering is also
   *        written to the file. However if the targeted file type does not
   *        support alpha component, the file will be generated with #RGB
   *        components only. See the 3rd column of the above table.
   */
  bool renderToFile( const SbString& filename, OutputFormat outputFormat = RGB ) const;


  /**
  * Render the given scene graph and save the result in the given buffer.
  * The buffer is resized if its current size is not equal to the
  * the necessary size to store the rendering result.
  *
  * If your application needs more control over creation of an output file,
  * the SoBufferObject
  * can be converted into an SbRasterImage using the appropriate constructor.
  * Once instantiated, the SbRasterImage can be passed as a parameter to the
  * write function of any class inheriting from SoRasterImageRW (SoJPEGImageRW
  * for example to generate a JPEG file).
  *
  * @param buffer stores the result of the rendering.
  * @param outputFormat defines which components to write to the output.
  *        If set to #RGBA, the alpha component of the rendering is also written
  *        to the buffer.
  */
  bool renderToBuffer( SoBufferObject* buffer, OutputFormat outputFormat = RGB ) const;

  /**
  * Defines the maximum sub-image (tile) size for rendering and the number of pixels
  * on the border of each sub-image that are not written on the final image.
  *
  * The default tile size is set to the maximum viewport size supported by the
  * hardware (GL_MAX_VIEWPORT_DIMS) and can be queried using the
  * \if_cpp getMaxTileSize() method.\endif
  * \if_java getMaxTileSize() method.\endif
  * \if_dotnet GetMaxTileSize() method.\endif
  *
  * The default value for the number of edge pixels is 2 pixels.
  *
  * Edge pixels are usually required to avoid visual "artifacts"
  * at the subimage boundaries. One source of artifacts is that when OpenGL clips lines, it draws a
  * line between the points where the line exits and re-enters the visible region.
  * This additional segment is not part of the actual geometry. A two pixel border
  * will hide this undesired segment unless the line width is greater than two, in
  * which case you may need to specify a larger number of edge pixels.
  * @param size is dimension of the tile.
  * @param numEdgePixels is the number of pixels on the edge of the tile which will not
  *                      be included in the final image.
  */
  void setTile(SbVec2i32 size, int numEdgePixels);

  /**
  * @see setTile().
  */
  SbVec2i32 getTileSize() const;

  /**
  * @see setTile().
  */
  int getNumEdgePixels() const;

  /**
  * @see setClearPolicy().
  */
  virtual ClearPolicy getClearPolicy() const;

  /**
  * Defines the color buffer and depth buffer clear policy. Default value is ClearPolicy#COLORBUFFER_AND_DEPTHBUFFER.
  * Effects such as motion blur can be created by disabling clear color buffer or depth buffer.
  * @useenum{ClearPolicy}. See also setClearColor and setClearDepth.
  *
  * @param policy color buffer and depth buffer clear policy.
  */
  virtual void setClearPolicy(ClearPolicy policy);

  /**
  * Defines the RGBA value used when the color buffer is cleared. Default is transparent black (0,0,0,0).
  * This is the value used to clear the color buffer when renderToFile() or renderToBuffer() are 
  * called with setClearPolicy() method set to @B COLORBUFFER @b or @B COLORBUFFER_AND_DEPTHBUFFER @b.
  * See also setClearPolicy.
  *
  * @param color RGBA value used to clear the color buffer.
  */
  virtual void setClearColor(const SbColorRGBA& color);

  /**
  * @see setClearColor().
  */
  virtual SbColorRGBA getClearColor() const;

  /**
  * Defines the depth value used when the depth buffer is cleared. The default value is 1.
  * This is the value used to clear the depth buffer when renderToFile() or renderToBuffer() are 
  * called with setClearPolicy() method set to @B DEPTHBUFFER @b or @B COLORBUFFER_AND_DEPTHBUFFER @b.
  * See also setClearPolicy.
  *
  * @param depth value used to clear the depth buffer. Value is clamped to the range [0,1].
  */
  virtual void setClearDepth(float depth);

  /**
  * @see setClearDepth().
  */
  virtual float getClearDepth() const;

  /**
   * Defines the image size to use for rendering.
   */
  virtual void setSize(const SbVec2i32& size);

  /**
   * @see setSize().
   */
  virtual SbVec2i32 getSize() const;

  /**
   * Defines the render action used for rendering.
   */
  virtual void setGLRenderAction( SoGLRenderAction* glAction );

  /**
   * Returns the render action used for rendering.
   */
  virtual SoGLRenderAction* getGLRenderAction() const;

  /**
   * Returns the event handler that raises when a new render starts.
   */
  virtual SbEventHandler<RenderEventArg&>& onStartRender();

  /**
  * Gets the maximum subimage (tile) size for rendering.
  */
  static SbVec2i32 getMaxTileSize();

  /**
  * Define the antialiasing mode.
  * @param mode The antialiasing algorithm. Default is NO_ANTIALIASING which turns off antialiasing.
  */
  virtual void setAntialiasingMode(SoSceneManager::AntialiasingMode mode);

  /**
  * @see setAntialiasingMode().
  */
  virtual SoSceneManager::AntialiasingMode getAntialiasingMode() const;

  /**
  * Define the antialiasing quality value.
  * @param quality The quality is a factor in the range [0.0,1.0]. @BR
  *        Default is 0.0. The value 0.0 turns off antialiasing.
  */
  virtual void setAntialiasingQuality(float quality);

  /**
  * @see setAntialiasingQuality().
  */
  virtual float getAntialiasingQuality() const;

  /**
   * Performs an offscreen rendering of the scene graph without saving the result.
   *
   * This method initiates the rendering process but does not store the resulting
   * image in any buffer or file. It is useful for scenarios where only the
   * rendering process is required, such as for benchmarking or updating the
   * internal state of the scene without needing to keep the rendered output.
   *
   * Unlike #renderToFile, which saves the rendered image to a provided buffer
   * object, or #renderToBuffer, which writes the rendered image directly to a
   * file, this method does not retain the rendered image in any form.
   */
  virtual RenderStatus render();

  /**
  * Event that can be sent to the application for each tile rendered.
  * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
  */
  struct INVENTOR_API EventArg : public SbEventArg
  {
  public:

    /**
    * Returns the SoOffscreenRenderArea that raised this event.
    */
    const SoOffscreenRenderArea* getSource() const;

    /**
    * Returns the number of tiles used to generate the entire image.
    */
    int getNumTiles() const;

    /**
    * Returns the origin of the tile in the entire image.
    */
    const SbVec2i32& getOrigin() const;

    /**
    * Returns the image of the rendered tile.
    * [OIVJAVA-WRAPPER-RETURN-TYPE USE_FORCE]
    */
    const SbRasterImage& getTile() const;

  private:
    friend class inventor::impl::SoOffscreenRenderAreaImpl;

    /** Default constructor */
    EventArg(const SoOffscreenRenderArea* renderarea, int numTiles, SoBufferObject* buffer);
    ~EventArg();

    const SoOffscreenRenderArea* m_renderarea;
    SbVec2i32 m_origin;
    int m_numTiles;
    SbRasterImage* m_tile;
  };

  /**
  * Event raised when a tile has been rendered.
  */
  SbEventHandler<EventArg&> onTileRendered;

protected:

  /**
   * Applies an SoGLRenderAction to the scene graph managed here in a sub viewport
   * defined by the given window in pixel (left, bottom, right, top).
   * The rendering may have been aborted, so check this function result
   * before calling swap buffer for example
   */
  virtual RenderStatus render( const SbVec4i32& window );

  #if SoDEPRECATED_BEGIN(10210)
  /**
  * Applies an SoGLRenderAction to the scene graph managed here in a sub viewport
  * defined by the given window in pixel (left, bottom, right, top).
  * The rendering may have been aborted, so check this function result
  * before calling swap buffer for example
  */
  SoDEPRECATED_METHOD(10210, "Use render(const SbVec4i32&) instead.")
  virtual RenderStatus render( const SbVec4s &window )
  {
    return render( SbVec4i32( window ) );
  }
  #endif /* SoDEPRECATED_BEGIN(10210) */

};


