/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SO_PICKED_POINT_
#define _SO_PICKED_POINT_

#include <Inventor/SbBasic.h>
#include <Inventor/SbLinear.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/SoLists.h>

class SoDetail;
class SoGetMatrixAction;
class SoPath;
class SoState;
class SoGetMatrixActionWrapper;
class SoRayPickAction;
class SoAction;
class SoTextureCoordinateBundle;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoPickedPoint
//
//  An SoPickedPoint represents a point on the surface of an object
//  that was picked by applying an SoRayPickAction to a scene. It
//  contains the point of intersection, the surface normal and texture
//  coordinates at that point, and other items.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Represents point on surface of picked object.
 * 
 * @ingroup General
 * 
 * @DESCRIPTION
 *   An SoPickedPoint represents a point on the surface of an object that was picked
 *   by applying an SoRayPickAction to a scene. It contains a path to the picked
 *   shape, the point of intersection, the surface normal and texture coordinates at
 *   that point, and other information.  Note that it is not always necessary to
 *   explicitly apply an SoRayPickAction to the scene.  The getPickedPoint method may
 *   also be called on an SoHandleEventAction or an SoEventCallback node.
 *   
 *   Each node in the picked path may have a corresponding instance of an SoDetail
 *   subclass. These detail instances are stored in the SoPickedPoint.
 * 
 * @SEE_ALSO
 *    SoRayPickAction,
 *    SoPickStyle,
 *    SoDetail,
 *    SoPath,
 *    SoEventCallback
 * 
 * 
 */
class INVENTOR_API SoPickedPoint {

 public:
  /**
   * Default constructor.
   */
  SoPickedPoint();

  /**
   * Copy constructor.
   */
  SoPickedPoint(const SoPickedPoint &pp);

  /**
   * Destructor.
   */
  ~SoPickedPoint();

  // Returns an instance that is a copy of this instance. The caller
  // is responsible for deleting the copy when done.
  SoPickedPoint *     copy() const;

  /**
   * Returns the intersection point in world space.
   */
  const SbVec3f &     getPoint() const         { return worldPoint; }
  /**
   * Returns the surface normal in world space.
   *
   * NOTE: It is possible to disable computation of normal vectors by
   * calling the enableNormalsGeneration() method on SoRayPickAction.
   * If normal vectors are disabled, this method returns 0,0,0.
   */
  const SbVec3f &     getNormal() const        { return worldNormal; }
  /**
   * Returns the texture coordinates in image space.
   *
   * NOTE: By default the SoRayPickAction does not compute texture coordinates
   * and this method returns 0,0,0,0.  To enable texture coordinate computation
   * call the enableTexCoordsGeneration() method on the pick action object or
   * set the environment variable OIV_PICK_GENERATE_ALL_PROPERTIES to true.
   */
  const SbVec4f &     getTextureCoords() const { return imageTexCoords; }

  /**
   * Returns the index into the current set of materials of the material active at
   * the intersection point. Note that if the materials are interpolated between
   * vertices, the index will correspond to the material at one of the vertices.
   */
  int                 getMaterialIndex() const { return materialIndex; }

  /**
   * Returns the path to the object that was intersected.
   */
  inline SoPath *            getPath() const          { return path; }

  /**
   * Returns whether the intersection is actually on the geometry of the character
   * that was hit, as opposed to being on the bounding box. The pick style (see
   * SoPickStyle) affects this decision.
   */
  SbBool              isOnGeometry() const     { return onGeometry; }

  //////////////////////
  //
  // The following methods all take a pointer to a node in the
  // returned path. If the node pointer is NULL (the default), the
  // information corresponding to the tail of the path is returned.
  //

  /**
  * Returns the detail that corresponds to the specified node in the path returned by
  * getPath(). If the node is null, the detail
  * corresponding to the tail of the pick path is returned.
  *
  * Caution: If there is no detail class associated with the node, null is returned.
  */
  const SoDetail *    getDetail(const SoNode *node = NULL) const;

  /**
  * Returns the transformation matrix between the object space and world
  * space corresponding to the given node in the path. If the node is NULL,
  * the matrix corresponding to the tail of the (full) path is
  * returned.
  *\if_cpp
  * By default, the node is NULL.
  *\endif
  */
  SbMatrix            getObjectToWorld(const SoNode *node = NULL) const;

  /**
  * Returns the transformation matrix between world space and the object
  * space corresponding to the given node in the path. If the node is NULL,
  * the matrix corresponding to the tail of the (full) path is
  * returned.
  *\if_cpp
  * By default, the node is NULL.
  *\endif
  */
  SbMatrix            getWorldToObject(const SoNode *node = NULL) const;

  /**
  * Returns the texture transformation matrix between the object space and 
  * image space corresponding to the given node in the path. If the node is
  * NULL, the matrix corresponding to the tail of the (full) path is
  * returned.
  *\if_cpp
  * By default, the node is NULL.
  *\endif
  */
  SbMatrix            getObjectToImage(const SoNode *node = NULL) const;

  /**
  * Returns the texture transformation matrix between image space and the
  * object space corresponding to the given node in the path. If the node is
  * NULL, the matrix corresponding to the tail of the (full) path is
  * returned.
  *\if_cpp
  * By default, the node is NULL.
  *\endif
  */
  SbMatrix            getImageToObject(const SoNode *node = NULL) const;

  /**
  * Returns the intersection point in
  * the object space corresponding to the given node in the path. If the node
  * is NULL, the information corresponding to the tail of the
  * (full) path is returned.
  *\if_cpp
  * By default, the node is NULL.
  *\endif
  */
  SbVec3f             getObjectPoint(const SoNode *node = NULL) const;

  /**
  * Returns the surface normal in
  * the object space corresponding to the given node in the path. If the node
  * is NULL, the information corresponding to the tail of the
  * (full) path is returned.
  *\if_cpp
  * By default, the node is NULL.
  *\endif
  */
  SbVec3f             getObjectNormal(const SoNode *node = NULL) const;

  /**
  * Returns the texture coordinates in
  * the object space corresponding to the given node in the path. If the node
  * is NULL, the information corresponding to the tail of the
  * (full) path is returned.
  *\if_cpp
  * By default, the node is NULL.
  *\endif
  */
  SbVec4f             getObjectTextureCoords(const SoNode *node=NULL) const;

  //
  //////////////////////

 SoEXTENDER public:

  // NOTE: these methods should be called ONLY by the node that
  // causes the SoPickedPoint instance to be created, and ONLY at
  // the time it is created. Setting or changing the normal or
  // texture coordinates at a later time may cause undefined results.

  // Sets the normal in object space
  void                setObjectNormal(const SbVec3f &normal);

  // Sets the texture coordinates in object space
  void                setObjectTextureCoords(const SbVec4f &texCoords);

  // Sets the material index
  void                setMaterialIndex(int index)  { materialIndex = index; }

  // Sets the detail corresponding to the given node in the path.
  // NULL may be passed to remove a detail. All detail pointers are
  // NULL by default.
  void                setDetail(SoDetail *detail, SoNode *node);

 SoINTERNAL public:

  // The constructor is internal since instances are created only by
  // the SoRayPickAction
  SoPickedPoint(const SoPath *path, SoState *state,
                const SbVec3f &objSpacePoint);

  // override the current path this is useful for alternate representations
  void overridePath(const SoPath* _path);

  // set TextureCoords during picking
  void setObjectTextureCoords( const SoRayPickAction *action, const SoTextureCoordinateBundle &tcb );

  /**
   * @brief Returns the intersection point in world space using double precision.
   * @details The use of this method makes sense only in case the environment
   * variable "OIV_DOUBLE_PRECISION" is set to true. Otherwise it returns the
   * exact same point as the floating one (worldPoint).
   * @return the intersection point in world space using double precision.
   */
  const SbVec3d& getPointD() const { return worldPointD; }

 private:
  SoGetMatrixAction *getGetMatrixAction() const;
  // This action is used to get the world-to-object matrix
  SoGetMatrixActionWrapper  *m_matrixActionWrapper;

  // Intersection point and normal in world space, and texture
  // coordinates in image space
  SbVec3f             worldPoint;
  SbVec3d             worldPointD;
  SbVec3f             worldNormal;
  SbVec4f             imageTexCoords;

  // ViewportRegion, which is needed when figuring out matrices
  SbViewportRegion    vpRegion;

  // Material index
  int                 materialIndex;

  // TRUE if pick was on geometry of shape as opposed to bbox
  SbBool              onGeometry;

  // The path to the shape that was picked
  SoPath              *path;

  // Details corresponding to nodes in path, one per node in path.
  // Many may be NULL.
  SoDetailList        details;

  // The pointer to the state allows us to access matrices to
  // convert from object space to world and image space. This
  // pointer is valid only during construction and setting of the
  // info inside the instance.
  SoState             *state;

  // Returns index in path of given node, or -1 if not found
  int                 getNodeIndex(const SoNode *node) const;

  // Applies SoGetMatrixAction to path to node
  void                getMatrix(const SoNode *node) const;

  // Multiplies SbVec4f by matrix - for transforming texture coords
  static SbVec4f multVecMatrix4(const SbMatrix &m, const SbVec4f v);

  // void operator=( const SoPickedPoint & );

  // evaluates the object space (norm, point) when action is known
  static void getObjectGeomlUsingModelMatrixElement( const SoAction* action, const SbVec3f &inVec, SbVec3f &outVec, bool normalize = true );

  // optimized methods for picking
  SbVec3f getObjectNormal( const SoAction *action ) const;

  SbVec3f getObjectPoint( const SoAction *action ) const;

};

#endif /* _SO_PICKED_POINT_ */

