/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#pragma once

#include <Inventor/SbEventArg.h>
#include <Inventor/SbEventHandler.h>

/**
* @VSGEXT Class for reporting progress of a task.
*
* @ingroup Basics
*
* @DESCRIPTION
* This class notifies the application when a task (for example a long computation)
* starts, stops and when it is in progress.
* This is useful, for example, to allow the application to update a progress bar.
*
* Potentially long processes are conceptually divided into sequential tasks, which
* are divided into subtasks, which may have multiple iteration steps.
* Not every process uses all of the above. See each specific use of this class
* for details about the specific process.
*
* See events #onBeginTask, #onEndTask, #onBeginSubTask, #onEndSubTask, #onEndStep.
*
* Each event has specific arguments: #SoProgressIndicator::TaskEventArg, #SoProgressIndicator::SubTaskEventArg, #SoProgressIndicator::StepEventArg.
*
* Currently used in: SoVolumeRender (setRenderProgress).
*
* @SEE_ALSO
*  SoVolumeRender
*/
class INVENTOR_API SoProgressIndicator
{

public:
  /**
   * Constructor.
   */
  SoProgressIndicator() {};

  virtual ~SoProgressIndicator() {};

  /**
   * Class given to callback when onBeginTask/onEndTask are raised.
   * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
   */
  struct INVENTOR_API TaskEventArg : public SbEventArg
  {
    TaskEventArg(SoNode* src, const SbString& eventName, unsigned int numTasks, unsigned int numTasksDone, unsigned int numSubTasks );
    ~TaskEventArg();

    /** @return source node that raised the event. */
    SoNode* getSource() const;

    /** @return name of the event. */
    const SbString& getEventName() const;

    /* @return number of tasks that will be done by the SoNode. */
    unsigned int getNumTasks() const;

    /* @return total number of tasks done since the beginning. */
    unsigned int getNumTasksDone() const;

    /* @return number of subtasks that will be executed during this task. */
    unsigned int getNumSubTasks() const;

  protected:
    unsigned int m_numTasksDone;

  private:
    SoNode* m_src;
    SbString* m_eventName; // unique
    unsigned int m_numTasks;
    unsigned int m_numSubTasks;
  };

  /**
   * Class given to callback when onBeginSubTask/onEndSubTask are raised.
   * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
   */
  struct INVENTOR_API SubTaskEventArg : public SbEventArg
  {
    SubTaskEventArg(SoNode* src, const SbString& eventName, unsigned int numSubTasks, unsigned int numSubTasksDone, unsigned int numSteps);

    ~SubTaskEventArg();

    /** @return source node that raised the event. */
    SoNode* getSource() const;

    /** @return name of the event. */
    const SbString& getEventName() const;

    /* @return number of subtask that will be done by the parent task. */
    unsigned int getNumSubTasks() const;

    /* @return total number of subtasks done since task beginning. */
    unsigned int getNumSubTasksDone() const;

    /* @return number of steps(iterations) that will be executed during this subtask. */
    unsigned int getNumSteps() const;

  protected:
    unsigned int m_numSubTasksDone;

  private:
    SoNode* m_src;
    SbString* m_eventName; // unique
    unsigned int m_numSubTasks;
    unsigned int m_numSteps;
  };

  /**
   * Class given to callback when an onEndStep is raised.
   * [OIV-WRAPPER-CLASS AUTO_PROPERTY]
   */
  struct INVENTOR_API StepEventArg : public SbEventArg
  {
    StepEventArg(SoNode* src, const SbString& eventName, unsigned int numSteps, unsigned int numStepsDone);

    ~StepEventArg();

    /** @return source node that raised the event. */
    SoNode* getSource() const;

    /** @return name of the event. */
    const SbString& getEventName() const;

    /* @return number of steps that will be done by the parent subtask. */
    unsigned int getNumSteps() const;

    /* @return total number of steps(iterations) done since subtask beginning. */
    unsigned int getNumStepsDone() const;

  protected:
    unsigned int m_numStepsDone;

  private:
    SoNode* m_src;
    SbString* m_eventName; // unique
    unsigned int m_numSteps;
  };

  /**
   * Event raised when a task begins.
   */
  SbEventHandler<TaskEventArg&> onBeginTask;

  /**
   * Event raised when a task ends.
   */
  SbEventHandler<TaskEventArg&> onEndTask;

  /**
   * Event raised when a sub-task begins.
   */
  SbEventHandler<SubTaskEventArg&> onBeginSubTask;

  /**
   * Event raised when a sub-task ends.
   */
  SbEventHandler<SubTaskEventArg&> onEndSubTask;

  /**
  * Event raised when a steps ends.
  * A step is inside a SubTask(Begin/End).
  */
  SbEventHandler<StepEventArg&> onEndStep;
};

