/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
#ifndef RenderArea_H
#define RenderArea_H

#include <Inventor/ViewerComponents/MFC/StdAfx.h>

#include <Inventor/sys/port.h>

#if defined(_WIN32) && !defined(OIV_DISABLE_AUTOLINK) && !defined(MFCViewerComponents_EXPORTS)
#  if !defined(__MFC_VIEWER_COMPONENTS_LIB)
#    if defined(_DEBUG)
#      define __MFC_VIEWER_COMPONENTS_LIB "fei_inventor_MFCViewerComponentsD.lib"
#    else
#      define __MFC_VIEWER_COMPONENTS_LIB "fei_inventor_MFCViewerComponents.lib"
#    endif
#  endif
#  pragma comment(lib, __MFC_VIEWER_COMPONENTS_LIB)
#endif

#include <Inventor/ViewerComponents/SoRenderAreaCore.h>

#include <Inventor/misc/SoRef.h>
#include <Inventor/sys/SoGL.h>
#include <Inventor/SbColorRGBA.h>

#include <Inventor/sys/port.h>

class SoRenderAreaCore;
class SoNode;

/**
 * Class to render an OpenInventor scene in an MFC OpenGL window
 *
 * @ingroup ViewerComponentsMFC
 *
 * @DESCRIPTION
 *
 * RenderArea creates a CWnd that can be used as a 3D rendering window.
 * Use the initializeGL() method to create the OpenGL context.
 * Use the setSceneGraph() method to specify the scene graph to be rendered.
 *
 * A SceneExaminer node is typically used to allow the user to manipulate the scene camera.
 *
 * @SEE_ALSO
 *    RenderAreaInteractive, RenderAreaExaminer, SceneInteractor, SceneExaminer
 */
class MFCVIEWERCOMPONENTS_API RenderArea : public CWnd, public SiRenderArea, public SiRenderAreaStereo, public SiRenderAreaAntialiasing, public SiRenderAreaStillSuperSampling
{
public:
  /**
   * Constructor
   */
  RenderArea();

  /**
   * Destructor
   */
  virtual ~RenderArea();

  /**
   * Set the scene graph
   */
  virtual void setSceneGraph( SoNode* sceneGraph );

  /**
   * Gets the scene graph to render.
   */
  virtual SoNode* getSceneGraph() const;

  /**
   * Defines the color buffer and depth buffer clear policy.
   * @useenum{ClearPolicy} Default is COLORBUFFER_AND_DEPTHBUFFER.
   */
  virtual void setClearPolicy(ClearPolicy policy);

  /**
   * Gets the color buffer and depth buffer clear policy.
   */
  virtual ClearPolicy getClearPolicy() const;

  /**
   * Defines the RGBA value used when the color buffer is cleared.
   * Default is transparent black (0,0,0,0).
   */
  virtual void setClearColor(const SbColorRGBA& color);

  /**
   * Gets the RGBA value used when the color buffer is cleared.
   */
  virtual SbColorRGBA getClearColor() const;

  /**
   * Defines the depth value used when the depth buffer is cleared.
   * Default is 1.  Range is 0..1.
   */
  virtual void setClearDepth(float depth);

  /**
   * Gets the depth value used when the depth buffer is cleared.
   */
  virtual float getClearDepth() const;

  /**
   * Defines the size used for rendering.
   */
  virtual void setSize(const SbVec2i32& size);

  /**
   * Gets the size used for rendering.
   */
  virtual SbVec2i32 getSize() const;

  /**
   * Defines the render action used for rendering.
   */
  virtual void setGLRenderAction( SoGLRenderAction* glAction );

  /**
   * Returns the render action used for rendering.
   */
  virtual SoGLRenderAction* getGLRenderAction() const;

  /**
  * Define the antialiasing mode.
  * @param mode The antialiasing algorithm. Default is NO_ANTIALIASING which turns off antialiasing.
  */
  virtual void setAntialiasingMode( SoSceneManager::AntialiasingMode mode ) override;

  /**
  * @see setAntialiasingMode().
  */
  virtual SoSceneManager::AntialiasingMode getAntialiasingMode() const override;

  /**
  * Define the antialiasing quality value.
  * @param quality The quality is a factor in the range [0.0,1.0]. @BR
  *        Default is 0.0. The value 0.0 turns off antialiasing.
  */
  virtual void setAntialiasingQuality( float quality ) override;

  /**
  * @see setAntialiasingQuality().
  */
  virtual float getAntialiasingQuality() const override;

  /**
  * Set quality for supersampling when "still" (not interacting).
  * When quality is greater than 0, still images will be automatically supersampled.
  * @param quality The quality is a factor in the range [0.0,1.0]. @BR
  *        Default value is 0.0. Use the value 0.0 to turn off still supersampling. 0.5 is a typical value.
  * See also setStillSuperSamplingDelay.
  */
  virtual void setStillSuperSamplingQuality( float quality ) override;

  /**
  * @see setStillSuperSamplingQuality().
  */
  virtual float getStillSuperSamplingQuality() const override;

  /**
  * Set delay for supersampling when "still" (not interacting).
  * If greater than 0, images will be supersampled after the specified delay.
  * @param delay The delay is in milliseconds. @BR
  *        If greater than 0, images will be supersampled after the specified delay
  * See also setStillSuperSamplingQuality.
  */
  virtual void setStillSuperSamplingDelay( unsigned int delay ) override;

  /**
  * @see setStillSuperSamplingDelay().
  */
  virtual unsigned int getStillSuperSamplingDelay() const override;

  /**
   * Returns the event handler that raises when a new render starts.
   */
  virtual SbEventHandler<RenderEventArg&>& onStartRender();

  /**
   * This method is called in order to create OpenGL context.
   */
  virtual void initializeGL( HDC deviceContext, bool enableStereo = false );

  /**
   * This method is called in order to destroy OpenGL context.
   */
  virtual void uninitializeGL();

  /**
   * This method is called in order to destroy OpenGL context.
   */
  virtual void paintGL() { OnPaint(); }

  /**
   * Return the width of the window
   */
  int getWidth() const
  {
    return m_width;
  }

  /**
   * Return the height of the window
   */
  int getHeight() const
  {
    return m_height;
  }

  /**
   * Activate/Deactivate stereo.
   */
  virtual void activateStereo( bool activated );

  /**
   * Returns true if stereo rendering is currently activated.
   *
   * @NOTES
   *   Stereo status can only be retrieved once rendering area is initialized.
   */
  virtual bool isStereoActivated() const;

  /**
   * Defines the stereo parameters.
   *
   * @NOTES
   *   Stereo parameters can only be set once rendering area is initialized.
   */
  virtual void setStereoParameters(SoStereoParameters* parameters);

  /**
   * @see setStereoParameters().
   */
  virtual SoStereoParameters* getStereoParameters() const;

  /**
   * Returns true if stereo buffering is enabled.
   */
  bool isRawStereoAvailable();

protected:

  /**
   * Render the scene graph.
   */
  virtual RenderStatus render();

  /**
   * Function call at the init of the window
   */
  BOOL PreCreateWindow( CREATESTRUCT& cs );

  /**
   * Function call when a paint is needed
   */
  afx_msg void OnPaint();

  /**
   * Function call when object background needs erasing (for example, when resizing)
   */
  afx_msg BOOL OnEraseBkgnd(CDC* pDC);

  /**
   * Function call when the size of the Wnd change
   */
  afx_msg void OnSize( UINT nType, int cx, int cy );

  /**
   * Instance of the class which provides the algorithms for viewing and managing
   * OpenInventor scene graphs.
   */
  SoRef<SoRenderAreaCore> m_renderAreaCore;

  /**
   * Scene graph used by this render area
   */
  SoNode* m_sceneGraph;

  /**
   * Clear color used by this render area
   */
  SbColorRGBA m_color;

  /**
   * Clear depth used by this render area
   */
  float m_depth;

  /**
   * Size used by this render area
   */
  SbVec2i32 m_size;

  /**
   * GLRenderAction used by this render area
   */
  SoGLRenderAction* m_glRenderAction;

  /**
   * Antialiasing mode used by this render area
   */
  SoSceneManager::AntialiasingMode m_antialiasingMode;

  /**
   * Antialiasing quality used by this render area
   */
  float m_antialiasingQuality;

  /**
   * Delay for still supersampling
   */
  unsigned int m_stillAntialiasingDelay;

  /**
   * Quality of still supersampling
   */
  float m_stillAntialiasingQuality;

  /**
   * Whether or not the stereo is activated in the render area
   */
  bool m_stereo;

  /**
   * Store the height of the window
   */
  int m_width;

  /**
   * Store the height of the window
   */
  int m_height;

  /**
   * OpenGL context used by the application
   */
  SoRef<SoGLContext> m_glContext;

  /**
   * Store the clear policy
   */
  ClearPolicy m_clearPolicy;

  // ********************************
  // windows stuff
  // ********************************

  /**
   * The device context for OpenGL
   */
  HDC m_hDC;

  /**
   * Render Context for OpenGL
   */
  HGLRC m_hRC;

  DECLARE_MESSAGE_MAP()
};

#endif // RenderArea_H
