#pragma once

#include <Inventor/sys/port.h>

#if defined( _WIN32 ) && !defined( OIV_DISABLE_AUTOLINK ) && !defined( QtViewerComponents_EXPORTS )
#if !defined( __QT_VIEWER_COMPONENTS_LIB )
#if defined( _DEBUG )
#define __QT_VIEWER_COMPONENTS_LIB "fei_inventor_QtViewerComponentsD.lib"
#else
#define __QT_VIEWER_COMPONENTS_LIB "fei_inventor_QtViewerComponents.lib"
#endif
#endif
#pragma comment( lib, __QT_VIEWER_COMPONENTS_LIB )
#endif

#include <Inventor/ViewerComponents/SoRenderAreaCore.h>

#include <QQuickFramebufferObject>

namespace openinventor {
namespace inventor {
namespace viewercomponents {
namespace qt {
namespace qml {

class QmlRenderer;

/**
 * @PREVIEWTAG
 *
 * Class to render an OpenInventor scene in a QML item.
 *
 * @ingroup ViewerComponentsQt
 *
 * @DESCRIPTION
 *
 * RenderArea creates a 3D rendering item.
 * Use the setSceneGraph() method to specify the scene graph to be rendered.
 *
 * @SEE_ALSO
 *    RenderAreaInteractive, RenderAreaOrbiter
 *
 * @PREVIEWFEATURES
 */
class QTVIEWERCOMPONENTS_API RenderArea : public QQuickFramebufferObject,
                                          public SiRenderArea
{
  Q_OBJECT

public:
  RenderArea();

  virtual ~RenderArea();

  /**
   * Sets the scene graph to render.
   */
  void setSceneGraph( SoNode* sceneGraph ) override;

  /**
   * Gets the scene graph to render.
   */
  SoNode* getSceneGraph() const override;

  /**
   * Defines the color buffer and depth buffer clear policy.
   * @useenum{ClearPolicy} Default is COLORBUFFER_AND_DEPTHBUFFER.
   */
  void setClearPolicy( ClearPolicy policy ) override;

  /**
   * Gets the color buffer and depth buffer clear policy.
   */
  ClearPolicy getClearPolicy() const override;

  /**
   * Defines the RGBA value used when the color buffer is cleared.
   * Default is transparent black (0,0,0,0).
   */
  void setClearColor( const SbColorRGBA& color ) override;

  /**
   * Gets the RGBA value used when the color buffer is cleared.
   */
  SbColorRGBA getClearColor() const override;

  /**
   * Defines the depth value used when the depth buffer is cleared.
   * Default is 1. Range is 0..1.
   */
  void setClearDepth( float depth ) override;

  /**
   * Gets the depth value used when the depth buffer is cleared.
   */
  float getClearDepth() const override;

  /**
   * Defines the size used for rendering.
   */
  void setSize( const SbVec2i32& size ) override;

  /**
   * Gets the size used for rendering.
   */
  SbVec2i32 getSize() const override;

  /**
   * Defines the render action used for rendering.
   */
  virtual void setGLRenderAction( SoGLRenderAction* glAction );

  /**
   * Returns the render action used for rendering.
   * This method returns nullptr if the show() of the main window is not done.
   */
  SoGLRenderAction* getGLRenderAction() const;

  /**
   * Sets the global transparency algorithm to use when rendering.
   * @param type The global transparency algorithm
   */
  virtual void setTransparencyType( SoGLRenderAction::TransparencyType type);

  /**
   * Gets the global transparency algorithm to use when rendering.
   */
  virtual SoGLRenderAction::TransparencyType getTransparencyType() const;

  /**
   * Defines the antialiasing mode.
   * @param mode The antialiasing algorithm. Default is NO_ANTIALIASING which turns off antialiasing.
   */
  void setAntialiasingMode( SoSceneManager::AntialiasingMode mode );

  /**
   * @see setAntialiasingMode().
   */
  SoSceneManager::AntialiasingMode getAntialiasingMode() const;

  /**
   * Defines the antialiasing quality value.
   * @param quality The quality is a factor in the range [0.0,1.0]. @BR
   *        Default is 0.0. The value 0.0 turns off antialiasing.
   */
  void setAntialiasingQuality( float quality );

  /**
   * @see setAntialiasingQuality().
   */
  float getAntialiasingQuality() const;

  /**
   * Sets quality for supersampling when "still" (not interacting).
   * When quality is greater than 0, still images will be automatically supersampled.
   * @param quality The quality is a factor in the range [0.0,1.0]. @BR
   *        Default value is 0.0. Use the value 0.0 to turn off still supersampling. 0.5 is a typical value.
   * See also setStillSuperSamplingDelay.
   */
  void setStillSuperSamplingQuality( float quality );

  /**
   * @see setStillSuperSamplingQuality().
   */
  float getStillSuperSamplingQuality() const;

  /**
   * Sets delay for supersampling when "still" (not interacting).
   * If greater than 0, images will be supersampled after the specified delay.
   * @param delay The delay is in milliseconds. @BR
   *        If greater than 0, images will be supersampled after the specified delay
   * See also setStillSuperSamplingQuality.
   */
  void setStillSuperSamplingDelay( unsigned int delay );

  /**
   * @see setStillSuperSamplingDelay().
   */
  unsigned int getStillSuperSamplingDelay() const;

  /**
   * Activate/Deactivate stereo.
   *
   * @NOTES
   *   Do nothing.
   *   QML does not allow having raw stereo.
   */
  void activateStereo( bool activated );

  /**
   * Always returns false.
   *
   * @NOTES
   *   Stereo status can only be retrieved once rendering area is initialized.
   *   QML does not allow having raw stereo.
   */
  bool isStereoActivated() const;

  /**
   * Defines the stereo parameters.
   *
   * @NOTES
   *   Stereo parameters can only be set once rendering area is initialized.
   *   Do nothing.
   *   QML does not allow having raw stereo.
   */
  void setStereoParameters( SoStereoParameters* parameters );

  /**
   * Returns nullptr.
   *
   * @see setStereoParameters().
   */
  SoStereoParameters* getStereoParameters() const;

  /**
   * Always returns false.
   *
   * @NOTES
   *   QML does not allow having raw stereo.
   */
  bool isRawStereoAvailable();

protected:
  /**
   * Equivalent to the update command.
   * Always returns SoRenderAreaCore::STILL.
   */
  RenderStatus render() override;

  /**
   * This method is called by Qt in order to treat events
   */
  virtual bool event( QEvent* e );

  /**
   * Returns the event handler that raises when a new render starts.
   */
  SbEventHandler<RenderEventArg&>& onStartRender() override;

  /*
   * Creates the QmlRenderer used to render the scene graph.
   */
  Renderer* createRenderer() const override;

  virtual void createRenderAreaCore( SbVec2i32 region );

  /**
   * Instance of the class which provides the algorithms for viewing and managing
   * OpenInventor scene graphs.
   */
  mutable SoRef<SoRenderAreaCore> m_renderAreaCore;

  /** Member for each setting */
  SoNode* m_sceneGraph;
  ClearPolicy m_clearPolicy;
  SbColorRGBA m_color;
  float m_depth;
  SbVec2i32 m_size;
  bool m_stereo;

  /**
   * GLRenderAction used by this render area
   */
  SoGLRenderAction* m_glRenderAction;

  /**
   * Transparency type used by this render area
   */
  SoGLRenderAction::TransparencyType m_transparencyType;

  /**
   * Antialiasing mode used by this render area
   */
  SoSceneManager::AntialiasingMode m_antialiasingMode;

  /**
   * Antialiasing quality used by this render area
   */
  float m_antialiasingQuality;

  /**
   * Delay for still supersampling
   */
  unsigned int m_stillAntialiasingDelay;

  /**
   * Quality of still supersampling
   */
  float m_stillAntialiasingQuality;

public Q_SLOTS:
  void onWindowChanged( QQuickWindow* window );
  void onScreenChanged();

private : friend class QmlRenderer;

};

} // namespace qml
} // namespace qt
} // namespace viewercomponents
} // namespace inventor
} // namespace openinventor
