#pragma once
#include <Inventor/ViewerComponents/Qt/QtEventToSoEvent.h>
#include <Inventor/ViewerComponents/Qt/QML/RenderArea.h>

class SceneInteractor;

namespace openinventor {
namespace inventor {
namespace viewercomponents {
namespace qt {
namespace qml {

/**
 * @PREVIEWTAG
 *
 * Class to render an OpenInventor scene in a QML item.
 * This class extends RenderArea to add mouse and keyboard interactions.
 *
 * @ingroup ViewerComponentsQt
 *
 * @SEE_ALSO
 * RenderArea, RenderAreaOrbiter
 *
 * @PREVIEWFEATURES
 */
class QTVIEWERCOMPONENTS_API RenderAreaInteractive : public RenderArea
{

public:
  /**
   * Camera clipping planes adjustment mode.
   * When adjustment mode is set to AUTO, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.
   * When adjustment mode is set to MANUAL, the user is expected to
   * manually set the camera near and far planes.
   */
  enum ClippingPlanesAdjustMode
  {
    AUTO,
    MANUAL
  };

  /**
   * Constructor.
   */
  RenderAreaInteractive();

  /**
   * Destructor.
   */
  virtual ~RenderAreaInteractive();

  /*
   * Sets the scene graph to render.
   */
  void setSceneGraph( SoNode* sceneGraph ) override;

  /**
   * Sets the camera clipping planes adjustment mode.
   * When adjustment mode is set to AUTO, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.
   * When adjustment mode is set to MANUAL, the user is expected to
   * manually set those planes. Updating clipping planes after a camera move is
   * not enough; if a dragger or a rendered shape is moved, they can disappear
   * or become partially clipped.
   * Default is AUTO.
   */
  void setClippingPlanesAdjustMode( ClippingPlanesAdjustMode mode );

  /**
   * Get the camera clipping planes adjustment mode.
   */
  ClippingPlanesAdjustMode getClippingPlanesAdjustMode();

  /**
   * Return the root of the scene graph.
   */
  virtual SceneInteractor* getSceneInteractor() const;

  /**
   * Moves the camera to view the scene defined by the given path.
   * Equivalent to calling the SoCamera method viewAll(). Camera position is changed, but not orientation.
   */
  virtual void viewAll( const SbViewportRegion& viewport );

  /**
   * Moves the camera to be aligned with the given direction vector while
   * keeping the "up" direction of the camera parallel to the specified up
   * vector.
   */
  virtual void viewAxis( const SbVec3f& direction, const SbVec3f& up );

  /**
   * Activate/Deactivate stereo.
   */
  virtual void activateStereo( bool activated );

  /**
   * Returns true if stereo can be activated.
   */
  virtual bool isStereoSupported() const;

  /**
   * Set the stereo offset.
   */
  void setStereoCameraOffset( float offset );

  /**
   * Set the stereo balance.
   */
  void setStereoCameraBalance( float balance );

  /**
   * Processes the passed event to the scene graph managed here.
   * Returns TRUE if the event was handled by a node.
   *
   * @NOTES
   *   Events can only be processed once rendering area is initialized.
   */
  SbBool processEvent( const SoEvent* event );

  /**
   * Processes the passed event to the scene graph managed here.
   * Returns TRUE if at least one event was handled by a node.
   *
   * @NOTES
   *   Events can only be processed once rendering area is initialized.
   */
  SbBool processEvents( const std::vector<const SoEvent*>& eventList );

protected:
  RenderAreaInteractive( bool buildRootSceneGraph );

  /**
   * Equivalent to the update command.
   * Always returns SoRenderAreaCore::STILL.
   */
  SoRenderAreaCore::RenderStatus render() override;

  /**
   * This method is called by Qt when a mouse button is pressed.
   * It starts the trackball rotation.
   *
   * This function also manages the interactive mode.
   * We switch on interactive (FORCE_INTERACTION) mode when we press a mouse button,
   * and switch back to AUTO mode when we release the button.
   */
  void mousePressEvent( QMouseEvent* event ) override;

  /**
   * This method is called by Qt when a mouse button is released.
   * It ends the trackball rotation.
   */
  void mouseReleaseEvent( QMouseEvent* event ) override;

  /**
   * This method is called by Qt when the mouse moves and one button is pressed.
   * It updates the camera position if a mouse button is pressed.
   */
  void mouseMoveEvent( QMouseEvent* event ) override;

  /**
   * This method is called by Qt when the mouse moves without any button pressed.
   */
  void hoverMoveEvent( QHoverEvent* event ) override;

  /**
   * This method is called by Qt when the mouse wheel is used.
   * It is used to change the zoom level.
   */
  void wheelEvent( QWheelEvent* event ) override;

  /**
   * This method is called by Qt when a key is pressed.
   */
  void keyPressEvent( QKeyEvent* event ) override;

  /**
   * This method is called by Qt when a key is released.
   */
  void keyReleaseEvent( QKeyEvent* event ) override;

  /**
   * This method is called by Qt when the cursor enters the window.
   */
  virtual void enterEvent( QEvent* event );

  /**
   * This method is called by Qt when the cursor leaves the window.
   */
  virtual void leaveEvent( QEvent* event );

  /**
   * This method is called by Qt when a double-click occurs.
   */
  void mouseDoubleClickEvent( QMouseEvent* event ) override;

  /**
   * This method is called by Qt for all events.
   * It is overridden to handle touch events.
   */
  bool event( QEvent* qevent ) override;

  /**
   * Builds internal scene graph.
   */
  void buildSceneGraph();

  void createRenderAreaCore( SbVec2i32 region ) override;

  SceneInteractor* m_rootSceneGraph;

private:
  void init( bool buildRootSceneGraph );

  void startRender( SiRenderArea::RenderEventArg& arg );

  /**
   * Check if there are events pending on system event queue and ask for
   * aborting event to keep interactivity.
   */
  static SbBool s_abortRenderCallback( SoAction* action, void* );

  SoGroup* m_appSceneGraph;
  ClippingPlanesAdjustMode m_clippingMode;

  QtEventToSoEvent m_eventbuilder;

  bool m_isDragging;
};

} // namespace qml
} // namespace qt
} // namespace viewercomponents
} // namespace inventor
} // namespace openinventor
