#ifndef ViewerExaminer_H
#define ViewerExaminer_H

#include <Inventor/sys/port.h>

#include <QtWidgets/QWidget>

#include <Inventor/ViewerComponents/Qt/RenderAreaExaminer.h>

class SoNode;

/*! \cond PRIVATE */
namespace Ui
{
  class ViewerExaminer;
}
/*! \endcond */

/**
 * Class to render an OpenInventor scene graph in a Qt widget, with navigation tools.
 * This class extends QWidget.
 *
 * @ingroup ViewerComponentsQt
 */
class QTVIEWERCOMPONENTS_API ViewerExaminer : public QWidget
{
 Q_OBJECT
public:

  /**
   * Constructor.
   */
  ViewerExaminer( QWidget* parent );

  ~ViewerExaminer();

  /**
   * This method loads an IV file using
   * SoInput::openFile and SoDB::readAll.
   *
   * @param filename This argument is the fullpath to an iv file.
   */
  void loadFile( QString filename );

  /**
   * Sets the scene graph. This is the Open Inventor scene which will be traversed for rendering and event processing.
   */
  void setSceneGraph( SoNode* sceneGraph );

  /**
   * Gets the render area examiner.
   */
  RenderAreaExaminer* getRenderArea();

  /**
  * Set navigation mode.
  */
  virtual void setNavigationMode( SceneExaminer::NavigationMode mode );

  /**
  * Returns the current navigation mode.
  */
  virtual SceneExaminer::NavigationMode getNavigationMode();

  /**
  * Get the camera clipping planes adjustment mode.
  */
  virtual RenderAreaInteractive::ClippingPlanesAdjustMode getClippingPlanesAdjustMode();

  /**
  * Set the camera clipping planes adjustment mode.
  */
  virtual void setClippingPlanesAdjustMode( RenderAreaInteractive::ClippingPlanesAdjustMode mode );


  /**
   * Moves the camera to be aligned with the given direction vector while
   * keeping the "up" direction of the camera parallel to the specified up
   * vector.
   */
  virtual void viewAxis( const SbVec3f& direction, const SbVec3f& up );

  /**
  * Select the camera type
  */
  virtual void setCameraType( SoType type );

  /*
   * Activate/Deactivate stereo.
   */
  void activateStereo( bool activated );

  /*
   * Returns true if stereo buffering is enabled.
   */
  bool isRawStereoAvailable();

  /*
   * Returns true if stereo can be activated.
   */
  bool isStereoSupported();

  /*
   * Set the stereo offset.
   */
  void setStereoCameraOffset( float offset );

  /*
   * Set the stereo balance.
   */
  void setStereoCameraBalance( float balance );

public Q_SLOTS:
  /**
   * Set the camera in order to see the whole scene
   */
  virtual void viewAll();

protected Q_SLOTS:

  /**
   * Set YZ view
   */
  virtual void setYZView();

  /**
   * Set XZ view
   */
  virtual void setXZView();

  /**
   * Set XY view
   */
  virtual void setXYView();

  /**
   * Save the camera position
   */
  virtual void saveCamera();

  /**
   * Restore the camera position
   */
  virtual void restoreCamera();

  /**
   * Select the perspective camera
   */
  virtual void setCameraPerspective();

  /**
   * Select the orthographic camera
   */
  virtual void setCameraOrtho();

  /**
   * Activate/Deactivate seek mode
   */
  virtual void setSeekMode();

  /**
   * Open preferences dialog
   */
  virtual void openPreferences();

private:

  // Instance of the ui class auto generated by moc from ViewerExaminer.ui
  Ui::ViewerExaminer *m_ui;
  RenderAreaExaminer* m_renderArea;
  SoNode* m_sceneGraph;
  
};

#endif // ViewerExaminer_H


