#pragma once

#include <Inventor/ViewerComponents/SoStereoParameters.h>

/**
 *
 * @ingroup Stereo
 *
 * @DESCRIPTION
 * The SoInterlacedStereoParameters class is used to specify the behavior
 * of the interlaced stereo.
 *
 * The interlacing can be either vertical or horizontal, as defined by the InterlacingMode.
 *
 * Some SHARP screens require that the green component be written one pixel to the right,
 * which is supported through the greenShift method.
 *
 * Reverse interlacing is there to allow for screen-aligned interlacing.
 * It is up to the user of the parameters to set the reverse interlacing when the
 * window coordinates are not aligned with the screen coordinates.
 * Screen interlacing changes the way odd and even lines are computed.
 * By default the left eye is rendered on even lines and the right eye on odd lines.
 * In screen interlacing even is computed from the position of the pixel on the screen,
 * in non-screen interlacing it is relative to the rendering window.
 *
 * The interlaced mode can work in two modes: fast or best. This is simply a matter
 * of quality, and rendering time is not impacted.
 * Due to the interlacing, lines close to horizontal can appear as
 * dotted lines. In best mode, those parts of the model are drawn as if they were
 * thicker. By default, the mode is fast.
 *
 */
class INVENTOR_API SoInterlacedStereoParameters : public SoStereoParameters
{
public:
  /** Interlaced modes */
  enum InterlacedMode
  {
    /** Horizontal (default) */
    HORIZONTAL = 0, // default
    /** Vertical */
    VERTICAL
  };

  SoInterlacedStereoParameters();
  bool isReverseInterlacing() const;
  void setReverseInterlacing( bool reversed );

  bool getQualityRendering() const;
  void setQualityRendering( bool qualityRendering );

  bool getGreenShift() const;
  void setGreenShift( bool greenShift );

  InterlacedMode getInterlacingMode() const;
  void setInterlacingMode( InterlacedMode mode );

private:
  bool m_greenShift;
  bool m_qualityRendering;
  bool m_reverseInterlacing;
  InterlacedMode m_interlacedMode;
};
