/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#ifndef _SO_RENDER_AREA_CORE_
#define _SO_RENDER_AREA_CORE_

#include <Inventor/sys/port.h>

#if defined(_WIN32) && !defined(OIV_DISABLE_AUTOLINK) && !defined(ViewerComponents_EXPORTS) && !defined(Inventor_EXPORTS)
#  if !defined(__VIEWER_COMPONENTS_LIB)
#    if defined(_DEBUG)
#      define __VIEWER_COMPONENTS_LIB "fei_inventor_ViewerComponentsD.lib"
#    else
#      define __VIEWER_COMPONENTS_LIB "fei_inventor_ViewerComponents.lib"
#    endif
#  endif
#  pragma comment(lib, __VIEWER_COMPONENTS_LIB)
#endif

#include <Inventor/misc/SoRefCounter.h>
#include <Inventor/SbLinear.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/SbPImpl.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/ViewerComponents/SiRenderAreaAntialiasing.h>
#include <Inventor/ViewerComponents/SiRenderAreaTransparency.h>
#include <Inventor/ViewerComponents/SiRenderAreaStereo.h>
#include <Inventor/ViewerComponents/SiRenderAreaInteractive.h>
#include <Inventor/ViewerComponents/SiRenderArea.h>
#include <Inventor/ViewerComponents/SiRenderAreaStillSuperSampling.h>

class SoGLContext;
class SoSceneManager;
class SoEvent;
class SoStereoParameters;

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoRenderAreaCore);

/**
* @VSGEXT @PREVIEWTAG Utility class to render a scene with an OpenGl context.
*
* @ingroup ViewerComponents
*
* @DESCRIPTION
*
*  \if_cpp 
*  [C++] Reference counting: @BR
*  This is a reference counted object, similar to a node or path.
*  It can only be destroyed by incrementing and decrementing the reference count.
*  The initial reference count is 0. You may use the ref() and unref() methods,
*  but we recommend using the SoRef "smart pointer" template.
*  \endif
*
* @SEE_ALSO
*  RenderArea
*
* @PREVIEWFEATURES
*
*/
class INVENTOR_API SoRenderAreaCore : public SoRefCounter, public SiRenderAreaAntialiasing, public SiRenderAreaTransparency, 
                                      public SiRenderAreaStereo, public SiRenderAreaInteractive, public SiRenderArea,
                                      public SiRenderAreaStillSuperSampling
{

  SO_PIMPL_BASE_PUBLIC_HEADER(SoRenderAreaCore);

public:

    /**
   * Build a new SoRenderAreaCore that uses an existing glContext.
   */
  SoRenderAreaCore(SoGLContext* glContext);
  
  /**
   * Defines the scene graph which is managed here. This is the Open Inventor
   * scene which will be traversed for rendering and event processing.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetSceneGraph(newScene)}]
   */
  virtual void setSceneGraph(SoNode *newScene);

  /**
   * @see setSceneGraph().
   */
  virtual SoNode* getSceneGraph() const;

  /**
   * Processes the passed event by applying an SoHandleEventAction to the scene graph
   * managed here. Returns TRUE if the event was handled by a node.
   */
  virtual SbBool processEvent(const SoEvent *event);

  /**
   * Processes the passed event by applying an SoHandleEventAction to the scene graph
   * managed here. Returns TRUE if at least one event was handled by a node.
   */
  virtual SbBool processEvents(const std::vector<const SoEvent*>& eventList);

  /**
   * Applies an SoGLRenderAction to the scene graph managed here.
   * See RenderStatus documentation for details on the returned value.
   * The rendering may have been aborted, so check this function result
   * before calling swap buffer for example
   */   
  virtual RenderStatus render();

  /**
   * Defines the size to use for rendering.
   * Typically this method must be called when the render area has been resized or moved.
   */
  virtual void setSize(const SbVec2i32& size);

  /**
   * @see setSize().
   */
  virtual SbVec2i32 getSize() const;

  /**
   * Defines the render action used for rendering.
   */
  virtual void setGLRenderAction( SoGLRenderAction* glAction );

  /**
   * Returns the render action used for rendering.
   */
  virtual SoGLRenderAction* getGLRenderAction() const;

  /**
   * Defines the algorithm for rendering transparent objects. 
   * Default is NO_SORT. See SoGLRenderAction for possible transparency types.
   * See also SoGLRenderAction::TransparencyType.
   *
   * Note: When using transparency types with DELAYED in the name, the depth buffer is
   *   not updated (depth buffer writes are disabled) while rendering transparent objects.
   *   As a result complex 3D shapes may not be rendered correctly.
   */
  virtual void setTransparencyType( SoGLRenderAction::TransparencyType type );

  /**
   * @see setTransparencyType().
   */
  virtual SoGLRenderAction::TransparencyType getTransparencyType() const;

  /**
   * Returns the scene manager.
   */
  SoSceneManager* getSceneManager() const;

#if SoDEPRECATED_BEGIN(101000)
  /**
   * Set the current interactive mode.
   * @param mode interactive mode. Default is AUTO.
   */
  SoDEPRECATED_METHOD( 101000, "Use SoInteractiveComplexity node instead." )
  virtual void setInteractiveMode(SoInteractiveComplexity::InteractiveMode mode);

  /**
   * @see setInteractiveMode().
   */
  SoDEPRECATED_METHOD( 101000, "Use SoInteractiveComplexity node instead." )
  virtual SoInteractiveComplexity::InteractiveMode getInteractiveMode() const;
#endif /** @DEPRECATED_END */

  /**
   * Defines the stereo parameters.
   * @param parameters Provides specific parameters for a stereo mode.
   *        Use one of the subclasses of SoStereoParameters.
   *
   * @NOTES
   *  - SoRenderAreaCore makes a copy of the parameter values.
   *    \if_cpp The caller is responsible for allocation and destruction of the params object.
   *    As a copy of the parameter values is done, the object can be destroyed after
   *    calling this method. \endif
   *
   */
  virtual void setStereoParameters(SoStereoParameters* parameters);

  /**
   * @see setStereoParameters().
   */
  virtual SoStereoParameters* getStereoParameters() const;

  /**
   * Activate (or deactivate) stereo rendering with specified parameters (see setStereoParameters).
   * The stereo mode is determined by which subclass is used to set the parameters.
   * For example, passing an SoRawStereoParameters object activates OpenGL "quad buffered"
   * stereo (if possible on the current graphics board).
   *
   * @NOTES
   *  - Stereo rendering needs a SoStereoCamera to be activated. The edited camera
   *    is the first stereo camera found in the current scene graph. If no stereo camera
   *    is found, a warning is displayed. This method must be called to update the edited camera
   *    if the scene graph changed.
   *
   */
  virtual void activateStereo(bool activate);

  /**
   * @see activateStereo().
   */
  virtual bool isStereoActivated() const;

  /**
   * Define the antialiasing mode.
   * @param mode The antialiasing algorithm. Default is NO_ANTIALIASING which turns off antialiasing.
   */
  virtual void setAntialiasingMode(SoSceneManager::AntialiasingMode mode);

  /**
   * @see setAntialiasingMode().
   */
  virtual SoSceneManager::AntialiasingMode getAntialiasingMode() const;

  /**
   * Define the antialiasing quality value.
   * @param quality The quality is a factor in the range [0.0,1.0]. @BR
   *        Default is 0.0. The value 0.0 turns off antialiasing.
   */
  virtual void setAntialiasingQuality(float quality);

  /**
   * @see setAntialiasingQuality().
   */
  virtual float getAntialiasingQuality() const;

  /**
   * Defines the color buffer and depth buffer clear policy.
   * See also setClearColor and setClearDepth.
   * @useenum{ClearPolicy} Default is COLORBUFFER_AND_DEPTHBUFFER.
   *
   * @param policy color buffer and depth buffer clear policy.
   */
  virtual void setClearPolicy(ClearPolicy policy);

  /**
   * @see setClearPolicy().
   */
  virtual ClearPolicy getClearPolicy() const;

  /**
   * Defines the RGBA value used when the color buffer is cleared.
   * See also setClearPolicy. Default is transparent black (0,0,0,0).
   *
   * This is equivalent to setBackgroundColor() in the classical view classes.
   *
   * @param color RGBA value used to clear the color buffer.
   */
  virtual void setClearColor(const SbColorRGBA& color);

  /**
   * @see setClearColor().
   */
  virtual SbColorRGBA getClearColor() const;

  /**
   * Defines the depth value used when the depth buffer is cleared.
   * See also setClearPolicy. Default is 1. Range is 0..1.
   *
   * @param depth value used to clear the depth buffer.
   */
  virtual void setClearDepth(float depth);

  /**
   * @see setClearDepth().
   */
  virtual float getClearDepth() const;

  /**
   * Returns the event handler that raises when a new render starts.
   */
  virtual SbEventHandler<RenderEventArg&>& onStartRender();

  /**
   * Set quality for supersampling when "still" (not interacting).
   * When quality is greater than 0, still images will be automatically supersampled.
   * @param quality The quality is a factor in the range [0.0,1.0]. @BR
   *        Default value is 0.0. Use the value 0.0 to turn off still supersampling. 0.5 is a typical value. 
   * See also setStillSuperSamplingDelay.
   */
  virtual void setStillSuperSamplingQuality(float quality);

  /**
   * @see setStillSuperSamplingQuality().
   */
  virtual float getStillSuperSamplingQuality() const;

  /**
   * Set delay for supersampling when "still" (not interacting).
   * If greater than 0, images will be supersampled after the specified delay.
   * @param delay The delay is in milliseconds. @BR
   *        If greater than 0, images will be supersampled after the specified delay
   * See also setStillSuperSamplingQuality.
   */
  virtual void setStillSuperSamplingDelay(unsigned int delay);

  /**
   * @see setStillSuperSamplingDelay().
   */
  virtual unsigned int getStillSuperSamplingDelay() const;

SoINTERNAL public:

  /**
  * Set GL context as invalid so it won't be used anymore.
  */
  void invalidateGLContext();

  /**
   * Gets the current OpenGl context used by the instance.
   */
  SoGLContext* getGLContext() const;


  /**
   * This constructor creates a default glContext.
   * WARNING: This default constructor is a beta features. It will probably 
   * be removed in future versions, so we strongly discourage to use it.
   */
  SoRenderAreaCore();

protected:

  /**
  * Destructor.
  */
  virtual ~SoRenderAreaCore();

};

#endif // _SO_RENDER_AREA_CORE_


