/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/

#ifndef RenderAreaInteractive_H
#define RenderAreaInteractive_H

#include <Inventor/ViewerComponents/Win/RenderArea.h>
#include <Inventor/ViewerComponents/Win/WinEventToSoEvent.h>

class SceneInteractor;
class SoGroup;

/**
 * Class to render an OpenInventor scene in a Win32 OpenGL window.
 * This class extends RenderArea to add mouse and keyboard interactions.
 *
 * @ingroup ViewerComponentsWin
 */
class WINVIEWERCOMPONENTS_API RenderAreaInteractive : public RenderArea, public SiRenderAreaInteractive
{

public:

  /**
   * Camera clipping planes adjustment mode.
   * When adjustment mode is set to AUTO, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.
   * When adjustment mode is set to MANUAL, the user is expected to
   * manually set the camera near and far planes.
   */
  enum ClippingPlanesAdjustMode
  {
    AUTO,
    MANUAL
  };

  /**
   * Constructor.
   */
  RenderAreaInteractive();

  /**
   * Destructor.
   */
  virtual ~RenderAreaInteractive();

  /**
   * Set the scene graph to render.
   */
  virtual void setSceneGraph( SoNode* sceneGraph );

  /**
   * Return the root of the scene graph.
   */
  SoDEPRECATED_METHOD( 10500, "Use getSceneInteractor() method instead." )
  SceneInteractor* getRootSceneGraph();

  /**
   * Return the root of the scene graph.
   */
  virtual SceneInteractor* getSceneInteractor() const;

  /**
   * Set the camera clipping planes adjustment mode.
   * When adjustment mode is set to AUTO, the camera near and far planes
   * are dynamically adjusted to be as tight as possible (least amount of stuff
   * is clipped) before each render traversal.
   * When adjustment mode is set to MANUAL, the user is expected to
   * manually set those planes. Updating clipping planes after a camera move is
   * not enough, if a dragger or a rendered shape is moved, they can disappear
   * or become partially clipped.
   * Default is AUTO.
   */
  void setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode);

  /**
   * Get the camera clipping planes adjustment mode.
   */
  ClippingPlanesAdjustMode getClippingPlanesAdjustMode();

  /**
   * Move the camera to view the scene defined by the given path.
   * Equivalent to calling the SoCamera method viewAll(). Camera position is changed, but not orientation.
   */
  void viewAll( const SbViewportRegion &viewport );

   /**
   * This method is called to process windows events
   */
  virtual void processEvents( UINT uMsg, WPARAM wParam, LPARAM lParam );

  /**
   * This method is called in order to create an OpenGL context.
   */
  virtual void initializeGL( int width, int height );

#if SoDEPRECATED_BEGIN(101000)
  /**
   * Sets the current interactive mode.
   */
  SoDEPRECATED_METHOD( 101000, "Use SoInteractiveComplexity node instead." )
  virtual void setInteractiveMode(SoInteractiveComplexity::InteractiveMode mode);

  /**
   * Gets the current interactive mode.
   */
  SoDEPRECATED_METHOD( 101000, "Use SoInteractiveComplexity node instead." )
  virtual SoInteractiveComplexity::InteractiveMode getInteractiveMode() const;
#endif /** @DEPRECATED_END */

  /**
   * Processes the passed event to the scene graph managed here.
   * Returns TRUE if the event was handled by a node.
   *
   * @NOTES
   *   Events can only be processed once rendering area is initialized.
   */
  virtual SbBool processEvent(const SoEvent *event);

  /**
   * Processes the passed event to the scene graph managed here.
   * Returns TRUE if at least one event was handled by a node.
   *
   * @NOTES
   *   Events can only be processed once rendering area is initialized.
   */
  virtual SbBool processEvents(const std::vector<const SoEvent*>& eventList);

  /**
   * This method is called when we have a wheel event.
   */
  virtual void wheelEvent( short zDelta );

  /**
   * This method is called when we have a mouse window enter event.
   */
  virtual void enterEvent();

  /**
   * This method is called when we have a mouse window leave event.
   */
  virtual void leaveEvent();

  /**
   * This method is called when we have a keyboard press event.
   */
  virtual void keyPressEvent( WPARAM param );

  /**
   * This method is called when we have a keyboard release event.
   */
  virtual void keyReleaseEvent( WPARAM param );

  /**
   * This method is called when we have a mouse press event.
   *
   * This function also manage the interactive mode.
   * We switch on interactive (FORCE_INTERACTION) mode when we press a mouse button,
   * and switch back to AUTO mode when we release the button.
   */
  virtual void mousePressEvent( LPARAM loc, SoMouseButtonEvent::Button button );

  /**
   * This method is called when we have a mouse release event.
   */
  virtual void mouseReleaseEvent( LPARAM loc, SoMouseButtonEvent::Button button );

  /**
   * This method is called when we have a mouse move event.
   */
  virtual void mouseMoveEvent( LPARAM loc );

  /**
   * This method is called when we have a mouse double click event.
   */
  virtual void mouseDoubleClickEvent( LPARAM loc, SoMouseButtonEvent::Button button );

  /**
   * This method is called when we have a touch event.
   */
  virtual void touchEvent( WPARAM wParam, LPARAM lParam );

protected:

  RenderAreaInteractive( bool buildRootSceneGraph );

  /**
   * Build internal scene graph.
   */
  void buildSceneGraph();

  /**
   * Render the scene graph.
   */
  virtual SoRenderAreaCore::RenderStatus render();

  SceneInteractor* m_rootSceneGraph;
  // Auto interactive mode
  bool m_isAutoInteractive;

private:

  void init( bool buildRootSceneGraph );

  void startRender( SiRenderArea::RenderEventArg& arg );

  SoGroup* m_appSceneGraph;
  ClippingPlanesAdjustMode m_clippingMode;
  SoInteractiveComplexity::InteractiveMode m_interactiveMode;


  /**
  * This method is called to begin capture mouse events outside the current window.
  */
  void setCaptureOnButtonDown();
  /**
  * This method is called to end capture mouse events outside the current window.
  */
  void releaseCaptureOnButtonUp();
  int m_countDownButtons;

};

#endif // RenderAreaInteractive_H
