#ifndef _SceneExaminer_
#define _SceneExaminer_

#include <Inventor/ViewerComponents/nodes/SceneInteractor.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <vector>

class SoMouseWheelEvent;
class SoKeyboardEvent;
class SoLocation2Event;
class SoScaleGestureEvent;
class SoEvent;
class SoTouchEvent;
class SoRotateGestureEvent;
class SoDoubleTapGestureEvent;
class SoLongTapGestureEvent;
class SeekAnimator;
class NavigationInteraction;
class SelectionInteraction;
class BaseInteraction;
class SoSensor;
class SoOneShotSensor;
class SiInteractionModeListener;

/**
* Tool class for building a basic interactive OpenInventor application 
* with scene "examiner" viewing behavior.
*
* @ingroup ViewerComponentsNodes
*
* The SceneExaminer is an extension of the SceneInteractor node that provides
* camera and headlight manipulations like panning, zooming and orbiting similar
* to the classic Open Inventor viewer classes SoXtExaminerViewer (NavigationMode ORBIT)
* and SoXtPlaneViewer (NavigationMode PLANE).
*
* Similar behavior includes a 'headlight', i.e. an SoDirectionalLight node automatically
* aligned with the camera's view direction.
*
* The SceneExaminer is not directly comparable with a classic OpenInventor viewer as
* it does not provide any GUI (no buttons, no popup menu) and fewer interactive
* features (no animation). However it does provide a touch
* event handler that allows manipulating a scene on a touch device.
*
* See parent class SceneInteractor for more details about the structure
* of the internal scene graph.
*
* The SceneExaminer uses an instance of SoCameraInteractor 
* to manipulate the camera in response to OpenInventor events.
*
* Notes:
* - Window system integration @BR
*   The SceneExaminer needs a component that integrates the Open Inventor 3D
*   rendering window with the native window system.  System dependent tasks
*   include creating a window, placing the window in the application's user
*   interface, initializing OpenGL and processing the native event/message loop.
*   System independent support for this is provided by the SoRenderAreaCore class.
*   Example components are provided for various window systems, for example,
*   WinRenderArea for native Microsoft Windows.
*
* - Event handling @BR
*   The SceneExaminer needs a component that builds OpenInventor events (SoEvent)
*   from native system events. System independent support for this is provided by
*   the SoEventBuilder class. Example components are provided for various window
*   systems, for example, WinEventToSoEvent for native Microsoft Windows.
*
* - Library @BR
*   A basic version of SceneExaminer is a supported part of the Open Inventor API
*   and a prebuilt library is provided.
*
* - Source code @BR
*   The basic version of SceneExaminer is also provided as source code
*   to allow applications to customize and build their own interactive tool class. @BR
*   See $OIVHOME/source/Inventor/ViewerComponents/nodes.
*
* - Interaction Modes @BR
*   Similar to the classic Open Inventor viewer classes, SceneExaminer is either in
*   NAVIGATION mode (the default, similar to viewing mode) or SELECTION mode.  The user
*   must press the ESC key to toggle between interaction modes. (Of course this behavior
*   can be modified or replaced by implementing an alternate version of SceneExaminer.)
*   In navigation mode, Open Inventor events are automatically handled to modify the camera
*   as defined in the Usage section below.  Events that are defined in this section are
*   not sent to the application scene graph, but all other events are sent to the
*   application scene graph. Also specific viewing behaviors can be disabled as needed
*   (see for example #enableZoom()). In selection mode, all events are sent to
*   the application scene graph.
*
* - Scene graph @BR
*   The application scene graph should be the last child of the SceneExaminer.
*   The initial application scene graph can be added by simply calling the inherited
*   method addChild().  But note that if you need to replace the application scene
*   graph, for example loading a new data set, do @I not@i call removeAllChildren().
*   That would also remove the SceneExaminer's camera, headlight and event handler nodes.
*   Add an SoSeparator to the SceneExaminer to serve as the "scene graph holder", then
*   add and remove the application scene graph from this node.
*
* - Clip planes @BR
*   SceneExaminer automatically adjusts the 'near' and 'far' clipping planes when
*   events modifying the camera are handled. This adjustment, based on the
*   bounding box of the scene, ensures that shapes will not be clipped as the
*   camera orbits and also that depth buffer precision is maximized. This
*   adjustment is only done in InteractionMode NAVIGATION and can be disabled by
*   setting the environment variable "OIV_SCENE_EXAMINER_AUTO_CLIPPING_PLANES" to
*   false.
*   @BR Note: Updating clipping planes after a camera move can be not sufficient.
*   If the scene graph is modified or if a dragger or a rendered shape is moved,
*   they can disappear or become partially clipped. A classic implementation
*   of a render area must adjust clipping planes before each rendering by calling the
*   provided method #adjustClippingPlanes(). See render area's implementations available
*   in $OIVHOME/source/Inventor/gui folder for examples of adjustClippingPlanes use.
*
* - Compatibility @BR
*   Please note that some interaction behaviors are different than the classic
*   Open Inventor viewer classes (e.g. SoXtExaminerViewer): @BR @BR
*   - In @B Orbit@b mode: @B Left Mouse + Shift: @b does : Zoom in/out.
*   - In @B Plane@b mode: @B Left Mouse + Middle Mouse or Left Mouse + Shift: @b does : Roll the scene.
*   - The @B Mouse wheel@b in both modes performs a dolly relative to the @I cursor position@i,
*     not the center of the viewport.
*     @BR @BR
*   - The classic @B Alt key@b behavior is not implemented.
*     This key is reserved for application use.
*   - The @B Right Mouse@b button does not display a popup menu.
*     This button is reserved for application use.
*
* @USAGE
*
* - @B Orbit@b mode:
*     @BR @BR
*   - With a mouse:
*      - @B Left Mouse: @b Rotate the scene or seek to point if seek mode is activated.
*      - @B Middle Mouse or Left Mouse + Ctrl: @b Pan the scene.
*      - @B Left Mouse + Middle Mouse or Middle Mouse + Ctrl or Left Mouse + Shift: @b Zoom in/out.
*      - @B Mouse Wheel: @b Zoom in/out (zoom center is the mouse cursor location).
*      - @B Escape key: @b Switch between navigation mode and selection mode.
*      - @B S key: @b Activate/Deactivate seek mode.
*        @BR @BR
*   - With a touchscreen:
*     - @B 1 finger: @b Rotate the scene.
*     - @B 2 fingers: @b Rotate the scene on the screen plane, zoom in/out and pan (rotation and zoom center are located between the two fingers).
*     - @B Double tap: @b Seek to the point located by the finger.
*     - @B Long tap: @b Enable/Disable selection mode.
*       @BR @BR
*   - Spin animation: @BR
*     When spin animation is enabled (#enableSpin()), press mouse button / touch, drag
*     in the direction of desired spin and release mouse button / touch while still dragging.
*     @BR @BR
* - @B Plane@b mode:
*     @BR @BR
*   - With a mouse:
*      - @B Left Mouse: @b Zoom in/out or seek to point if seek mode is activated.
*      - @B Middle Mouse or Left Mouse + Ctrl: @b Pan the scene.
*      - @B Left Mouse + Middle Mouse or Middle Mouse + Ctrl or Left Mouse + Shift: @b Roll the scene.
*      - @B Mouse Wheel: @b Zoom in/out (zoom center is the mouse cursor location).
*      - @B Escape key: @b Switch between navigation mode and selection mode.
*      - @B S key: @b Activate/Deactivate seek mode.
*        @BR @BR
*   - With a touchscreen:
*     - @B 1 finger: @b Pan the scene.
*     - @B 2 fingers: @b Rotate the scene on the screen plane, zoom in/out and pan (rotation and zoom center are located between the two fingers).
*     - @B Double tap: @b Seek to the point located by the finger.
*     - @B Long tap: @b Enable/Disable selection mode.
*
* \htmlonly </UL> \endhtmlonly
*
* @SEE_ALSO
*   SceneInteractor, SceneOrbiter, SoCameraInteractor, SiInteractionModeListener
* [OIV-WRAPPER-CLASS NO_WRAP]
*/
class VIEWERCOMPONENTS_API SceneExaminer : public SceneInteractor
{

public:
  
  /**
  * Interaction Mode (navigation or selection)
  */
  enum InteractionMode
  {
    NAVIGATION,
    SELECTION
  };

  /**
  * Navigation Mode
  */
  enum NavigationMode
  {
    ORBIT,
    PLANE
  };

  /** Constructor */
  SceneExaminer();

  /** Destructor */
  virtual ~SceneExaminer();

  /**
  * Enable or disable selection mode. Default is true.
  */
  void enableSelection(bool enabled);

  /**
  * Returns if selection is enabled.
  */
  bool isSelectionEnabled();

  /**
  * Enable or disable zoom. Default is true.
  */
  void enableZoom(bool enabled);

  /**
  * Returns if zoom is enabled.
  */
  bool isZoomEnabled();

  /**
  * Enable or disable camera panning. Default is true.
  */
  void enablePan(bool enabled);

  /**
  * Returns if camera panning is enabled.
  */
  bool isPanEnabled();

  /**
  * Enable or disable camera orbiting. Default is true.
  */
  void enableOrbit(bool enabled);

  /**
  * Returns if camera orbiting is enabled.
  */
  bool isOrbitEnabled();

  /**
  * Enable or disable camera rotation. Default is true.
  */
  void enableRotate(bool enabled);

  /**
  * Returns if camera rotation is enabled.
  */
  bool isRotateEnabled();

  /**
  * Enable or disable seek. Default is true.
  */
  void enableSeek(bool enabled);

  /**
  * Returns if seek is enabled.
  */
  bool isSeekEnabled();

  /**
  * Set navigation mode. Default is ORBIT.
  */
  void setNavigationMode(SceneExaminer::NavigationMode mode);

  /**
   * Returns true if spin animation is enabled
   */
  bool isSpinEnabled() const;

  /**
   * Enables or disables the spin animation. Default is false.
   *
   * When spin animation is enabled, the user can use the mouse to start a spin
   * animation by "flinging".  Press the mouse button / touch, drag in the
   * direction of desired spin and release the mouse button / touch while still
   * dragging.  This is equivalent to spin animation in the classic viewer classes,
   * e.g. SoWinExaminerViewer.
   */
  void enableSpin(bool enable);

  /**
  * Returns the current navigation mode.
  */
  SceneExaminer::NavigationMode getNavigationMode();

  /**
  * Set interaction mode to navigation or selection. Default is NAVIGATION.
  */
  void setInteractionMode(SceneExaminer::InteractionMode mode);

  /**
  * Returns the current interaction mode.
  */
  SceneExaminer::InteractionMode getInteractionMode();

  /**
   * Select perspective or orthographic camera. Default is perspective.
   */
  virtual void setCameraMode( SceneInteractor::CameraMode mode );

  /**
  * Set the interaction into or out off seek mode (default is off).
  */
  void setSeekMode( bool onOrOff );

  /**
  * Add interaction listener.
  */
  void addInteractionModeListener( SiInteractionModeListener* listener );

  /**
  * Remove interaction listener.
  */
  void removeInteractionModeListener( SiInteractionModeListener* listener );

protected:
  virtual void mouseWheelMoved( SoMouseWheelEvent* wheelEvent, SoHandleEventAction* action );
  virtual void mouseMoved( SoLocation2Event* mouseEvent, SoHandleEventAction* action );
  virtual void mousePressed( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action );
  virtual void mouseReleased( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action );
  virtual void keyPressed( SoKeyboardEvent* keyEvent, SoHandleEventAction* action );
  virtual void keyReleased( SoKeyboardEvent* keyEvent, SoHandleEventAction* action );
  virtual void touch( SoTouchEvent* touchEvent, SoHandleEventAction* action );
  virtual void zoom( SoScaleGestureEvent* scaleEvent, SoHandleEventAction* action );
  virtual void rotate( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action );
  virtual void doubleTap(SoDoubleTapGestureEvent* doubleTapEvent, SoHandleEventAction* action);
  virtual void longTap(SoLongTapGestureEvent* longTapEvent, SoHandleEventAction* action);

private:
  void switchInteractionMode(SoHandleEventAction* action);

  NavigationInteraction* m_navigation;
  SelectionInteraction* m_selection;
  BaseInteraction* m_currentInteraction;
  bool m_isSelectionEnabled;
  std::vector<SiInteractionModeListener*> m_interactionModeListeners;
  SoOneShotSensor* m_modeSwitchDelayer;
};

#endif // _SceneExaminer_
