#ifndef _SceneInteractor_
#define _SceneInteractor_

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/ViewerComponents/SoCameraInteractor.h>

class SoMouseButtonEvent;
class SoMouseWheelEvent;
class SoKeyboardEvent;
class SoLocation2Event;
class SoScaleGestureEvent;
class SoTouchEvent;
class SoRotateGestureEvent;
class SoDoubleTapGestureEvent;
class SoLongTapGestureEvent;
class SoCamera;
class SoEventCallback;

/**
* Base class for building a basic OpenInventor application 
* without using the classic viewer classes.
*
* @ingroup ViewerComponentsNodes
*
* The SceneInteractor is a simple extension of the SoSeparator node that allows
* handling of Open Inventor events. This class should be overridden as it provides 
* only empty event handlers.  See, for example, SceneExaminer and SceneOrbiter.
* 
* This node is intended to be used as the root of a scene graph. 
* The SceneInteractor is a custom SoSeparator whose children are:
*  - An SoEventCallback node that handles keyboard, mouse and touch events.
*
*  - A camera (switch between perspective and orthographic camera).
*
*  - A headlight.
*
*  - The application's scene graph, which should be the last child.
*
* The SceneInteractor uses an instance of SoCameraInteractor in order 
* to manipulate the camera in response to OpenInventor events.
*
* Class diagram of the SceneInteractor showing the relationship between the 
* SoEventCallback, the SoCamera, the SoDirectionalLight (used as
* headlight) and the SoCameraInteractor.
* @IMAGE SceneInteractor.png
*
* Detail of the scene graph rooted by a SceneInteractor:
* @IMAGE SceneInteractorSceneGraph.png
*
* Notes:
*
* - A basic version of SceneInteractor is a supported part of the Open Inventor API,
*   but the prebuilt library is located in the example folders,
*   not in the usual folder.
*
* - The basic version of SceneInteractor is also provided as source code in the example folders
*   to allow applications to customize and build their own interactive tool class. @BR
*   See $OIVHOME/source/Inventor/gui/ViewerComponents
*
* - This class can only be used with ViewerComponents
*   and will not work with other viewer (like SoXtRenderArea)
*
* @SEE_ALSO
*   SceneExaminer, SoCameraInteractor
* [OIV-WRAPPER-CLASS NO_WRAP]
*/
class VIEWERCOMPONENTS_API SceneInteractor : public SoSeparator
{
public:

  /** Constructor */
  SceneInteractor();

  /** Destructor */
  virtual ~SceneInteractor();

  /**
   * Adjust near and far clipping planes to minimize clipping of objects in the
   * scene. This adjustment, based on the bounding box of the scene, ensures
   * that shapes will not be clipped and also that depth buffer precision is
   * maximized. This method should be called before each render traversal.
   */
  void adjustClippingPlanes( const SbViewportRegion &vpRegion );

  /**
   * Move the camera to view the scene defined by the given path. 
   * Equivalent to calling the SoCamera method viewAll(). Camera position is changed, but not orientation. 
   */
  void viewAll(const SbViewportRegion &viewport);

  /**
   * Moves the camera to be aligned with the given direction vector while
   * keeping the "up" direction of the camera parallel to the specified up
   * vector.
   */
  void viewAxis( const SbVec3f& direction, const SbVec3f& up );

  /**
  * Returns the current camera interactor.
  */
  SoCameraInteractor* getCameraInteractor();

  /**
  * Returns the current camera.
  */
  SoCamera* getCamera() const;

  /**
  * Type of camera (perspective or orthographic)
  */
  enum CameraMode
  {
    PERSPECTIVE,
    ORTHOGRAPHIC
  };

  /**
  * Select perspective or orthographic camera. Default is perspective.
  */
  virtual void setCameraMode(SceneInteractor::CameraMode mode);

  /**
  * Returns the current camera mode.
  */
  SceneInteractor::CameraMode getCameraMode();

  /**
  * Enable or disable headlight. Default is true.
  */
  void enableHeadLight(bool enabled);

  /**
  * Returns if headlight is enabled.
  */
  bool isHeadLightEnabled();

protected:
  SoRef<SoEventCallback> m_eventCallBack;
  SoCameraInteractor* m_cameraInteractor;

  virtual void mouseWheelMoved( SoMouseWheelEvent* wheelEvent, SoHandleEventAction* action );
  virtual void mouseMoved( SoLocation2Event* mouseEvent, SoHandleEventAction* action );
  virtual void mousePressed( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action );
  virtual void mouseReleased( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action );
  virtual void keyPressed( SoKeyboardEvent* keyEvent, SoHandleEventAction* action );
  virtual void keyReleased( SoKeyboardEvent* keyEvent, SoHandleEventAction* action );
  virtual void touch( SoTouchEvent* touchEvent, SoHandleEventAction* action );
  virtual void zoom( SoScaleGestureEvent* scaleEvent, SoHandleEventAction* action );
  virtual void rotate( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action );
  virtual void doubleTap(SoDoubleTapGestureEvent* doubleTapEvent, SoHandleEventAction* action);
  virtual void longTap(SoLongTapGestureEvent* longTapEvent, SoHandleEventAction* action);

  // Events callbacks
  static void mouseMoveCB(void * userdata, SoEventCallback * node);
  static void mouseCB(void * userdata, SoEventCallback * node);
  static void keyboardCB(void * userdata, SoEventCallback * node);
  static void touchCB(void * userdata, SoEventCallback * node);
  static void zoomCB(void * userdata, SoEventCallback * node);
  static void rotateCB(void * userdata, SoEventCallback * node);
  static void doubleTapCB(void * userdata, SoEventCallback * node);
  static void longTapCB(void * userdata, SoEventCallback * node);
  static void wheelCB(void * userdata, SoEventCallback * node);

private:
  SoRef<SoRotation> m_headlightRot;
  SoRef<SoSwitch> m_cameraSwitch;
  SoRef<SoSwitch> m_headlightSwitch;

  SoRef<SoCameraInteractor> m_perspInteractor;
  SoRef<SoCameraInteractor> m_orthoInteractor;

};

#endif // _SceneInteractor_
